/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "GtkClientHooks.h"
#include <gtkmm.h>
#include <string>
#include "GtkClientHooksEvents.h"
#include <sys/time.h>
#include <time.h>


namespace OmGtk {

GtkClientHooks::GtkClientHooks()
: m_events(2048) // FIXME: size?
{
}


GtkClientHooks::~GtkClientHooks()
{
}


/** Process all queued events that came from the OSC thread.
 *
 * This function is to be called from the Gtk thread only.
 */
void
GtkClientHooks::process_events()
{
	// Maximum number of events to process per call
	static const int MAX_EVENTS = 5;
	
	Event* ev = NULL;

	int events = 0;
	
	while ((!m_events.empty()) && events++ < MAX_EVENTS) {
		ev = m_events.front();
		m_events.pop();
		
		ev->execute();
		delete ev;
	}
}


/** Push an error event onto the queue.
 */
void
GtkClientHooks::error(const string& msg)
{
	ErrorEvent* ev = new ErrorEvent(msg);
	
	push_event(ev);
}


/** Push an engine enabled event onto the queue.
 */
void
GtkClientHooks::engine_enabled()
{
	EngineEnabledEvent* ev = new EngineEnabledEvent();
	
	push_event(ev);
}


/** Push an engine disabled event onto the queue.
 */
void
GtkClientHooks::engine_disabled()
{
	EngineDisabledEvent* ev = new EngineDisabledEvent();
	
	push_event(ev);
}


/** Push a new patch event onto the queue.
 */
void
GtkClientHooks::new_patch(PatchModel* pm)
{
	NewPatchEvent* ev = new NewPatchEvent(pm);
	
	push_event(ev);
}


/** Push a new port event onto the queue.
 */
void
GtkClientHooks::new_port(PortModel* port_model)
{
	NewPortEvent* ev = new NewPortEvent(port_model);
	
	push_event(ev);
}


/** Push a port removal event onto the queue.
 */
void
GtkClientHooks::port_removal(const string& path)
{
	PortRemovalEvent* ev = new PortRemovalEvent(path);
	
	push_event(ev);
}


/** Push a patch destruction event onto the queue.
 */
void
GtkClientHooks::patch_destruction(const string& path)
{
	PatchDestructionEvent* ev = new PatchDestructionEvent(path);
	
	push_event(ev);
}


/** Push a patch enabled event onto the queue.
 */
void
GtkClientHooks::patch_enabled(const string& path)
{
	PatchEnabledEvent* ev = new PatchEnabledEvent(path);
	
	push_event(ev);
}


/** Push a patch disabled event onto the queue.
 */
void
GtkClientHooks::patch_disabled(const string& path)
{
	PatchDisabledEvent* ev = new PatchDisabledEvent(path);
	
	push_event(ev);
}


/** Push a new node event onto the queue.
 */
void
GtkClientHooks::new_node(NodeModel* nm)
{
	NewNodeEvent* ev = new NewNodeEvent(nm);
	
	push_event(ev);
}


/** Push a node removal event onto the queue.
 */
void
GtkClientHooks::node_removal(const string& path)
{
	NodeRemovalEvent* ev = new NodeRemovalEvent(path);
	
	push_event(ev);
}


/** Push a new object_renamed event onto the queue.
 */
void
GtkClientHooks::object_renamed(const string& old_path, const string& new_path)
{
	ObjectRenamedEvent* ev = new ObjectRenamedEvent(old_path, new_path);
	
	push_event(ev);
}


/** Push a new connection event onto the queue.
 */
void
GtkClientHooks::connection(ConnectionModel* cm)
{
	ConnectionEvent* ev = new ConnectionEvent(cm);
	
	push_event(ev);
}


/** Push a disconnection event onto the queue.
 */
void
GtkClientHooks::disconnection(const string& src_port_path, const string& dst_port_path)
{
	DisconnectionEvent* ev = new DisconnectionEvent(src_port_path, dst_port_path);
	
	push_event(ev);
}


/** Push a metadata_update event onto the queue.
 */
void
GtkClientHooks::metadata_update(MetadataModel* mm)
{
	MetadataUpdateEvent* ev = new MetadataUpdateEvent(mm);
	
	push_event(ev);
}


/** Push a control_change event onto the queue.
 */
void
GtkClientHooks::control_change(ControlModel* cm)
{
	if (cm->port_path() != m_ignore_port_path) {
		ControlChangeEvent* ev = new ControlChangeEvent(cm);
		push_event(ev);
	}
}


/** Push a new_plugin event onto the queue.
 */
void
GtkClientHooks::new_plugin(PluginModel* pi)
{
	NewPluginEvent* ev = new NewPluginEvent(pi);
	
	push_event(ev);
}


} // namespace OmGtk
