/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef OMGTKAPP_H
#define OMGTKAPP_H

#include <string>
#include <map>
#include <list>
#include <libgnomecanvasmm.h>
#include <gtkmm.h>
#include <libglademm.h>
#include "PluginModel.h"
using std::string; using std::map; using std::list;

namespace LibOmClient { class PatchModel; }
using namespace LibOmClient;

namespace Om { class PluginModel; }
using LibOmClient::PluginModel;


namespace OmGtk {

class PatchWindow;
class PatchController;
class NewPatchWindow;
class LoadPatchWindow;
class MessagesWindow;
class ConfigWindow;
class OmGtkObject;
class OmModule;
class OmPort;
class OmPatchBayArea;
class PatchTreeView;

/** Master om_gtk class, virtually the entire app is contained within.
 *
 * \ingroup OmGtk
 */
class OmGtkApp
{
public:
	OmGtkApp();
	~OmGtkApp();

	void error_message(const string& msg);

	void engine_enabled(bool e);
	void patch_enabled(const string& path);
	void patch_disabled(const string& path);
	void patch_renamed(const string& old_path, const string& new_path);
	void patch_window_hidden(const string& path);

	void set_status_bar_text(const string& msg) { m_status_bar->push(msg); }

	// FIXME These lookup functions are waaaaaaay too slow, and a bottleneck
	
	inline PatchController* patch(const string& path) const
	{
		map<string, PatchController*>::const_iterator i = m_patches.find(path);
		return (i == m_patches.end()) ? NULL : (*i).second;
	}
	
	map<string, PatchController*>& patches() { return m_patches; }

	OmModule* module(const string& path) const;
	OmPort* port(const string& path) const;

	void add_patch(PatchModel* pm, bool show_window=false);
	void destroy_patch(const string& path);
	void new_plugin(const PluginModel* pi);
	void show_patch_menu(GdkEventButton* ev);

	void quit();

	void show_engine_error_dialog() { m_engine_error_dialog->show(); }

	const list<const PluginModel*>& plugins() const { return m_plugin_models; }
	
	Gtk::Window* window()  { return m_main_window; }
	
protected:
	void event_open_patch();
	void event_new_patch();
	void event_load_session();
	void event_save_session_as();
	void event_configuration();
	void event_quit_and_kill();
	void event_process_toggled();
	void event_patch_selected();
	void event_patch_activated(const Gtk::TreeModel::Path& path, Gtk::TreeView::Column* col);
	void event_show_about();
	void event_patch_enabled_toggled(const Glib::ustring& path_str);
	void event_patch_visible_toggled(const Glib::ustring& path_str);

	bool idle_callback();

	Gtk::TreeModel::iterator find_patch(Gtk::TreeModel::Children root, const string& path);

	map<string, PatchController*> m_patches;
	
	list<const PluginModel*> m_plugin_models;
	
	Gtk::Window*         m_main_window;
	NewPatchWindow*      m_new_patch_window;
	LoadPatchWindow*     m_load_patch_window;
	MessagesWindow*      m_messages_window;
	ConfigWindow*        m_config_window;
	PatchTreeView*       m_patches_treeview;
	Gtk::Window*         m_about_window;
	Gtk::MenuItem*       m_menu_file_open_patch;
	Gtk::MenuItem*       m_menu_file_new_patch;
	Gtk::MenuItem*       m_menu_file_load_session;
	Gtk::MenuItem*       m_menu_file_save_session_as;
	Gtk::MenuItem*       m_menu_file_configuration;
	Gtk::MenuItem*       m_menu_file_quit;
	Gtk::MenuItem*       m_menu_file_quit_and_kill;
	Gtk::CheckButton*    m_process_checkbutton;
	Gtk::MenuItem*       m_menu_help_about;
	Gtk::Button*         m_about_close_button;
	Gtk::Dialog*         m_engine_error_dialog;
	Gtk::Button*         m_engine_error_close_button;
	Gtk::ScrolledWindow* m_controls_scrollwin;
	Gtk::HPaned*         m_patches_controls_hpane;
	Gtk::Viewport*       m_controls_viewport;
	Gtk::Statusbar*      m_status_bar;

	/** Used to avoid feedback loops with (eg) process checkbutton
	 * FIXME: Maybe this should be globally implemented at the Controller level,
	 * disable all command sending while handling events to avoid feedback
	 * issues with widget event callbacks?  This same pattern is duplicated
	 * too much... */
	bool m_enable_signal;

	/** Patch which has controls currently shown */
	PatchController* m_current_shown_patch;
	
	/** Not visible, just a place for root patch modules to live */
	PatchController* m_root_patch_controller;
	
	
	struct PatchTreeModelColumns : public Gtk::TreeModel::ColumnRecord
	{
		PatchTreeModelColumns()
		{ add(name_col); add(enabled_col); add(visible_col); add(patch_controller_col); }
		
		Gtk::TreeModelColumn<Glib::ustring>    name_col;
		Gtk::TreeModelColumn<bool>             enabled_col;
		Gtk::TreeModelColumn<bool>             visible_col;
		Gtk::TreeModelColumn<PatchController*> patch_controller_col;
	};

	PatchTreeModelColumns            m_patch_tree_columns;
	Glib::RefPtr<Gtk::TreeStore>     m_patch_treestore;
	Glib::RefPtr<Gtk::TreeSelection> m_patch_tree_selection;
};


/** Derived TreeView class to support context menus for patches */
class PatchTreeView : public Gtk::TreeView {
public:
	PatchTreeView(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& xml)
	: Gtk::TreeView(cobject)
	{}

	void set_app(OmGtkApp* app) { m_app = app; }
	
	bool on_button_press_event(GdkEventButton* ev) {
		bool ret = Gtk::TreeView::on_button_press_event(ev);
	
		if ((ev->type == GDK_BUTTON_PRESS) && (ev->button == 3))
			m_app->show_patch_menu(ev);

		return ret;
	}
	
private:
	OmGtkApp* m_app;

}; // struct PatchTreeView

	
} // namespace OmGtk

#endif // OMGTKAPP_H
