/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "CreatePatchEvent.h"
#include "Patch.h"
#include "Node.h"
#include "NodeTree.h"
#include "Plugin.h"
#include "Om.h"
#include "OmApp.h"
#include "OSCSender.h"
#include "JackDriver.h"
#include "SlowEventQueue.h"
#include "OmPath.h"
#include "util.h"
#include "ObjectStore.h"

namespace Om {


CreatePatchEvent::CreatePatchEvent(Request* request, const string& path, int poly)
: SlowEvent(request, true),
  m_path(path),
  m_patch(NULL),
  m_parent(NULL),
  m_process_order(NULL),
  m_patch_listnode(NULL),
  m_patch_treenode(NULL),
  m_poly(poly),
  m_error(NO_ERROR),
  m_acquired_mutex(false)
{
}


void
CreatePatchEvent::prepare()
{
	Patch* existing = om->object_store()->find_patch(m_path);
	if (existing != NULL) {
		m_error = PATCH_EXISTS;
		SlowEvent::prepare();
		return;
	}

	if (m_poly < 1) {
		m_error = INVALID_POLY;
		SlowEvent::prepare();
		return;
	}
	
	m_parent = om->object_store()->find_patch(OmPath::parent(m_path));

	uint poly = 1;
	if (m_parent != NULL && m_poly > 1 && m_poly == static_cast<int>(m_parent->internal_poly()))
		poly = m_poly;
	
	m_patch = new Patch(OmPath::name(m_path), poly, m_parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size(), m_poly);
	
	m_patch_listnode = new ListNode<Patch*>(m_patch);
		
	if (m_parent != NULL) {
		m_patch_treenode = new TreeNode(m_patch);
		m_parent->node_remove_mutex().soft_lock();
		m_acquired_mutex = true;

		m_parent->add_node(m_patch_treenode);

		if (m_parent->process())
			m_process_order = Om::find_process_order(m_parent);
	}
	
	m_patch->activate();
	
	SlowEvent::prepare();
}


void
CreatePatchEvent::execute(uint sample_offset)
{
	SlowEvent::execute(sample_offset);

	if (m_patch_listnode != NULL && m_patch_listnode->elem() != NULL)
		om->add_patch(m_patch_listnode);
	
	if (m_patch_treenode != NULL && m_parent != NULL) {
		m_patch_treenode->node()->add_to_patch();
		
		if (m_parent->process_order() != NULL)
			om->maid()->push(m_parent->process_order());
		m_parent->process_order(m_process_order);
	}
}


void
CreatePatchEvent::post_process()
{
	if (m_acquired_mutex)
		m_parent->node_remove_mutex().soft_unlock();
		
	if (m_patch_listnode != NULL && m_patch_listnode->elem() != NULL) {
		m_request->respond_ok();
		//m_patch->send_creation_messages();
		om->osc_sender()->send_node_creation_messages(m_patch);
	} else {
		if (m_error == PATCH_EXISTS) {
			string msg = "Unable to create patch: ";
			msg += m_path += " already exists.";
			m_request->respond_error(msg);
		} else if (m_error == INVALID_POLY) {
			string msg = "Unable to create patch ";
			msg.append(m_path).append(": ").append("Invalid polyphony requested.");
			m_request->respond_error(msg);
		} else {
			m_request->respond_error("Unable to load patch.");
		}
	}
	om->jack_driver()->slow_event_queue()->signal_blocking_event_finished();
}


} // namespace Om

