/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.


             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 


             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.


             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.


             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.


             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.


             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 



*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */


// File:  session.c
//
// Session manager related functions
//

//#include <windows.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>  // for memcmp() et al

#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"


// session_mgr_find()
//
// search for the specified session.  returning a pointer to the session
// is dangerous
//
// Returns:  SESSION * or NULL
//
SESSION *
session_mgr_find( CK_SESSION_HANDLE handle )
{
   DL_NODE  * node   = NULL;
   SESSION  * result = NULL;
   CK_RV      rc;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return NULL;
   }
   node = sess_list;

   while (node) {
      SESSION *s = (SESSION *)node->data;

      if (s->handle == handle) {
         result = s;
         break;
      }

      node = node->next;
   }

   MY_UnlockMutex( &sess_list_mutex );
   return result;
}


// session_mgr_new()
//
// creates a new session structure and adds it to the process's list
// of sessions
//
// Args:  CK_ULONG      flags : session flags                   (INPUT)
//        SESSION **     sess : new session pointer             (OUTPUT)
//
// Returns:  CK_RV
//
CK_RV
session_mgr_new( CK_ULONG flags, SESSION **sess )
{
   SESSION  * new_session  = NULL;
   SESSION  * s            = NULL;
   DL_NODE  * node         = NULL;
   CK_BBOOL   user_session = FALSE;
   CK_BBOOL   so_session   = FALSE;
   CK_BBOOL   pkcs_locked  = TRUE;
   CK_BBOOL   sess_locked  = TRUE;
   CK_RV      rc;


   new_session = (SESSION *)malloc(sizeof(SESSION));
   if (!new_session) {
      st_err_log(0, __FILE__, __LINE__);
      rc = CKR_HOST_MEMORY;
      goto done;
   }

   memset( new_session, 0x0, sizeof(SESSION) );

   // find an unused session handle.  session handles will wrap
   // automatically...
   //

   rc = MY_LockMutex( &pkcs_mutex );      // this protects next_session_handle
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return rc;
   }
   pkcs_locked = TRUE;

   do {
      s = session_mgr_find( next_session_handle );
      if (s != NULL)
         next_session_handle++;
      else
         new_session->handle = next_session_handle++;
   } while (s != NULL);

   MY_UnlockMutex( &pkcs_mutex );
   pkcs_locked = FALSE;


   new_session->session_info.slotID        = 1;
   new_session->session_info.flags         = flags;
   new_session->session_info.ulDeviceError = 0;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return rc;
   }
   sess_locked = TRUE;

   // determine the login/logout status of the new session.  PKCS 11 requires
   // that all sessions belonging to a process have the same login/logout status
   //
   node = sess_list;
   while (node) {
      SESSION *s = (SESSION *)node->data;
      if (s->session_info.state == CKS_RW_SO_FUNCTIONS) {
         so_session = TRUE;
         break;
      }

      if ((s->session_info.state == CKS_RO_USER_FUNCTIONS) ||
          (s->session_info.state == CKS_RW_USER_FUNCTIONS))
      {
         user_session = TRUE;
         break;
      }

      node = node->next;
   }

// SAB XXX login does not drop after all sessions are closed XXX
   if ( global_login_state == CKS_RW_SO_FUNCTIONS) {
  	so_session = TRUE;
   }
   if ((global_login_state == CKS_RO_USER_FUNCTIONS) ||
       (global_login_state == CKS_RW_USER_FUNCTIONS)) {
	user_session = TRUE;
   }

// END SAB login state carry

   // we don't have to worry about having a user and SO session at the same time.
   // that is prevented in the login routine
   //
   if (user_session) {
      if (new_session->session_info.flags & CKF_RW_SESSION)
         new_session->session_info.state = CKS_RW_USER_FUNCTIONS;
      else
         new_session->session_info.state = CKS_RO_USER_FUNCTIONS;
   }
   else if (so_session) {
      new_session->session_info.state = CKS_RW_SO_FUNCTIONS;
   }
   else {
      if (new_session->session_info.flags & CKF_RW_SESSION)
         new_session->session_info.state = CKS_RW_PUBLIC_SESSION;
      else
         new_session->session_info.state = CKS_RO_PUBLIC_SESSION;
   }

   sess_list = dlist_add_as_first( sess_list, new_session );
   *sess = new_session;

done:
   if (pkcs_locked)
      MY_UnlockMutex( &pkcs_mutex );

   if (sess_locked)
      MY_UnlockMutex( &sess_list_mutex );

   if (rc != CKR_OK && new_session != NULL){
      st_err_log(147, __FILE__, __LINE__); 
      free( new_session );
   }
   return rc;
}


// session_mgr_so_session_exists()
//
// determines whether a RW_SO session exists for the specified process
//
// Returns:  TRUE or FALSE
//
CK_BBOOL
session_mgr_so_session_exists( void )
{
   DL_NODE *node = NULL;
   CK_RV    rc;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return rc;
   }
   node = sess_list;
   while (node) {
      SESSION *s = (SESSION *)node->data;
      if (s->session_info.state == CKS_RW_SO_FUNCTIONS) {
         rc = TRUE;
         goto done;
      }

      node = node->next;
   }

   rc = FALSE;

done:
   MY_UnlockMutex( &sess_list_mutex );
   return rc;
}


// session_mgr_user_session_exists()
//
// determines whether a USER session exists for the specified process
//
// Returns:  TRUE or FALSE
//
CK_BBOOL
session_mgr_user_session_exists( void )
{
   DL_NODE *node = NULL;
   CK_RV    rc;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return rc;
   }
   node = sess_list;
   while (node) {
      SESSION *s = (SESSION *)node->data;
      if ((s->session_info.state == CKS_RO_USER_FUNCTIONS) ||
          (s->session_info.state == CKS_RW_USER_FUNCTIONS))
      {
         rc = TRUE;
         goto done;
      }

      node = node->next;
   }

   rc = FALSE;

done:
   MY_UnlockMutex( &sess_list_mutex );
   return rc;
}


// session_mgr_public_session_exists()
//
// determines whether a PUBLIC session exists for the specified process
//
// Returns:  TRUE or FALSE
//
CK_BBOOL
session_mgr_public_session_exists( void )
{
   DL_NODE *node = NULL;
   CK_RV    rc;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return rc;
   }
   node = sess_list;
   while (node) {
      SESSION *s = (SESSION *)node->data;
      if ((s->session_info.state == CKS_RO_PUBLIC_SESSION) ||
          (s->session_info.state == CKS_RW_PUBLIC_SESSION))
      {
          rc = TRUE;
          goto done;
      }

      node = node->next;
   }

   rc = FALSE;

done:
   MY_UnlockMutex( &sess_list_mutex );
   return rc;
}


// session_mgr_readonly_exists()
//
// determines whether the specified process owns any read-only sessions.  this is useful
// because the SO cannot log in if a read-only session exists.
//
CK_BBOOL
session_mgr_readonly_exists( void )
{
   DL_NODE *node = NULL;
   CK_RV    rc;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return rc;
   }
   node = sess_list;
   while (node) {
      SESSION *s = (SESSION *)node->data;
      if ((s->session_info.flags & CKF_RW_SESSION) == 0) {
         rc = TRUE;
         goto done;
      }

      node = node->next;
   }

   rc = FALSE;

done:
   MY_UnlockMutex( &sess_list_mutex );
   return rc;
}


// session_mgr_close_session()
//
// removes the specified session from the process' session list
//
// Args:   PROCESS *    proc  :  parent process
//         SESSION * session  :  session to remove
//
// Returns:  TRUE on success else FALSE
//
CK_RV
session_mgr_close_session( SESSION *sess )
{
   DL_NODE  * node = NULL;
   CK_RV      rc = CKR_OK;

   if (!sess)
      return FALSE;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return CKR_FUNCTION_FAILED;
   }
   node = dlist_find( sess_list, sess );
   if (!node) {
      st_err_log(4, __FILE__, __LINE__, __FUNCTION__); 
      rc = CKR_FUNCTION_FAILED;
      goto done;
   }

   object_mgr_purge_session_objects( sess, ALL );

   if (sess->find_list)
      free( sess->find_list );

   if (sess->encr_ctx.context)
      free( sess->encr_ctx.context );

   if (sess->encr_ctx.mech.pParameter)
      free( sess->encr_ctx.mech.pParameter );

   if (sess->decr_ctx.context)
      free( sess->decr_ctx.context );

   if (sess->decr_ctx.mech.pParameter)
      free( sess->decr_ctx.mech.pParameter );

   if (sess->digest_ctx.context)
      free( sess->digest_ctx.context );

   if (sess->digest_ctx.mech.pParameter)
      free( sess->digest_ctx.mech.pParameter );

   if (sess->sign_ctx.context)
      free( sess->sign_ctx.context );

   if (sess->sign_ctx.mech.pParameter)
      free( sess->sign_ctx.mech.pParameter );

   if (sess->verify_ctx.context)
      free( sess->verify_ctx.context );

   if (sess->verify_ctx.mech.pParameter)
      free( sess->verify_ctx.mech.pParameter );

   free( sess );

   sess_list = dlist_remove_node( sess_list, node );

   // XXX XXX  Not having this is a problem
   //  for IHS.  The spec states that there is an implicit logout
   //  when the last session is closed.  Cannonicaly this is what other
   //  implementaitons do.  however on linux for some reason IHS can't seem 
   //  to keep the session open, which means that they go through the login
   //  path EVERY time, which of course causes a reload of the private 
   //  objects EVERY time.   If we are logged out, we MUST purge the private
   //  objects from this process..  
   //
   if (sess_list == NULL) {
	// SAB  XXX  if all sessions are closed.  Is this effectivly logging out
	   object_mgr_purge_private_token_objects();
   
		global_login_state = 0;
      // The objects really need to be purged .. but this impacts the
      // performance under linux.   So we need to make sure that the 
      // login state is valid.    I don't really like this.
    	MY_LockMutex( &obj_list_mutex );
   	object_mgr_purge_map((SESSION *)0xFFFF, PRIVATE);
     	MY_UnlockMutex( &obj_list_mutex );
   }

done:
   MY_UnlockMutex( &sess_list_mutex );
   return rc;
}


// session_mgr_close_all_sessions()
//
// removes all sessions from the specified process
//
CK_RV
session_mgr_close_all_sessions( void )
{
   CK_RV   rc = CKR_OK;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return CKR_FUNCTION_FAILED;
   }
   while (sess_list) {
      SESSION *sess = (SESSION *)sess_list->data;

      object_mgr_purge_session_objects( sess, ALL );

      if (sess->find_list)
         free( sess->find_list );

      if (sess->encr_ctx.context)
         free( sess->encr_ctx.context );

      if (sess->encr_ctx.mech.pParameter)
         free( sess->encr_ctx.mech.pParameter);

      if (sess->decr_ctx.context)
         free( sess->decr_ctx.context );

      if (sess->decr_ctx.mech.pParameter)
         free( sess->decr_ctx.mech.pParameter);

      if (sess->digest_ctx.context)
         free( sess->digest_ctx.context );

      if (sess->digest_ctx.mech.pParameter)
         free( sess->digest_ctx.mech.pParameter);

      if (sess->sign_ctx.context)
         free( sess->sign_ctx.context );

      if (sess->sign_ctx.mech.pParameter)
         free( sess->sign_ctx.mech.pParameter);

      if (sess->verify_ctx.context)
         free( sess->verify_ctx.context );

      if (sess->verify_ctx.mech.pParameter)
         free( sess->verify_ctx.mech.pParameter);

      free( sess );

      sess_list = dlist_remove_node( sess_list, sess_list );
   }

   MY_UnlockMutex( &sess_list_mutex );
   return CKR_OK;
}


// session_mgr_login_all()
//
// changes the login status of all sessions in the token
//
// Arg:  CK_USER_TYPE  user_type : USER or SO
//
CK_RV
session_mgr_login_all( CK_USER_TYPE user_type )
{
   DL_NODE  * node = NULL;
   CK_RV      rc = CKR_OK;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return CKR_FUNCTION_FAILED;
   }
   node = sess_list;
   while (node) {
      SESSION *s = (SESSION *)node->data;

      if (s->session_info.flags & CKF_RW_SESSION) {
         if (user_type == CKU_USER)
            s->session_info.state = CKS_RW_USER_FUNCTIONS;
         else
            s->session_info.state = CKS_RW_SO_FUNCTIONS;
      }
      else {
         if (user_type == CKU_USER)
            s->session_info.state = CKS_RO_USER_FUNCTIONS;
      }

      global_login_state = s->session_info.state; // SAB 
      node = node->next;
   }

   MY_UnlockMutex( &sess_list_mutex );
   return CKR_OK;
}


// session_mgr_logout_all()
//
// changes the login status of all sessions in the token
//
CK_RV
session_mgr_logout_all( void )
{
   DL_NODE  * node = NULL;
   SESSION  * s    = NULL;
   CK_RV      rc   = CKR_OK;

   rc = MY_LockMutex( &sess_list_mutex );
   if (rc != CKR_OK){
      st_err_log(146, __FILE__, __LINE__); 
      return CKR_FUNCTION_FAILED;
   }
   node = sess_list;
   while (node) {
      s = (SESSION *)node->data;

      // all sessions get logged out so destroy any private objects
      // public objects are left alone
      //
      object_mgr_purge_session_objects( s, PRIVATE );

      if (s->session_info.flags & CKF_RW_SESSION)
         s->session_info.state = CKS_RW_PUBLIC_SESSION;
      else
         s->session_info.state = CKS_RO_PUBLIC_SESSION;

      global_login_state = s->session_info.state; // SAB 

      node = node->next;
   }

   MY_UnlockMutex( &sess_list_mutex );
   return CKR_OK;
}


//
//
CK_RV
session_mgr_get_op_state( SESSION   *sess,
                          CK_BBOOL   length_only,
                          CK_BYTE   *data,
                          CK_ULONG  *data_len )
{
   OP_STATE_DATA  *op_data = NULL;
   CK_ULONG        op_data_len = 0;
   CK_ULONG        offset;

   if (!sess){
      st_err_log(4, __FILE__, __LINE__, __FUNCTION__); 
      return CKR_FUNCTION_FAILED;
   }

   // ensure that at least one operation is active
   //
   if (sess->find_active == TRUE){
      st_err_log(71, __FILE__, __LINE__); 
      return CKR_STATE_UNSAVEABLE;
   }
   if (sess->encr_ctx.active == TRUE) {
      if (op_data != NULL){
         st_err_log(71, __FILE__, __LINE__); 
         return CKR_STATE_UNSAVEABLE;
      }
      op_data_len = sizeof(OP_STATE_DATA)      +
                    sizeof(ENCR_DECR_CONTEXT)  +
                    sess->encr_ctx.context_len +
                    sess->encr_ctx.mech.ulParameterLen;

      if (length_only == FALSE) {
         op_data = (OP_STATE_DATA *)data;

         op_data->data_len         = op_data_len - sizeof(OP_STATE_DATA);
         op_data->session_state    = sess->session_info.state;
         op_data->active_operation = STATE_ENCR;

         offset = sizeof(OP_STATE_DATA);

         memcpy( (CK_BYTE *)op_data + offset,
                 &sess->encr_ctx,
                 sizeof(ENCR_DECR_CONTEXT) );

         offset += sizeof(ENCR_DECR_CONTEXT);

         if (sess->encr_ctx.context_len != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->encr_ctx.context,
                    sess->encr_ctx.context_len );

            offset += sess->encr_ctx.context_len;
         }

         if (sess->encr_ctx.mech.ulParameterLen != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->encr_ctx.mech.pParameter,
                    sess->encr_ctx.mech.ulParameterLen );
         }
      }
   }

   if (sess->decr_ctx.active == TRUE) {
      if (op_data != NULL){
         st_err_log(71, __FILE__, __LINE__); 
         return CKR_STATE_UNSAVEABLE;
      }
      op_data_len = sizeof(OP_STATE_DATA)      +
                    sizeof(ENCR_DECR_CONTEXT)  +
                    sess->decr_ctx.context_len +
                    sess->decr_ctx.mech.ulParameterLen;

      if (length_only == FALSE) {
         op_data = (OP_STATE_DATA *)data;

         op_data->data_len         = op_data_len - sizeof(OP_STATE_DATA);
         op_data->session_state    = sess->session_info.state;
         op_data->active_operation = STATE_DECR;

         offset = sizeof(OP_STATE_DATA);

         memcpy( (CK_BYTE *)op_data + offset,
                 &sess->decr_ctx,
                 sizeof(ENCR_DECR_CONTEXT) );

         offset += sizeof(ENCR_DECR_CONTEXT);

         if (sess->decr_ctx.context_len != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->decr_ctx.context,
                    sess->decr_ctx.context_len );

            offset += sess->decr_ctx.context_len;
         }

         if (sess->decr_ctx.mech.ulParameterLen != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->decr_ctx.mech.pParameter,
                    sess->decr_ctx.mech.ulParameterLen );
         }
      }
   }

   if (sess->digest_ctx.active == TRUE) {
      if (op_data != NULL){
         st_err_log(71, __FILE__, __LINE__); 
         return CKR_STATE_UNSAVEABLE;
      }
      op_data_len = sizeof(OP_STATE_DATA)        +
                    sizeof(DIGEST_CONTEXT)       +
                    sess->digest_ctx.context_len +
                    sess->digest_ctx.mech.ulParameterLen;

      if (length_only == FALSE) {
         op_data = (OP_STATE_DATA *)data;

         op_data->data_len         = op_data_len - sizeof(OP_STATE_DATA);
         op_data->session_state    = sess->session_info.state;
         op_data->active_operation = STATE_DIGEST;

         offset = sizeof(OP_STATE_DATA);

         memcpy( (CK_BYTE *)op_data + offset,
                 &sess->digest_ctx,
                 sizeof(DIGEST_CONTEXT) );

         offset += sizeof(DIGEST_CONTEXT);

         if (sess->digest_ctx.context_len != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->digest_ctx.context,
                    sess->digest_ctx.context_len );

            offset += sess->digest_ctx.context_len;
         }

         if (sess->digest_ctx.mech.ulParameterLen != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->digest_ctx.mech.pParameter,
                    sess->digest_ctx.mech.ulParameterLen );
         }
      }
   }

   if (sess->sign_ctx.active == TRUE) {
      if (op_data != NULL){
         st_err_log(71, __FILE__, __LINE__); 
         return CKR_STATE_UNSAVEABLE;
      }
      op_data_len = sizeof(OP_STATE_DATA)       +
                    sizeof(SIGN_VERIFY_CONTEXT) +
                    sess->sign_ctx.context_len  +
                    sess->sign_ctx.mech.ulParameterLen;

      if (length_only == FALSE) {
         op_data = (OP_STATE_DATA *)data;

         op_data->data_len         = op_data_len - sizeof(OP_STATE_DATA);
         op_data->session_state    = sess->session_info.state;
         op_data->active_operation = STATE_SIGN;

         offset = sizeof(OP_STATE_DATA);

         memcpy( (CK_BYTE *)op_data + offset,
                 &sess->sign_ctx,
                 sizeof(SIGN_VERIFY_CONTEXT) );

         offset += sizeof(SIGN_VERIFY_CONTEXT);

         if (sess->sign_ctx.context_len != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->sign_ctx.context,
                    sess->sign_ctx.context_len );

            offset += sess->sign_ctx.context_len;
         }

         if (sess->sign_ctx.mech.ulParameterLen != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->sign_ctx.mech.pParameter,
                    sess->sign_ctx.mech.ulParameterLen );
         }
      }
   }

   if (sess->verify_ctx.active == TRUE) {
      if (op_data != NULL){
         st_err_log(71, __FILE__, __LINE__); 
         return CKR_STATE_UNSAVEABLE;
      }
      op_data_len = sizeof(OP_STATE_DATA)        +
                    sizeof(SIGN_VERIFY_CONTEXT)  +
                    sess->verify_ctx.context_len +
                    sess->verify_ctx.mech.ulParameterLen;

      if (length_only == FALSE) {
         op_data = (OP_STATE_DATA *)data;

         op_data->data_len         = op_data_len - sizeof(OP_STATE_DATA);
         op_data->session_state    = sess->session_info.state;
         op_data->active_operation = STATE_SIGN;

         offset = sizeof(OP_STATE_DATA);

         memcpy( (CK_BYTE *)op_data + offset,
                 &sess->verify_ctx,
                 sizeof(SIGN_VERIFY_CONTEXT) );

         offset += sizeof(SIGN_VERIFY_CONTEXT);

         if (sess->verify_ctx.context_len != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->verify_ctx.context,
                    sess->verify_ctx.context_len );

            offset += sess->verify_ctx.context_len;
         }

         if (sess->verify_ctx.mech.ulParameterLen != 0) {
            memcpy( (CK_BYTE *)op_data + offset,
                    sess->verify_ctx.mech.pParameter,
                    sess->verify_ctx.mech.ulParameterLen );
         }
      }
   }

   *data_len = op_data_len;
   return CKR_OK;
}


//
//
CK_RV
session_mgr_set_op_state( SESSION           * sess,
                          CK_OBJECT_HANDLE    encr_key,
                          CK_OBJECT_HANDLE    auth_key,
                          CK_BYTE           * data,
                          CK_ULONG            data_len )
{
   OP_STATE_DATA  *op_data    = NULL;
   CK_BYTE        *mech_param = NULL;
   CK_BYTE        *context    = NULL;
   CK_BYTE        *ptr1       = NULL;
   CK_BYTE        *ptr2       = NULL;
   CK_BYTE        *ptr3       = NULL;
   CK_ULONG        len;


   if (!sess || !data){
      st_err_log(4, __FILE__, __LINE__, __FUNCTION__); 
      return CKR_FUNCTION_FAILED;
   }
   op_data = (OP_STATE_DATA *)data;

   // make sure the session states are compatible
   //
   if (sess->session_info.state != op_data->session_state){
      st_err_log(69, __FILE__, __LINE__); 
      return CKR_SAVED_STATE_INVALID;
   }
   // validate the new state information.  don't touch the session
   // until the new state is valid.
   //
   switch (op_data->active_operation) {
      case STATE_ENCR:
      case STATE_DECR:
         {
            ENCR_DECR_CONTEXT *ctx = (ENCR_DECR_CONTEXT *)(data + sizeof(OP_STATE_DATA));

            len = sizeof(ENCR_DECR_CONTEXT) + ctx->context_len + ctx->mech.ulParameterLen;
            if (len != op_data->data_len){
               st_err_log(69, __FILE__, __LINE__); 
               return CKR_SAVED_STATE_INVALID;
            }
            if (auth_key != 0){
               st_err_log(21, __FILE__, __LINE__); 
               return CKR_KEY_NOT_NEEDED;
            }
            if (encr_key == 0){
               st_err_log(23, __FILE__, __LINE__); 
               return CKR_KEY_NEEDED;
            }
            ptr1 = (CK_BYTE *)ctx;
            ptr2 = ptr1 + sizeof(ENCR_DECR_CONTEXT);
            ptr3 = ptr2 + ctx->context_len;

            if (ctx->context_len) {
               context = (CK_BYTE *)malloc( ctx->context_len );
               if (!context){
                  st_err_log(0, __FILE__, __LINE__);
                  return CKR_HOST_MEMORY;
               }
               memcpy( context, ptr2, ctx->context_len );
            }

            if (ctx->mech.ulParameterLen) {
               mech_param = (CK_BYTE *)malloc( ctx->mech.ulParameterLen );
               if (!mech_param) {
                  if (context)
                     free( context );
                  st_err_log(0, __FILE__, __LINE__);
                  return CKR_HOST_MEMORY;
               }
               memcpy( mech_param, ptr3, ctx->mech.ulParameterLen );
            }
         }
         break;

      case STATE_SIGN:
      case STATE_VERIFY:
         {
            SIGN_VERIFY_CONTEXT *ctx = (SIGN_VERIFY_CONTEXT *)(data + sizeof(OP_STATE_DATA));

            len = sizeof(SIGN_VERIFY_CONTEXT) + ctx->context_len + ctx->mech.ulParameterLen;
            if (len != op_data->data_len){
               st_err_log(69, __FILE__, __LINE__); 
               return CKR_SAVED_STATE_INVALID;
            }
            if (auth_key == 0){
               st_err_log(23, __FILE__, __LINE__); 
               return CKR_KEY_NEEDED;
            }
            if (encr_key != 0){
               st_err_log(21, __FILE__, __LINE__); 
               return CKR_KEY_NOT_NEEDED;
            }
            ptr1 = (CK_BYTE *)ctx;
            ptr2 = ptr1 + sizeof(SIGN_VERIFY_CONTEXT);
            ptr3 = ptr2 + ctx->context_len;

            if (ctx->context_len) {
               context = (CK_BYTE *)malloc( ctx->context_len );
               if (!context){
                  st_err_log(0, __FILE__, __LINE__);
                  return CKR_HOST_MEMORY;
               }
               memcpy( context, ptr2, ctx->context_len );
            }

            if (ctx->mech.ulParameterLen) {
               mech_param = (CK_BYTE *)malloc( ctx->mech.ulParameterLen );
               if (!mech_param) {
                  if (context)
                     free( context );
                  st_err_log(0, __FILE__, __LINE__);
                  return CKR_HOST_MEMORY;
               }
               memcpy( mech_param, ptr3, ctx->mech.ulParameterLen );
            }
         }
         break;

      case STATE_DIGEST:
         {
            DIGEST_CONTEXT *ctx = (DIGEST_CONTEXT *)(data + sizeof(OP_STATE_DATA));

            len = sizeof(DIGEST_CONTEXT) + ctx->context_len + ctx->mech.ulParameterLen;
            if (len != op_data->data_len){
               st_err_log(69, __FILE__, __LINE__); 
               return CKR_SAVED_STATE_INVALID;
            }
            if (auth_key != 0){
               st_err_log(23, __FILE__, __LINE__); 
               return CKR_KEY_NOT_NEEDED;
            }
            if (encr_key != 0){
               st_err_log(23, __FILE__, __LINE__); 
               return CKR_KEY_NOT_NEEDED;
            }
            ptr1 = (CK_BYTE *)ctx;
            ptr2 = ptr1 + sizeof(DIGEST_CONTEXT);
            ptr3 = ptr2 + ctx->context_len;

            if (ctx->context_len) {
               context = (CK_BYTE *)malloc( ctx->context_len );
               if (!context){
                  st_err_log(0, __FILE__, __LINE__);
                  return CKR_HOST_MEMORY;
               }
               memcpy( context, ptr2, ctx->context_len );
            }

            if (ctx->mech.ulParameterLen) {
               mech_param = (CK_BYTE *)malloc( ctx->mech.ulParameterLen );
               if (!mech_param) {
                  if (context)
                     free( context );
                  st_err_log(0, __FILE__, __LINE__);
                  return CKR_HOST_MEMORY;
               }
               memcpy( mech_param, ptr3, ctx->mech.ulParameterLen );
            }
         }
         break;

      default:
         st_err_log(69, __FILE__, __LINE__); 
         return CKR_SAVED_STATE_INVALID;
   }


   // state information looks okay.  cleanup the current session state, first
   //
   if (sess->encr_ctx.active)
      encr_mgr_cleanup( &sess->encr_ctx );

   if (sess->decr_ctx.active)
      decr_mgr_cleanup( &sess->decr_ctx );

   if (sess->digest_ctx.active)
      digest_mgr_cleanup( &sess->digest_ctx );

   if (sess->sign_ctx.active)
      sign_mgr_cleanup( &sess->sign_ctx );

   if (sess->verify_ctx.active)
      verify_mgr_cleanup( &sess->verify_ctx );


   // copy the new state information
   //
   switch (op_data->active_operation) {
      case STATE_ENCR:
         memcpy( &sess->encr_ctx, ptr1, sizeof(ENCR_DECR_CONTEXT) );

         sess->encr_ctx.key             = encr_key;
         sess->encr_ctx.context         = context;
         sess->encr_ctx.mech.pParameter = mech_param;
         break;

      case STATE_DECR:
         memcpy( &sess->decr_ctx, ptr1, sizeof(ENCR_DECR_CONTEXT) );

         sess->decr_ctx.key             = encr_key;
         sess->decr_ctx.context         = context;
         sess->decr_ctx.mech.pParameter = mech_param;
         break;

      case STATE_SIGN:
         memcpy( &sess->sign_ctx, ptr1, sizeof(SIGN_VERIFY_CONTEXT) );

         sess->sign_ctx.key             = auth_key;
         sess->sign_ctx.context         = context;
         sess->sign_ctx.mech.pParameter = mech_param;
         break;

      case STATE_VERIFY:
         memcpy( &sess->verify_ctx, ptr1, sizeof(SIGN_VERIFY_CONTEXT) );

         sess->verify_ctx.key             = auth_key;
         sess->verify_ctx.context         = context;
         sess->verify_ctx.mech.pParameter = mech_param;
         break;

      case STATE_DIGEST:
         memcpy( &sess->digest_ctx, ptr1, sizeof(DIGEST_CONTEXT) );

         sess->digest_ctx.context         = context;
         sess->digest_ctx.mech.pParameter = mech_param;
         break;
   }

   return CKR_OK;
}

// Return TRUE if the session we're in has its PIN
// expired.
CK_BBOOL pin_expired(CK_SESSION_INFO *si, CK_FLAGS flags)
{
   // If this is an SO session
   if (	(flags & CKF_SO_PIN_TO_BE_CHANGED) &&
	   (si->state == CKS_RW_SO_FUNCTIONS) )
	   return TRUE;
	   
   // Else we're a User session
   return( (flags & CKF_USER_PIN_TO_BE_CHANGED) &&
	  ((si->state == CKS_RO_USER_FUNCTIONS) ||
	   (si->state == CKS_RW_USER_FUNCTIONS)) );
}	   

// Return TRUE if the session we're in has its PIN
// locked.
CK_BBOOL pin_locked(CK_SESSION_INFO *si, CK_FLAGS flags)
{
   // If this is an SO session
   if (	(flags & CKF_SO_PIN_LOCKED) &&
	   (si->state == CKS_RW_SO_FUNCTIONS) ) 
	   return TRUE;
	   
   // Else we're a User session
   return( (flags & CKF_USER_PIN_LOCKED) &&
	  ((si->state == CKS_RO_USER_FUNCTIONS) ||
	   (si->state == CKS_RW_USER_FUNCTIONS)) );
}

// Increment the login flags after an incorrect password
// has been passed to C_Login. New for v2.11. - KEY
void set_login_flags(CK_USER_TYPE userType, CK_FLAGS_32 *flags)
{
	if(userType == CKU_USER) {
		if(*flags & CKF_USER_PIN_FINAL_TRY) {
			*flags |= CKF_USER_PIN_LOCKED;
			*flags &= ~(CKF_USER_PIN_FINAL_TRY);
		} else if (*flags & CKF_USER_PIN_COUNT_LOW) {
			*flags |= CKF_USER_PIN_FINAL_TRY;
			*flags &= ~(CKF_USER_PIN_COUNT_LOW);
		} else {
			*flags |= CKF_USER_PIN_COUNT_LOW;
		}
	} else {
		if(*flags & CKF_SO_PIN_FINAL_TRY) {
			*flags |= CKF_SO_PIN_LOCKED;
			*flags &= ~(CKF_SO_PIN_FINAL_TRY);
		} else if (*flags & CKF_SO_PIN_COUNT_LOW) {
			*flags |= CKF_SO_PIN_FINAL_TRY;
			*flags &= ~(CKF_SO_PIN_COUNT_LOW);
		} else {
			*flags |= CKF_SO_PIN_COUNT_LOW;
		}
	}
}


