/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "AccessorExpression.h"

#include <llvm/Constants.h>
#include <llvm/Instructions.h>

#include "../CodeGenerator_p.h"
#include "../ExpressionResult_p.h"
#include "../Function.h"
#include "../Debug.h"
#include "../VariableNG_p.h"
#include "../Visitor_p.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../Utils_p.h"

using namespace GTLCore::AST;

//------------------- AccessorExpression -------------------//

GTLCore::ExpressionResult AccessorExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* _bb ) const
{
  llvm::Value* ptr_ = pointer( _gc, _bb );
  GTL_ASSERT(ptr_);
  const Visitor* visitor = Visitor::getVisitorFor( type() );
  GTL_DEBUG("Type = " << *type() << " ptr = " << *ptr_);
  GTLCore::ExpressionResult result = visitor->get( _gc, _bb, ptr_, type() );
  GTL_DEBUG("Result = " << result );
  return result;
}

llvm::BasicBlock* AccessorExpression::affect( GenerationContext& _gc, llvm::BasicBlock* _bb, const ExpressionResult& _value )
{
  llvm::Value* ptr_ = pointer( _gc, _bb );
  const Visitor* visitor = Visitor::getVisitorFor( type() );
  return visitor->set( _gc, _bb, ptr_, type(), _value.value(), _value.type(), allocatedInMemory());
}

//------------------- ArraySizeAccessorExpression -------------------//

ArraySizeAccessorExpression::ArraySizeAccessorExpression(AccessorExpression* _parent )
  : m_parent( _parent )
{
}

bool ArraySizeAccessorExpression::isConstant() const
{
  return true;
}
const GTLCore::Type* ArraySizeAccessorExpression::type() const
{
  return GTLCore::Type::Integer32;
}


llvm::Value* ArraySizeAccessorExpression::pointer(GenerationContext& _gc, llvm::BasicBlock* _bb) const
{
  GTL_DEBUG("ArraySizeAccessorExpression::pointer");
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::Int32Ty, 0)); // Access the structure of the array
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::Int32Ty, 0)); // Access the size of the array
  return llvm::GetElementPtrInst::Create( m_parent->pointer( _gc, _bb),
                                      indexes.begin(), indexes.end(), "", _bb);
}

void ArraySizeAccessorExpression::markAsReturnExpression()
{
}

//------------------- StructAccessorExpression -------------------//

StructAccessorExpression::StructAccessorExpression(AccessorExpression* _parent, unsigned int _index) : m_parent(_parent), m_index(_index)
{
  GTL_ASSERT( m_index < m_parent->type()->structDataMembers()->size());
}

StructAccessorExpression::~StructAccessorExpression()
{
  delete m_parent;
}

llvm::Value* StructAccessorExpression::pointer(GenerationContext& _gc, llvm::BasicBlock* _bb) const
{
  GTL_DEBUG("StructAccessorExpression::pointer");
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(0));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(m_index));
  return llvm::GetElementPtrInst::Create( m_parent->pointer( _gc, _bb) , indexes.begin(), indexes.end(), "", _bb);
}

bool StructAccessorExpression::isConstant() const
{
  return m_parent->isConstant();
}

const GTLCore::Type* StructAccessorExpression::type() const
{
  return (*m_parent->type()->structDataMembers())[ m_index].type();
}

void StructAccessorExpression::markAsReturnExpression()
{
  m_parent->markAsReturnExpression();
}

//------------------- ArrayAccessorExpression -------------------//

ArrayAccessorExpression::ArrayAccessorExpression(AccessorExpression * _parent, Expression* _index) : m_parent(_parent), m_index(_index)
{
}

ArrayAccessorExpression::~ArrayAccessorExpression()
{
  delete m_parent;
  delete m_index;
}

llvm::Value* ArrayAccessorExpression::pointer(GenerationContext& _gc, llvm::BasicBlock* _bb) const
{
  GTL_DEBUG("ArrayAccessorExpression::pointer");
  llvm::Value* ptr_ = m_parent->pointer( _gc, _bb);
  GTL_ASSERT(ptr_);
  const Visitor* visitor = Visitor::getVisitorFor( m_parent->type() );
  llvm::Value* index = m_index->generateValue(_gc, _bb).value();
  return visitor->pointerToIndex(_gc, _bb, ptr_, m_parent->type(), index );
}

bool ArrayAccessorExpression::isConstant() const
{
  return m_parent->isConstant();
}

const GTLCore::Type* ArrayAccessorExpression::type() const
{
  const Visitor* visitor = Visitor::getVisitorFor( m_parent->type() );
  const GTLCore::Type* type = visitor->pointerToIndexType( m_parent->type() );
  GTL_ASSERT( type );
  return type;
}

void ArrayAccessorExpression::markAsReturnExpression()
{
  m_parent->markAsReturnExpression();
}

//------------------- VariableAccessorExpression -------------------//

VariableAccessorExpression::VariableAccessorExpression( GTLCore::VariableNG* _variable ) : m_variable(_variable)
{
}

llvm::Value* VariableAccessorExpression::pointer(GenerationContext& _gc, llvm::BasicBlock* bb) const
{
  GTL_DEBUG("VariableAccessorExpression::pointer");
  return m_variable->pointer();
}

bool VariableAccessorExpression::isConstant() const
{
  return m_variable->constant();
}

const GTLCore::Type* VariableAccessorExpression::type() const
{
  return m_variable->type();
}

void VariableAccessorExpression::markAsReturnExpression()
{
  m_variable->setAllocateInMemory( true );
  GTL_DEBUG( m_variable->allocatedInMemory() );
}

bool VariableAccessorExpression::allocatedInMemory() const
{
  return m_variable->allocatedInMemory();
}

llvm::BasicBlock* VariableAccessorExpression::affect( GenerationContext& _gc, llvm::BasicBlock* _bb, const ExpressionResult& _value )
{
  if( _value.functionResult() and (type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE ) and not m_variable->constantPointer() )
  {
    return m_variable->replacePointer( _gc, _bb, _value.value() );
  } else {
    return AccessorExpression::affect( _gc, _bb, _value );
  }
}

//------------------- FunctionMemberAccessorExpression -------------------//

FunctionMemberAccessorExpression::FunctionMemberAccessorExpression( AccessorExpression * _parent, 
                                          const Type::StructFunctionMember* _member,
                                          const std::list<AST::Expression*>& _arguments )
  : m_parent( _parent ), m_member(_member), m_arguments(_arguments)
{
  m_arguments.push_front( _parent );
}

FunctionMemberAccessorExpression::~FunctionMemberAccessorExpression()
{
  deleteAll( m_arguments );
// don't delete m_parent since it's part of the arguments list
}

bool FunctionMemberAccessorExpression::isConstant() const
{
  return true;
}

const GTLCore::Type* FunctionMemberAccessorExpression::type() const
{
  return m_member->returnType();
}

GTLCore::ExpressionResult FunctionMemberAccessorExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  GTL_DEBUG("Call member funcion: " << m_member->function()->name());
  return _gc.codeGenerator()->callFunction( _gc, bb, m_member->function(), m_arguments );
}

llvm::Value* FunctionMemberAccessorExpression::pointer(GenerationContext& _gc, llvm::BasicBlock* bb) const
{
  GTL_ABORT("No pointer for function member accessor.");
  return 0;
}

bool FunctionMemberAccessorExpression::allocatedInMemory() const
{
  GTL_ABORT("No memory is allocated for a function member accessor.");
  return false;
}

void FunctionMemberAccessorExpression::markAsReturnExpression()
{
  // There is no easy way to know if an Array/Structure given as argument of a function
  // will or will not be returned by the function, hence the need to mark parameters
  // as part of a return expression
  for( std::list<Expression*>::iterator it = m_arguments.begin();
       it != m_arguments.end(); ++it)
  {
    (*it)->markAsReturnExpression();
  }
}
