/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Tree.h"

#include <llvm/Constant.h>
#include <llvm/GlobalVariable.h>

#include "FunctionDeclaration.h"

#include "GTLCore/CodeGenerator_p.h"
#include "GTLCore/ExpressionResult_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/VariableNG_p.h"
#include "GTLCore/Utils_p.h"
#include "GTLCore/ModuleData_p.h"

#include "Expression.h"

using namespace GTLCore::AST;

GlobalConstantDeclaration::GlobalConstantDeclaration( const GTLCore::ScopedName& _name, const GTLCore::Type* _type, Expression* _initialiser) :
    m_name( _name), m_initialiser(_initialiser), m_variable( new GTLCore::VariableNG( _type, true ) ), m_type(_type)
{
}

GlobalConstantDeclaration::~GlobalConstantDeclaration()
{
  delete m_initialiser;
  delete m_variable;
}


void GlobalConstantDeclaration::generate( ModuleData* _module, GTLCore::CodeGenerator* _codeGenerator)
{
  GTLCore::GenerationContext gc( _codeGenerator, 0, 0, _module);
  
  llvm::Constant* value = _codeGenerator->convertConstantTo( m_initialiser->generateValue( gc, 0 ).constant(), m_initialiser->type() , m_type );
  llvm::Value* pointer = new llvm::GlobalVariable( value->getType(), true, llvm::GlobalValue::InternalLinkage, value, "", _module->llvmModule() );
  m_variable->initialise( gc, pointer);
}


Tree::Tree()
{
}

Tree::~Tree()
{
  deleteAll( m_functions );
  deleteAll( m_globalConstants );
}

const std::list<FunctionDeclaration*>& Tree::functionsDeclarations() const
{
  return m_functions;
}

void Tree::append(FunctionDeclaration* fd)
{
  m_functions.push_back( fd );
}
const std::list<GlobalConstantDeclaration*>& Tree::globalConstantDeclarations() const
{
  return m_globalConstants;
}

void Tree::append(GlobalConstantDeclaration* constant)
{
  m_globalConstants.push_back( constant );
}

bool Tree::containsGlobalConstant( const GTLCore::ScopedName& _name ) const
{
  for( std::list<GlobalConstantDeclaration*>::const_iterator it = m_globalConstants.begin();
       it != m_globalConstants.end(); ++it)
  {
    if( (*it)->name() == _name ) return true;
  }
  return false;
}


void Tree::generate( ModuleData* _module, GTLCore::CodeGenerator* _codeGenerator)
{
  // Generate constants
  for(std::list<GlobalConstantDeclaration*>::iterator it = m_globalConstants.begin();
      it != m_globalConstants.end(); ++it)
  {
    (*it)->generate( _module, _codeGenerator);
  }
  // Generate functions
  for(std::list<FunctionDeclaration*>::iterator it = m_functions.begin();
      it != m_functions.end(); ++it)
  {
    (*it)->generate( _module, _codeGenerator);
  }
}
