/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VariableNG_p.h"

#include <llvm/Constants.h>
#include <llvm/Instructions.h>

#include "Debug.h"
#include "ExpressionResult_p.h"
#include "GenerationContext_p.h"
#include "Type.h"
#include "Type_p.h"
#include "Visitor_p.h"

using namespace GTLCore;


struct VariableNG::Private
{
  const Visitor* visitor;
  const GTLCore::Type* type;
  bool constant;
  bool isInitialised;
  llvm::Value* pointer;
  bool allocatedInMemory;
  bool constantPointer;
};

VariableNG::VariableNG(const GTLCore::Type* _type, bool _constant ) : d(new Private)
{
  d->type = _type;
  d->constant = _constant;
  d->isInitialised = false;
  d->visitor = Visitor::getVisitorFor( _type );
  d->pointer = 0;
  d->allocatedInMemory = false;
  d->constantPointer = false;
  if( _type->d->isNestedArray() or _type->d->isComplexStructure() )
  {
    setAllocateInMemory( true );
  }
}

VariableNG::~VariableNG()
{
  delete d;
}
bool VariableNG::constant() const
{
  return d->constant;
}
const GTLCore::Type* VariableNG::type() const
{
  return d->type;
}

llvm::BasicBlock* VariableNG::initialise( GTLCore::GenerationContext& _generationContext, llvm::BasicBlock* _bb, const ExpressionResult& _initialiser, const std::list<llvm::Value*>& _initialSize )
{
  GTL_DEBUG( _initialSize.size() );
  llvm::Value* pointer = 0;
  if( _initialiser.value() and _initialiser.functionResult()
      and (type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE ) )
  {
    initialise( _generationContext, _initialiser.value() );
  } else {
    if( d->allocatedInMemory
        and (type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE ) )
    {
      pointer = new llvm::MallocInst( d->type->d->type(), llvm::ConstantInt::get(llvm::Type::Int32Ty, 1), "", _bb);
    } else {
      pointer = new llvm::AllocaInst( d->type->d->type(), llvm::ConstantInt::get(llvm::Type::Int32Ty, 1), "", _bb);
    }
    initialise( _generationContext, pointer );
    _bb = d->visitor->initialise( _generationContext, _bb, d->pointer, d->type, _initialSize, d->allocatedInMemory);
    
    if( _initialiser.value() )
    {
      _bb = d->visitor->set( _generationContext, _bb, d->pointer, d->type, _initialiser.value(), _initialiser.type() , d->allocatedInMemory);
    }
  }
  d->constantPointer = false;
  return _bb;
}

void VariableNG::initialise( GTLCore::GenerationContext& /*_generationContext*/, llvm::Value* _pointer )
{
  GTL_ASSERT(not d->isInitialised);
  d->isInitialised = true;
  d->pointer = _pointer;
  d->constantPointer = true;
}

llvm::Value* VariableNG::get(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock)
{
  llvm::Value* v = d->visitor->get( _generationContext, _currentBlock, d->pointer, d->type ).value();
  return v;
}

llvm::BasicBlock* VariableNG::set(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _value, const GTLCore::Type* _valueType )
{
  GTL_ASSERT(not constant() );
  return d->visitor->set( _generationContext, _currentBlock, d->pointer, d->type, _value, _valueType, d->allocatedInMemory);
}

llvm::BasicBlock* VariableNG::replacePointer( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  GTL_ASSERT( _pointer->getType() == d->pointer->getType() );
  GTL_ASSERT( not d->constantPointer );
  _currentBlock = cleanUp( _generationContext, _currentBlock, 0 );
  d->pointer = _pointer;
  return _currentBlock;
}

llvm::Value* VariableNG::pointer()
{
  return d->pointer;
}

void VariableNG::setAllocateInMemory( bool v )
{
  GTL_ASSERT( not d->isInitialised );
  if(type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE )
  {
    d->allocatedInMemory = v;
  }
}

bool VariableNG::allocatedInMemory() const
{
  return d->allocatedInMemory;
}

bool VariableNG::constantPointer() const
{
  return d->constantPointer;
}

llvm::BasicBlock* VariableNG::cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _donttouch )
{
  if( d->allocatedInMemory and not d->constantPointer )
  {
    llvm::BasicBlock* bb = d->visitor->cleanUp( _generationContext, _currentBlock, d->pointer, d->type, _donttouch, d->allocatedInMemory );
    new llvm::FreeInst( d->pointer, bb);
    return bb;
  }
  return _currentBlock;
}
