/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "PngDC.h"

#include <png.h>

#include <config-endian.h>

#include <GTLCore/Macros_p.h>
#include <GTLImageIO/ImageDCRegistry.h>
#include <GTLCore/Image.h>
#include <GTLCore/Type.h>
#include <GTLCore/Region.h>
#include <GTLCore/Debug.h>
#include <GTLCore/PixelDescription.h>

STATIC_INITIALISATION( PngDC )
{
  GTLImageIO::ImageDCRegistry::instance()->registerDC( new PngDC );
}

PngDC::PngDC()
{
  addReadWriteExtension("png");
}

PngDC::~PngDC()
{
}

GTLCore::AbstractImage* PngDC::decode( const GTLCore::String& _fileName, GTLCore::Region* _region, GTLCore::String* _errorMessage ) const
{
  FILE *fp = fopen( _fileName.c_str(), "rb"); 
  COND_TELL_ERROR( fp, "Can't open file: " + _fileName);
  
  // Create info structure
  png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);
  COND_TELL_ERROR( png_ptr, "Can't initialize libpng." );
  
  // Create info structure
  png_infop info_ptr = png_create_info_struct(png_ptr);
  if(not info_ptr) 
  {
    png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
    fclose(fp);
    TELL_ERROR( "Can't initialize libpng." );
  }
  
  // Create info structure
  png_infop end_info = png_create_info_struct(png_ptr);
  if (!end_info) 
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    fclose(fp);
    TELL_ERROR( "Can't initialize libpng." );
  }
  
  // Initialise jump
  if (setjmp(png_jmpbuf(png_ptr))) 
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
    fclose(fp);
    TELL_ERROR( "Can't initialize libpng." );
  }
  
  // Initialise IO
  png_init_io(png_ptr, fp);
  
  // Read information
  png_read_info(png_ptr, info_ptr);
  
  // Query information
  png_uint_32 width;
  png_uint_32 height;
  int bit_depth;
  int color_type;

  png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type, 0, 0, 0);
  
  const GTLCore::Type* channelType = 0;
  switch( bit_depth )
  {
    case 8:
      channelType = GTLCore::Type::UnsignedInteger8;
      break;
    case 16:
      channelType = GTLCore::Type::UnsignedInteger16;
      break;
    default:
      png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
      fclose(fp);
      TELL_ERROR( "Unsupported bit depth: " + GTLCore::String::number( bit_depth ) );
  }
#ifndef WORDS_BIGENDIAN
  if (bit_depth > 8)
    png_set_swap(png_ptr);
#endif
  GTL_ASSERT( channelType );
  int channelsCount = 0;
  switch( color_type )
  {
    case PNG_COLOR_TYPE_GRAY:
      channelsCount = 1;
      break;
    case PNG_COLOR_TYPE_GRAY_ALPHA:
      channelsCount = 2;
      break;
    case PNG_COLOR_TYPE_RGB:
      channelsCount = 3;
      break;
    case PNG_COLOR_TYPE_RGB_ALPHA:
      channelsCount = 4;
      break;
    default:
      png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
      fclose(fp);
      TELL_ERROR( "Unsupported color type: " + GTLCore::String::number( color_type ) );
  }
  GTL_ASSERT( channelsCount != 0);
  GTLCore::PixelDescription pixelDescription( channelType, channelsCount );
  
  // Allocate buffer image
  GTLCore::Image* image = new GTLCore::Image( width, height, pixelDescription );
  if( _region )
  {
    _region->setWidth( width );
    _region->setHeight( height );
  }
  
  png_bytep* row_pointers = new png_bytep[height];

  for (uint y = 0; y < height; y++)
  {
    row_pointers[y] = reinterpret_cast<png_byte*>(image->data(0, y));
  }
  // Read data
  png_read_image(png_ptr, row_pointers);
  
  // Cleanup
  png_read_end(png_ptr, end_info);
  png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
  delete[] row_pointers;
  fclose(fp);
  
  return image;
}

bool PngDC::encode( const GTLCore::AbstractImage* _image, const GTLCore::Region& _region, const GTLCore::String& _fileName, const GTLImageIO::Options* , GTLCore::String* _errorMessage ) const
{
  GTL_ASSERT( _image );
  FILE *fp = fopen(_fileName.c_str(), "wb");
  COND_TELL_ERROR( fp, "Can't open file: " + _fileName);
  
  png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);
  COND_TELL_ERROR( png_ptr, "Can't initialize libpng." );
  
  // Create info structure
  png_infop info_ptr = png_create_info_struct(png_ptr);
  if(not info_ptr) 
  {
    png_destroy_write_struct(&png_ptr, 0);
    fclose(fp);
    TELL_ERROR( "Can't initialize libpng." );
  }
  
  // Set error handler
  if (setjmp(png_jmpbuf(png_ptr))) 
  {
    png_destroy_write_struct(&png_ptr, &info_ptr); 
    fclose(fp); 
    TELL_ERROR( "Can't initialize libpng." );
  }
  
  // Check that all channels have the same type
  if( not _image->pixelDescription().sameTypeChannels() )
  {
    png_destroy_write_struct(&png_ptr, &info_ptr); 
    fclose(fp); 
    TELL_ERROR( "All channels must have the same type." );
  }
  // Guess the bit depth
  int bit_depth = 0;
  switch( _image->pixelDescription().channelTypes()[0]->dataType() )
  {
    case GTLCore::Type::UNSIGNED_INTEGER8:
      bit_depth = 8;
      break;
    case GTLCore::Type::UNSIGNED_INTEGER16:
      bit_depth = 16;
      break;
    default:
      png_destroy_write_struct(&png_ptr, &info_ptr);
      fclose(fp);
      TELL_ERROR( "Unsupported bit depth: " + GTLCore::String::number( bit_depth ) );
  }
  
  // Guess the color type
  int color_type;
  switch( _image->pixelDescription().channels() )
  {
    case 1:
      color_type = PNG_COLOR_TYPE_GRAY;
      break;
    case 2:
      color_type = PNG_COLOR_TYPE_GRAY_ALPHA;
      break;
    case 3:
      color_type = PNG_COLOR_TYPE_RGB;
      break;
    case 4:
      color_type = PNG_COLOR_TYPE_RGB_ALPHA;
      break;
    default:
      png_destroy_write_struct(&png_ptr, &info_ptr);
      fclose(fp);
      TELL_ERROR( "Unsupported number of channels: " + GTLCore::String::number( int(_image->pixelDescription().channels() ) ) );
  }
  // Set information
  png_set_IHDR( png_ptr, info_ptr, _region.width(), _region.height()
  , bit_depth, color_type, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
  png_init_io(png_ptr, fp );
  png_write_info(png_ptr, info_ptr);
  
#ifndef WORDS_BIGENDIAN
  if (bit_depth > 8)
    png_set_swap(png_ptr);
#endif
  

  
  int pixel_size = (_image->pixelDescription().bitsSize() / 8);
  png_bytep row_pointer = new png_byte[ pixel_size * _region.width()  ];

  for(int y = 0; y < _region.height(); ++y)
  {
    for(int x = 0; x < _region.width(); ++x)
    {
      memcpy( row_pointer + x * pixel_size, _image->data( x, y ), pixel_size );
    }
    png_write_row(png_ptr, row_pointer);
  }
  // Cleanup
  delete[] row_pointer;
  png_write_end(png_ptr, info_ptr);
  png_destroy_write_struct(&png_ptr, &info_ptr);
  fclose(fp);
  return true;
}
