/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>

// OpenCTL Headers
#include <OpenCTL/Debug.h>
#include <GTLCore/Function.h>
#include <OpenCTL/Module.h>
#include <OpenCTL/ModulesManager.h>
#include <GTLCore/Value.h>
#include <OpenCTL/Version.h>

void printVersion()
{
  std::cout << OpenCTL::LibraryShortName() << " - " << OpenCTL::LibraryName() << " - " << OpenCTL::LibraryVersionString() << std::endl;
  std::cout << OpenCTL::LibraryCopyright() << std::endl;
  std::cout << "CTL Version : " << OpenCTL::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : ctli [option] fileName.ctl" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help         print this message" << std::endl;
  std::cout << "  -v --version      print the version information" << std::endl;
}

#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  GTLCore::String functionName = "main";
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        OpenCTL::ModulesManager::instance()->addDirectory(argv[ai]);
      }
    } else {
      if( ai != argc - 1)
      {
        std::cerr << "Invalid command line parameters." << std::endl;
      } else {
        fileName = argv[ai];
      }
    }
  }
  if( fileName == "")
  {
    printHelp();
  } else {
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::String str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    OpenCTL::Module p( fileName );
    p.setSource( source );
    p.compile();
    if(not p.isCompiled())
    {
      std::cout << "Error: " << std::endl << p.compilationErrorsMessage() << std::endl;
      return EXIT_FAILURE;
    }
    const GTLCore::Function* function = p.function(functionName);
    if(not function)
    {
      std::cout << "Error: no '"<< functionName << "' function" << std::endl;
      return EXIT_FAILURE;
    }
    OCTL_DEBUG("Call function: " << functionName);
    GTLCore::Value v = function->call( std::vector<GTLCore::Value>() );
    OCTL_DEBUG("Call done");
    return v.asInt32();
  }
  return EXIT_SUCCESS;
}
