/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "BinaryExpression.h"

#include <llvm/Instructions.h>

#include <GTLCore/Debug.h>
#include <GTLCore/CodeGenerator_p.h>
#include <GTLCore/ExpressionResult_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/Visitor_p.h>

#include "AccessorExpression.h"
#include "Expression.h"

using namespace GTLCore::AST;

BinaryExpression::~BinaryExpression()
{
  delete m_lhs;
  delete m_rhs;
}


const GTLCore::Type* BinaryExpression::type() const
{
  const GTLCore::Type* type1 = m_lhs->type();
  const GTLCore::Type* type2 = m_rhs->type();
  GTL_ASSERT( type1 == type2 );
  UNUSED(type2);
  return type1;
}

void BinaryExpression::markAsReturnExpression()
{
  m_lhs->markAsReturnExpression();
  m_rhs->markAsReturnExpression();
}

const GTLCore::Type* OrBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult OrBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createOrExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb) );
}

const GTLCore::Type* AndBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult AndBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createAndExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb));
}

GTLCore::ExpressionResult BitOrBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createBitOrExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

GTLCore::ExpressionResult BitXorBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createBitXorExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

GTLCore::ExpressionResult BitAndBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createBitAndExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

const GTLCore::Type* EqualEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult EqualEqualBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createEqualExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

const GTLCore::Type* DifferentBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult DifferentBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createDifferentExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

const GTLCore::Type* InferiorEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult InferiorEqualBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createInferiorOrEqualExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

const GTLCore::Type* InferiorBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult InferiorBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createStrictInferiorExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

const GTLCore::Type* SupperiorEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult SupperiorEqualBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createSupperiorOrEqualExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

const GTLCore::Type* SupperiorBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult SupperiorBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createStrictSupperiorExpression( bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

GTLCore::ExpressionResult RightShiftBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createRightShiftExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb) );
}

GTLCore::ExpressionResult LeftShiftBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createLeftShiftExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb) );
}

GTLCore::ExpressionResult AdditionBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createAdditionExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb) );
}

GTLCore::ExpressionResult SubstractionBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createSubstractionExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb));
}

GTLCore::ExpressionResult MultiplicationBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createMultiplicationExpression( bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb) );
}

GTLCore::ExpressionResult DivisionBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createDivisionExpression(bb, leftHandSide()->generateValue( _gc, bb), rightHandSide()->generateValue( _gc, bb) );
}

GTLCore::ExpressionResult ModuloBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->createModuloExpression(bb, leftHandSide()->generateValue( _gc, bb), leftHandSide()->type(), rightHandSide()->generateValue( _gc, bb), rightHandSide()->type() );
}

AssignementBinaryExpression::AssignementBinaryExpression( AccessorExpression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs ), m_lhs(lhs)
{
  GTL_ASSERT(not lhs->isConstant() );
}

llvm::BasicBlock* AssignementBinaryExpression::generateStatement( GenerationContext& _gc, llvm::BasicBlock* _bb) const
{
  GTL_ASSERT( m_lhs);
  GTL_DEBUG( *m_lhs->type() );
  GTL_DEBUG( *rightHandSide()->type() );
  ExpressionResult rhsValue = rightHandSide()->generateValue( _gc, _bb);
  return m_lhs->affect( _gc, _bb, rhsValue );
}

GTLCore::ExpressionResult AssignementBinaryExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* _bb ) const
{
  GTL_ASSERT( m_lhs);
  llvm::BasicBlock* bbr = generateStatement( _gc, _bb );
  GTL_ASSERT( bbr == _bb);
  UNUSED( bbr );
  return GTLCore::ExpressionResult( m_lhs->generateValue( _gc, _bb) );
}

