/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "FunctionDeclaration.h"

#include <llvm/DerivedTypes.h>
#include <llvm/Function.h>
#include <llvm/Instructions.h>
#include <llvm/Type.h>

#include <GTLCore/CodeGenerator_p.h>
#include <GTLCore/ExpressionResult_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/Type_p.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Function_p.h>
#include <GTLCore/Utils_p.h>
#include <GTLCore/ModuleData_p.h>

#include "Expression.h"
#include "Statement.h"

using namespace GTLCore::AST;

FunctionParameter::~FunctionParameter()
{
  delete m_initialiser;
}

FunctionDeclaration::~FunctionDeclaration()
{
  deleteAll( m_parameters );
  deleteAll( m_parametersVariable );
  delete m_statement;
}

FunctionDeclaration::FunctionDeclaration(const GTLCore::ScopedName& _name, const GTLCore::Type* _returnType, std::vector< FunctionParameter* > _parameters ) : m_parameters(_parameters), m_statement(0)
{
  int minimumParameters = -1; //_parameters.size() - 1;
  std::vector<Parameter> parameters;
  for( unsigned int i = 0; i < m_parameters.size(); ++i)
  {
    // Compute the minimum number of parameters
    if( m_parameters[i]->initialiser() and minimumParameters == -1 )
    {
      minimumParameters = i;
    }
    parameters.push_back( m_parameters[i]->parameter() );
    // Create parameters variable NG
    const GTLCore::Type* type = m_parameters[i]->parameter().type();
    m_parametersVariable.push_back( new GTLCore::VariableNG( type,
                                    ( type->dataType() == Type::STRUCTURE or type->dataType() == Type::ARRAY ) and ( not m_parameters[i]->parameter().output() ), false ) );
  }
  m_functionData = new Function::Data( parameters, minimumParameters );
  m_function = new Function( _name, _returnType, m_functionData );
}

GTLCore::Function* FunctionDeclaration::function()
{
  return m_function;
}

void FunctionDeclaration::setStatement( Statement* statement )
{
  GTL_ASSERT( not m_statement);
  m_statement = statement;
}

void FunctionDeclaration::generate( ModuleData* _module, GTLCore::CodeGenerator* _codeGenerator)
{
  GTL_DEBUG("Generate function " << m_function->name() );
  m_functionData->setModule( _module );
  // Create the list of parameters as in llvm
  std::vector<const llvm::Type*> params;
  for(std::vector< Parameter >::const_iterator it = m_function->parameters().begin();
      it != m_function->parameters().end(); ++it)
  {
    if( it->output()  )
    {
      params.push_back( llvm::PointerType::get( it->type()->d->type(), 0 ) );
    } else {
      params.push_back( it->type()->d->asArgumentType() );
    }
  }
  
  // List of functions
  std::vector<llvm::Function*> functions( params.size() + 1 );
  
  // Generate the full function
  const llvm::Type* returnType = m_function->returnType()->d->asArgumentType();
  llvm::FunctionType* definitionType = llvm::FunctionType::get( returnType, params, false );
  GTLCore::String fullFunctionName = GTLCore::Function::Data::symbolName( m_function->name(), m_function->parameters());
  llvm::Function* fullFunction = _codeGenerator->createFunction(definitionType, fullFunctionName );
  functions[ params.size() ] = fullFunction;
  std::vector< Parameter > gtlParams = m_function->parameters(); // Use a copy of this to create the symbol name
  // Generates functions for default parameters
  llvm::Function* previousFunction = fullFunction;
  for( unsigned int i = m_function->parameters().size(); i > m_function->d->data->minimumParameters(); i--)
  {
    // Create the function type
    params.pop_back(); // remove the last parameter
    gtlParams.pop_back(); // remove the last parameter
    llvm::FunctionType* definitionType = llvm::FunctionType::get( m_function->returnType()->d->type(), params, false );
    // Create the function
    llvm::Function* function = _codeGenerator->createFunction( definitionType, 
        GTLCore::Function::Data::symbolName(m_function->name() , gtlParams ) );
    GenerationContext generationContext( _codeGenerator, function, m_function, _module);
    // Create a block for the call to the function with one more parameter
    llvm::BasicBlock* BB = llvm::BasicBlock::Create("CallForwarder");
    function->getBasicBlockList().push_back( BB );
    llvm::Value* defaultValue = m_parameters[i - 1]->initialiser()->generateValue( generationContext , BB).value();
    if( m_parameters[i - 1]->parameter().type()->dataType() != Type::STRUCTURE
        and m_parameters[i - 1]->parameter().type()->dataType() != Type::ARRAY )
    {
      defaultValue = _codeGenerator->convertValueTo( BB, defaultValue, m_parameters[i - 1]->initialiser()->type(), m_parameters[i - 1]->parameter().type());
    }
    // Prepare the arguments to call the function with one more parameter
    std::vector<llvm::Value*> arguments;
    for(  llvm::Function::arg_iterator it = function->arg_begin();
          it != function->arg_end(); ++it )
    {
      arguments.push_back( it );
    }
    arguments.push_back( defaultValue);
    // Make the call
    llvm::CallInst *callPrevious = llvm::CallInst::Create(previousFunction, arguments.begin(), arguments.end(), "", BB);
    callPrevious->setTailCall(false);
    if( m_function->returnType()->dataType() == Type::VOID )
    {
      llvm::ReturnInst::Create( BB );
    } else {
      llvm::ReturnInst::Create( callPrevious , BB);
    }
    previousFunction = function;
    functions[ params.size() ] = function;
  }
  
  // Set the list of functions
  m_functionData->setFunctions( functions );
  // Generate the code of the function
  llvm::BasicBlock* firstBlock = llvm::BasicBlock::Create();
  fullFunction->getBasicBlockList().push_back( firstBlock );
  GTL_DEBUG(*fullFunction);
  // Initialise the context
  GenerationContext generationContext( _codeGenerator, fullFunction, m_function, _module);
  // Initialize the arguments
  {
    llvm::Function::arg_iterator arg_it = fullFunction->arg_begin();   // Get the arg.
    std::vector< GTLCore::VariableNG* >::iterator pv_it = m_parametersVariable.begin();
    for(std::vector< Parameter >::const_iterator it = m_function->parameters().begin();
        it != m_function->parameters().end(); ++it, ++arg_it, ++pv_it)
    {
      if( it->output())
      {
        (*pv_it)->initialise( generationContext, firstBlock, arg_it );
      } else {
        if( it->type()->dataType() == Type::STRUCTURE or it->type()->dataType() == Type::ARRAY )
        {
          (*pv_it)->initialise( generationContext, firstBlock, arg_it );
         } else {
          firstBlock = (*pv_it)->initialise( generationContext, firstBlock, ExpressionResult( arg_it, it->type()), std::list<llvm::Value*>() );
        }
      }
    }
  }
  // Call the first statement
  llvm::BasicBlock* lastBlock = m_statement->generateStatement( generationContext, firstBlock);
  if(not lastBlock->getTerminator() )
  {
    GTL_DEBUG("No return instruction for function : " << m_function->name() << " creating one");
    if( m_function->returnType() == GTLCore::Type::Void )
    {
      llvm::ReturnInst::Create(lastBlock);
    } else
    {
      llvm::ReturnInst::Create( _codeGenerator->convertConstantTo( _codeGenerator->integerToConstant( 0), GTLCore::Type::Integer32, m_function->returnType() ), lastBlock  );
    }
  }
}
