/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "BufferImage.h"

#include <cstring>

#include <GTLCore/Buffer.h>
#include <GTLCore/Region.h>

#include "Debug.h"

using namespace GTLCore;

struct BufferImage::Private
{
  GTLCore::Buffer* buffer;
  int lineWidth;
  int pixelSize;
  int width;
  int height;
  char* defaultPixel;
};

BufferImage::BufferImage( int _width, int _height, GTLCore::Buffer* _buffer, const GTLCore::PixelDescription& _pixelDescription ) : AbstractImage(_pixelDescription), d(new Private)
{
  d->buffer = _buffer;
  d->lineWidth = _width * pixelSize();
  d->width = _width;
  d->height = _height;
  d->pixelSize = pixelSize();
  d->defaultPixel = new char[ d->pixelSize ];
  memset( d->defaultPixel, 0, d->pixelSize );
  GTL_DEBUG( d->buffer->size() << " " << d->lineWidth << " " << _height);
  GTL_ASSERT( d->buffer->size() == d->lineWidth * _height );
}

BufferImage::~BufferImage()
{
  delete[] d->defaultPixel;
  delete d->buffer;
  delete d;
}

char* BufferImage::data( int _x, int _y )
{
  if( _x >= 0 and _y >= 0 and _x < d->width and _y < d->height )
  {
    return d->buffer->rawData() + (_x * d->pixelSize + _y * d->lineWidth);
  } else {
    return d->defaultPixel;
  }
}

const char* BufferImage::data( int _x, int _y ) const
{
  if( _x >= 0 and _y >= 0 and _x < d->width and _y < d->height )
  {
    return d->buffer->rawData() + (_x * d->pixelSize + _y * d->lineWidth);
  } else {
    return d->defaultPixel;
  }
}

int BufferImage::lineWidth() const
{
  return d->lineWidth;
}

int BufferImage::width() const
{
  return d->width;
}

int BufferImage::height() const
{
  return d->height;
}

const GTLCore::Buffer* BufferImage::buffer() const
{
  return d->buffer;
}

GTLCore::Buffer* BufferImage::buffer()
{
  return d->buffer;
}

class BufferImageConstIterator : public AbstractImage::ConstIterator 
{
  public:
    BufferImageConstIterator( const Buffer* buffer, const Region& region) : m_buffer(buffer), m_region(region), m_done(false)
    {
    }
    bool next()
    {
      bool oldDone = m_done;
      m_done = true;
      return not oldDone;
    }
    bool hasNext() const
    {
      return m_done;
    }
    const Buffer* current() const
    {
      if(m_done)
      {
        return m_buffer;
      } else {
        return 0;
      }
    }
    Region area() const
    {
      return m_region;
    }
  private:
    const Buffer* m_buffer;
    Region m_region;
    bool m_done;
};

AbstractImage::ConstIterator* BufferImage::createIterator() const
{
  return new BufferImageConstIterator(buffer(), Region(0, 0, width(), height()));
}

class BufferImageIterator : public AbstractImage::Iterator 
{
  public:
    BufferImageIterator( Buffer* buffer, const Region& region) : m_buffer(buffer), m_region(region), m_done(false)
    {
    }
    bool next()
    {
      bool oldDone = m_done;
      m_done = true;
      return not oldDone;
    }
    bool hasNext() const
    {
      return m_done;
    }
    const Buffer* current() const
    {
      if(m_done)
      {
        return m_buffer;
      } else {
        return 0;
      }
    }
    Buffer* current()
    {
      if(m_done)
      {
        return m_buffer;
      } else {
        return 0;
      }
    }
    Region area() const
    {
      return m_region;
    }
  private:
    Buffer* m_buffer;
    Region m_region;
    bool m_done;
};

AbstractImage::Iterator* BufferImage::createIterator()
{
  return new BufferImageIterator(buffer(), Region(0, 0, width(), height()));
}
