/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CompilerBase_p.h"

#include <map>

#include "ConvertCenter_p.h"
#include "Debug.h"
#include "ErrorMessage.h"
#include "ScopedName.h"
#include "Function.h"
#include "GTLCore/Function_p.h"
#include "Value.h"
#include "Macros_p.h"
#include "ModuleData_p.h"

#include <cstdarg>

using namespace GTLCore;

struct CompilerBase::Private {
  GTLCore::String moduleName;
  std::list<ErrorMessage> errorMessages;
  std::map< GTLCore::ScopedName, GTLCore::Function* > functions;
  std::list<GTLCore::Function*> functionsToDelete;
  ConvertCenter* convertCenter;
  ModuleData* moduleData;
};

CompilerBase::CompilerBase() : d(new Private)
{
  d->convertCenter = new ConvertCenter;
}
CompilerBase::~CompilerBase()
{
  delete d->convertCenter;
  delete d;
}

void CompilerBase::appendError( const GTLCore::ErrorMessage& _msg)
{
  GTLCore::ErrorMessage msg = _msg;
  if( msg.fileName() == "")
  {
    msg.setFileName( d->moduleName );
  }
  GTL_DEBUG( msg.fileName() << " at " << msg.line() << ": " << msg.errorMessage() );
  d->errorMessages.push_back(msg);
}

void CompilerBase::appendError( GTLCore::ErrorMessage* msg)
{
  GTL_ASSERT(msg);
  if( msg->fileName() == "")
  {
    GTL_ASSERT(d);
    msg->setFileName( d->moduleName );
  }
  GTL_DEBUG( msg->fileName() << " at " << msg->line() << ": " << msg->errorMessage() );
  d->errorMessages.push_back(*msg);
  delete msg;
}

void CompilerBase::appendErrors( const std::list<GTLCore::ErrorMessage>& _msgs)
{
  d->errorMessages.insert( d->errorMessages.begin(), _msgs.begin(), _msgs.end() );
}

const std::list<GTLCore::ErrorMessage>& CompilerBase::errorMessages() const
{
  return d->errorMessages;
}

void CompilerBase::setModuleName( const GTLCore::String& _moduleName)
{
  d->moduleName = _moduleName;
}
const GTLCore::String& CompilerBase::moduleName() const
{
  return d->moduleName;
}

GTLCore::Function* CompilerBase::function( const GTLCore::ScopedName&  name)
{
  GTL_DEBUG( "Look for function: " << name );
  std::map< GTLCore::ScopedName, GTLCore::Function* >::iterator it = d->functions.find( name );
  if( it == d->functions.end() )
  {
    if( name.nameSpace() == "" )
    {
      GTL_DEBUG(" Available functions");
      for( std::map< GTLCore::ScopedName, GTLCore::Function* >::iterator it2 = d->functions.begin(); it2 != d->functions.end(); ++it2)
      {
        GTL_DEBUG( it2->first );
      }
      return 0;
    }
    GTLCore::ScopedName  globalName( "", name.name() );
    return function( globalName );
  } else {
    return it->second;
  }
}

bool CompilerBase::declareFunction( const GTLCore::ScopedName& name, GTLCore::Function* func)
{
  std::map< GTLCore::ScopedName, GTLCore::Function* >::iterator it = d->functions.find( name );
  if( it == d->functions.end() )
  {
    d->functions[name] = func;
    return true;
  } else {
    return false;
  }
}

ConvertCenter* CompilerBase::convertCenter()
{
  return d->convertCenter;
}

std::list<GTLCore::Function*>& CompilerBase::functionsToDelete()
{
  return d->functionsToDelete;
}

void CompilerBase::createStdLibFunction(const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, int count, ...)
{
  std::vector<GTLCore::Parameter> arguments;
  va_list argp;
  va_start(argp, count);
  for(int i = 0; i < count; ++i)
  {
    const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
    arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
  }
  va_end(argp);
  GTLCore::Function* function = GTLCore::Function::Private::createExternalFunction( d->moduleData, _name, _symbolName, retType, arguments );
  bool success = declareFunction(GTLCore::ScopedName("", _name), function);
  GTL_ASSERT( success );
  UNUSED( success );
  functionsToDelete().push_back(function);
}

void CompilerBase::setModuleData( ModuleData* moduleData )
{
  d->moduleData = moduleData;
}
