/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_ERROR_MESSAGES_H_
#define _GTLCORE_ERROR_MESSAGES_H_

#include <GTLCore/String.h>

namespace GTLCore {
  class ErrorMessage;
  /**
   * This namespace contains functions that return common error messages.
   */
  namespace ErrorMessages {
    /**
     * Error message that happen when some part of the language is not
     * yet implemented.
     */
    ErrorMessage* unimplemented(const String& what);
    /**
      * Error message that happen when the user try to set a value to a
      * constant after initialisation.
      * For instance:
      * @code
      * const int a = 0;
      * a = 2; // trigger assigningValueToConstant()
      * @endcode
      */
    ErrorMessage* assigningValueToConstant();
    /**
      * Error that happen when the user try to use an array or structure
      * accessor on a primitive type.
      * For instance:
      * @code
      * int a;
      * a[2] = 10; // trigger variableIsPrimitive()
      * @endcode
      */
    ErrorMessage* variableIsPrimitive();
    /**
      * Error that happen when manipulating an array with a statement
      * not intended for arrays.
      * For instance:
      * @code
      * int a[2];
      * a.hello = 0;
      * @endcode
      */
    ErrorMessage* variableIsArray();
    /**
      * Error that happen when manipulating a structure with a statement
      * not intended for structures.
      * For instance:
      * @code
      * MyStruct a;
      * a[2] = 0;
      * a.size;
      * @endcode
      */
    ErrorMessage* variableIsStructure();
    /**
     * Error message that happen when an unknown member of a structure is used.
     * For instance:
     * @code
     * struct MyStruct {
     *  int member;
     * };
     * MyStruct a;
     * a.notamember;
     * @endcode
     */
    ErrorMessage* unknownStructureMember(const String& member);
    
    
  }
}

#endif
