/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_VALUE_H_
#define _GTLCORE_VALUE_H_

#include <vector>

namespace GTLCore {
  class Type;
  /**
   * Value is a class use to hold a primitive value, like a float, an integer or a boolean.
   * @ingroup GTLCore
   */
  class Value {
    public:
      Value();
      Value(float v);
      Value(bool v);
      Value(int v);
      Value(unsigned int v);
      Value(const std::vector< Value >& v, const GTLCore::Type* _type = 0);
      Value(const Value& v);
      Value operator=(const Value& rhs);
      ~Value();
    public:
      bool operator==(const Value& rhs) const;
    public:
      /**
       * @return the value as a float
       */
      float asFloat() const;
      /**
       * Set the value to be a float.
       */
      void setFloat( float _v );
      /**
       * @return the value as a boolean
       */
      bool asBoolean() const;
      /**
       * @return the value as an integer
       */
      int asInt32() const;
      /**
       * Set the value to be an integer
       */
      void setInt32(int v);
      /**
       * @return the value as an unsigned integer
       */
      unsigned int asUnsignedInt32() const;
      /**
       * @return the value as an array, or 0 if not an array
       */
      const std::vector< Value >* asArray() const;
      /**
       * @return the value as an array, or 0 if not an array
       */
      void setArray( const std::vector< Value >& _array, const GTLCore::Type* _type );
      /**
       * @return the type of the value.
       */
      const Type* type() const;
    private:
      struct Private;
      Private* const d;
  };
}

#endif
