/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ImageDCRegistry.h"

#include <llvm/System/Path.h>
#include <llvm/System/DynamicLibrary.h>

#include <GTLCore/Debug.h>
#include "ImageDC.h"

using namespace GTLImageIO;

struct ImageDCRegistry::Private {
  std::list< ImageDC* > imageDCS;
  GTLCore::String decodableFilter;
  GTLCore::String encodableFilter;
  void loadExtensionsIn(const GTLCore::String& _extensions);
};

void ImageDCRegistry::Private::loadExtensionsIn(const GTLCore::String& _extensions)
{
  llvm::sys::Path extensionPath(_extensions);
  std::set<llvm::sys::Path> paths;
  GTLCore::String errMsg;
  GTL_DEBUG("Loading extensions from " << _GTLIMAGEIO_EXTENSIONS_INSTALL_DIR_ );
  if( extensionPath.getDirectoryContents( paths, &errMsg))
  {
    GTL_DEBUG("Can't find exensions: " << errMsg );
  } else {
    for( std::set<llvm::sys::Path>::iterator it = paths.begin();
         it != paths.end(); ++it)
    {
      GTL_DEBUG("Loading: " << it->toString() );
      if( llvm::sys::DynamicLibrary::LoadLibraryPermanently( it->toString().c_str(), &errMsg ) )
      {
        GTL_ERROR("Can't load " << it->toString() << " : " << errMsg );
      }
    }
  }
}

ImageDCRegistry* global_instance = 0;

ImageDCRegistry::ImageDCRegistry() : d(new Private)
{
}

ImageDCRegistry::~ImageDCRegistry()
{
  delete d;
}

void ImageDCRegistry::initialise()
{
  d->loadExtensionsIn(_GTLIMAGEIO_EXTENSIONS_INSTALL_DIR_);
#ifdef _GTLIMAGEIO_EXTENSIONS_PNG_DIR_
  d->loadExtensionsIn(_GTLIMAGEIO_EXTENSIONS_PNG_DIR_);
#endif
#ifdef _GTLIMAGEIO_EXTENSIONS_RAW_DIR_
  d->loadExtensionsIn(_GTLIMAGEIO_EXTENSIONS_RAW_DIR_);
#endif
}

ImageDCRegistry* ImageDCRegistry::instance()
{
  if( not global_instance )
  {
    global_instance = new ImageDCRegistry();
    global_instance->initialise();
  }
  return global_instance;
}

void ImageDCRegistry::registerDC( ImageDC* _imageDC )
{
  d->imageDCS.push_back( _imageDC );
  d->decodableFilter += _imageDC->decodableFilter();
  d->encodableFilter += _imageDC->encodableFilter();
}

const ImageDC* ImageDCRegistry::decoder( const GTLCore::String& _fileName ) const
{
  for( std::list< ImageDC* >::iterator it = d->imageDCS.begin();
       it != d->imageDCS.end(); ++it)
  {
    if( (*it)->canDecodeImage( _fileName ) )
    {
      return *it;
    }
  }
  return 0;
}

const ImageDC* ImageDCRegistry::encoder( const GTLCore::String& _fileName ) const
{
  for( std::list< ImageDC* >::iterator it = d->imageDCS.begin();
       it != d->imageDCS.end(); ++it)
  {
    if( (*it)->canEncodeImage( _fileName ) )
    {
      return *it;
    }
  }
  return 0;
}

const GTLCore::String& ImageDCRegistry::decodableFilter() const
{
  return d->decodableFilter;
}

const GTLCore::String& ImageDCRegistry::encodableFilter() const
{
  return d->encodableFilter;
}
