/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Kernel.h"

#include <sstream>
#include <fstream>

#include <llvm/Module.h>
#include <llvm/ModuleProvider.h>

#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Region.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/wrappers/Allocate.h"

#include "Debug.h"
#include "CodeGenerator_p.h"
#include "Compiler_p.h"
#include "Wrapper_p.h"
#include "wrappers/ImageWrap_p.h"
#include "wrappers/RegionWrap_p.h"
#include "Library_p.h"

#include "Kernel_p.h"

using namespace OpenShiva;

Kernel::Kernel(const GTLCore::String& _name, int _channelsNb ) : Library( true, _channelsNb), d(new Private )
{
  d->self = this;
  d->evaluatePixelesFunction = 0;
  d->wrapper = 0;
}

Kernel::Kernel(int _channelsNb ) : Library( true, _channelsNb), d(new Private )
{
  d->self = this;
  d->evaluatePixelesFunction = 0;
  d->wrapper = 0;
}

Kernel::~Kernel()
{
  cleanup();
  delete d->wrapper;
  delete d;
}

void Kernel::cleanup()
{
  Library::cleanup();
  d->evaluatePixelesFunction = 0; // It's deleted by the moduleData
}

void Kernel::postCompilation()
{
  // Create a wrapper
  d->wrapper = new Wrapper(this, Library::d->m_moduleData);
  d->determineTypes();
  // Create the generateEvaluatePixeles LLVM function
  d->evaluatePixelesFunction = CodeGenerator::generateEvaluatePixeles( this, Library::d->count_channels_generic );
  // Call evaluateDepends as needed
  GTLCore::Function* evaluateDependentsFunc = Library::d->m_moduleData->function( name(), "evaluateDependents");
  if( evaluateDependentsFunc )
  {

    void (*func)() = (void (*)())GTLCore::VirtualMachine::instance()->getPointerToFunction( evaluateDependentsFunc, 0 );
    (*func)();
  }
}

void Kernel::evaluatePixeles( const GTLCore::Region& _region, const std::list< GTLCore::AbstractImage* >& _inputImages, GTLCore::AbstractImage* _outputImage) const
{
  SHIVA_DEBUG( _region.x() << " " << _region.y() << " " << _region.width() << " " << _region.height());
  SHIVA_ASSERT( d->evaluatePixelesFunction );
  
  // Wrap input images
  SHIVA_ASSERT( d->wrapper );
  const void** inputImages = new const void*[ _inputImages.size() ];
  int i = 0;
  for( std::list< GTLCore::AbstractImage* >::const_iterator it = _inputImages.begin();
       it != _inputImages.end(); ++it)
  {
    inputImages[i] = (const void*)d->wrapper->wrapImage( *it );
  }
  
  // Wrap output image
  ImageWrap* owrap = d->wrapper->wrapImage( _outputImage );
  
  // Call function
  SHIVA_DEBUG("Call function");
  void (*func)( int, int, int, int, const void**, void*) = ( void(*)(int, int, int, int, const void**, void*))GTLCore::VirtualMachine::instance()->getPointerToFunction( d->evaluatePixelesFunction);
  SHIVA_ASSERT(func);
  (*func)( _region.x(), _region.y(), _region.width(), _region.height(), inputImages, owrap);
  for( std::size_t i = 0; i < _inputImages.size(); ++i)
  {
    delete (ImageWrap*)inputImages[i];
  }
  delete[] inputImages;
  delete owrap;
  SHIVA_DEBUG( "done" );
}

int Kernel::runTest() const
{
  SHIVA_ASSERT( isCompiled() );
  const GTLCore::Function* f = Library::d->m_moduleData->function( name(), "runTest");
  SHIVA_ASSERT( f );
  GTLCore::Value v = f->call( std::vector< GTLCore::Value >() );
  return v.asInt32();
}

bool Kernel::hasTestFunction() const
{
  return Library::d->m_moduleData->function( name(), "runTest");
}

GTLCore::Region Kernel::needed( GTLCore::Region output_region, int input_index, const std::list< GTLCore::Region>& input_DOD)
{
  GTLCore::Function* neededFunction = Library::d->m_moduleData->function( name(),"needed");
  RegionWrap* (*func)( RegionWrap*, int, ArrayWrap* ) = ( RegionWrap* (*)( RegionWrap*, int, ArrayWrap* ) )GTLCore::VirtualMachine::instance()->getPointerToFunction( neededFunction, 3);
  RegionWrap* rwrap = (*func)( regionToRegionWrap( output_region ), input_index, regionListToArrayWrap( input_DOD ) );
  
  GTLCore::Region region = regionWrapToRegion( rwrap );
  gtlFree( rwrap );
  return region;
}

bool Kernel::hasNeededFunction() const
{
  return Library::d->m_moduleData->function( name(), "needed");
}

GTLCore::Region Kernel::changed( GTLCore::Region changed_input_region, int input_index, const std::list< GTLCore::Region>& input_DOD)
{
  GTLCore::Function* changedFunction = Library::d->m_moduleData->function( name(), "changed");
  RegionWrap* (*func)( RegionWrap*, int, ArrayWrap* ) = ( RegionWrap* (*)( RegionWrap*, int, ArrayWrap* ) )GTLCore::VirtualMachine::instance()->getPointerToFunction( changedFunction, 3);
  ArrayWrap* aw = regionListToArrayWrap( input_DOD );
  RegionWrap* rwrap = (*func)( regionToRegionWrap( changed_input_region ), input_index, aw ); // TODO leak
  
  GTLCore::Region region = regionWrapToRegion( rwrap );
  gtlFree( rwrap );
  gtlFreeAllArray<RegionWrap*>( aw );
  return region;
}

bool Kernel::hasChangedFunction() const
{
  return Library::d->m_moduleData->function( name(), "changed");
}

GTLCore::Region Kernel::generated()
{
 GTLCore::Function* f = Library::d->m_moduleData->function( name(), "generated");
 SHIVA_ASSERT( f );

 RegionWrap* (*func)() = (RegionWrap* (*)())GTLCore::VirtualMachine::instance()->getPointerToFunction( f, 0 );
 
 RegionWrap* rwrap = (*func)();
 GTLCore::Region region = regionWrapToRegion( rwrap );
 gtlFree( rwrap );
 return region;
}

bool Kernel::hasGeneratedFunction() const
{
  return Library::d->m_moduleData->function( name(), "generated");
}
