/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Lexer_p.h"

using namespace OpenShiva;

#include "GTLCore/Token_p.h"

#include "Debug.h"

Lexer::Lexer(std::istream* istream) : GTLCore::LexerBase(istream)
{
}

Lexer::~Lexer()
{
}

GTLCore::Token Lexer::nextToken()
{
  int lastChar = getNextNonSeparatorChar();
  int initial_line = line() - 1;
  int initial_col = column() - 1;
  if( eof() ) return GTLCore::Token(GTLCore::Token::END_OF_FILE, line(), initial_col);
  GTLCore::String identifierStr;
  // Test for comment
  GTLCore::Token commentToken;
  if( ignoreComment( commentToken, lastChar ) )
  {
    return commentToken;
  }
  // if it is alpha, it's an identifier or a keyword
  if(isalpha(lastChar) or lastChar == '_')
  {
    identifierStr = getIdentifier(lastChar);
    IDENTIFIER_IS_KEYWORD( "and", AND );
    IDENTIFIER_IS_KEYWORD( "or", OR );
    IDENTIFIER_IS_KEYWORD( "not", NOT );
    IDENTIFIER_IS_KEYWORD( "kernel", KERNEL );
    IDENTIFIER_IS_KEYWORD( "library", LIBRARY );
    IDENTIFIER_IS_KEYWORD( "bool", BOOL );
    IDENTIFIER_IS_KEYWORD( "bool2", BOOL2 );
    IDENTIFIER_IS_KEYWORD( "bool3", BOOL3 );
    IDENTIFIER_IS_KEYWORD( "bool4", BOOL4 );
    IDENTIFIER_IS_KEYWORD( "const", CONST );
    IDENTIFIER_IS_KEYWORD( "float", FLOAT );
    IDENTIFIER_IS_KEYWORD( "float2", FLOAT2 );
    IDENTIFIER_IS_KEYWORD( "float3", FLOAT3 );
    IDENTIFIER_IS_KEYWORD( "float4", FLOAT4 );
    IDENTIFIER_IS_KEYWORD( "else", ELSE );
    IDENTIFIER_IS_KEYWORD( "for", FOR );
    IDENTIFIER_IS_KEYWORD( "if", IF );
    IDENTIFIER_IS_KEYWORD( "import", IMPORT );
    IDENTIFIER_IS_KEYWORD( "int", INT );
    IDENTIFIER_IS_KEYWORD( "int2", INT2 );
    IDENTIFIER_IS_KEYWORD( "int3", INT3 );
    IDENTIFIER_IS_KEYWORD( "int4", INT4 );
    IDENTIFIER_IS_KEYWORD( "long", LONG );
    IDENTIFIER_IS_KEYWORD( "return", RETURN );
    IDENTIFIER_IS_KEYWORD( "size", SIZE );
    IDENTIFIER_IS_KEYWORD( "struct", STRUCT );
    IDENTIFIER_IS_KEYWORD( "unsigned", UNSIGNED );
    IDENTIFIER_IS_KEYWORD( "void", VOID );
    IDENTIFIER_IS_KEYWORD( "while", WHILE );
    IDENTIFIER_IS_KEYWORD( "input", INPUT );
    IDENTIFIER_IS_KEYWORD( "output", OUTPUT );
    IDENTIFIER_IS_KEYWORD( "in", INPUT );
    IDENTIFIER_IS_KEYWORD( "out", OUTPUT );
    IDENTIFIER_IS_KEYWORD( "print", PRINT );
    IDENTIFIER_IS_KEYWORD( "true", TTRUE );
    IDENTIFIER_IS_KEYWORD( "false", TFALSE );
    IDENTIFIER_IS_KEYWORD( "dependent", DEPENDENT );
    return GTLCore::Token(GTLCore::Token::IDENTIFIER, identifierStr,line(), initial_col);
  } else if( isdigit(lastChar) )
  { // if it's a digit 
    return getDigit(lastChar);
  } else if( lastChar == '"' ) {
    return getString(lastChar);
  } else {
    CHAR_IS_TOKEN(';', SEMI );
    if( lastChar == ':' and getNextChar() == ':' )
    {
        return GTLCore::Token(GTLCore::Token::COLONCOLON, line(), initial_col);
    }
    CHAR_IS_TOKEN( ',', COMA );
    CHAR_IS_TOKEN( '.', DOT );
    CHAR_IS_TOKEN( ':', COLON );
    CHAR_IS_TOKEN( '{', STARTBRACE );
    CHAR_IS_TOKEN( '}', ENDBRACE );
    CHAR_IS_TOKEN( '(', STARTBRACKET );
    CHAR_IS_TOKEN( ')', ENDBRACKET );
    CHAR_IS_TOKEN( '[', STARTBOXBRACKET );
    CHAR_IS_TOKEN( ']', ENDBOXBRACKET );
    CHAR_IS_TOKEN_OR_TOKEN( '=', '=', EQUAL, EQUALEQUAL );
    CHAR_IS_TOKEN_OR_TOKEN( '!', '=', NOT, DIFFERENT );
    CHAR_IS_TOKEN_OR_TOKEN( '&', '&', BITAND, AND);
    CHAR_IS_TOKEN_OR_TOKEN( '|', '|', BITOR, OR );
    CHAR_IS_TOKEN( '^', BITXOR );
    CHAR_IS_TOKEN_OR_TOKEN_OR_TOKEN( '<', '=', '<', INFERIOR, INFERIOREQUAL, LEFTSHIFT );
    CHAR_IS_TOKEN_OR_TOKEN_OR_TOKEN( '>', '=', '>', SUPPERIOR, SUPPERIOREQUAL, RIGHTSHIFT );
    CHAR_IS_TOKEN_OR_TOKEN_OR_TOKEN( '+', '+', '=', PLUS, PLUSPLUS, PLUSEQUAL );
    CHAR_IS_TOKEN_OR_TOKEN_OR_TOKEN( '-', '-', '=', MINUS, MINUSMINUS, MINUSEQUAL );
    CHAR_IS_TOKEN_OR_TOKEN( '*', '=', MULTIPLY, MULTIPLYEQUAL);
    CHAR_IS_TOKEN_OR_TOKEN( '/', '=', DIVIDE, DIVIDEEQUAL);
    CHAR_IS_TOKEN( '%', MODULO );
    CHAR_IS_TOKEN( '~', TILDE );
  }
  if( lastChar > 128 ) return nextToken();
  identifierStr = lastChar;
  SHIVA_DEBUG("Unknown token : " << lastChar << " '" << identifierStr << "' at " << initial_line << "," << initial_col);
  SHIVA_ASSERT( not isspace(lastChar));
  return GTLCore::Token(GTLCore::Token::UNKNOWN, initial_line, initial_col);
}

