/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Library.h"
#include "Library_p.h"

#include <sstream>

#include <llvm/Module.h>
#include <llvm/ModuleProvider.h>
#include <llvm/System/DynamicLibrary.h>

#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Region.h"
#include "GTLCore/Type.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/wrappers/Allocate.h"
#include "GTLCore/Metadata/Group.h"
#include "GTLCore/Metadata/ParameterEntry.h"
#include "Metadata.h"
#include "GTLCore/Metadata/Factory_p.h"

#include "Debug.h"
#include "CodeGenerator_p.h"
#include "Compiler_p.h"
#include "Wrapper_p.h"
#include "wrappers/ImageWrap_p.h"
#include "wrappers/RegionWrap_p.h"


#include "Kernel_p.h"

using namespace OpenShiva;

void Library::Private::metadataToParameters( const GTLCore::Metadata::Group* group)
{
  SHIVA_ASSERT( group );
  foreach( const GTLCore::Metadata::Entry* entry, group->entries() )
  {
    if( const GTLCore::Metadata::ParameterEntry* pe = entry->asParameterEntry() )
    {
      parameters[ pe->name() ] = pe->defaultValue();
    } else if( const GTLCore::Metadata::Group* ge = entry->asGroup() )
    {
      metadataToParameters( ge );
    }
  }
}

void Library::Private::initparameters()
{
  const Metadata* metadata = source.metadata();
  parameters.clear();
  if( metadata and metadata->parameters() )
  {
    metadataToParameters( metadata->parameters() );
  }
}

// -------- Library -------- //

Library::Library( bool _isKernel , int _channelsNb) : d(new Private)
{
  GTL_ASSERT(_channelsNb > 0);
  d->name = "";
  d->compiled = false;
  d->moduleProvider = 0;
  d->m_moduleData = 0;
  d->count_channels_generic = _channelsNb;
  d->isKernel = _isKernel;
  d->isStandardLibrary = false;
}

Library::~Library()
{
  cleanup();
  delete d;
}

void Library::cleanup()
{
  if(d->moduleProvider)
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->moduleProvider);
    delete d->moduleProvider;
    d->moduleProvider = 0;
  }
  delete d->m_moduleData;
  d->m_moduleData = 0;
}


const GTLCore::String& Library::name() const
{
  return d->name;
}

void Library::setSource(const GTLCore::String& _source )
{
  d->source.setSource(_source);
  d->initparameters();
}

void Library::setSource(const Source& source)
{
    d->source = source;
    d->initparameters();
}

const Source& Library::source() const
{
    return d->source;
}

void Library::loadFromFile(const GTLCore::String& _fileName)
{
  d->isStandardLibrary = _fileName.endWith( "shivastdlib.shiva" );
  d->source.loadFromFile(_fileName);
  d->initparameters();
}

void Library::postCompilation()
{
}

void Library::compile()
{
  if(not d->source.metadata() )
  {
    d->compilationErrors = d->source.metadataCompilationErrors();
    return;
  }
  if(d->source.source().empty()) return;
  cleanup();
  d->m_moduleData = new GTLCore::ModuleData(new llvm::Module(d->name));
  Compiler c( d->isKernel,  d->count_channels_generic );
  Wrapper::fillTypesManager( d->m_moduleData, d->m_moduleData->typesManager(), c.convertCenter() , d->count_channels_generic );
  GTLCore::String nameSpace;
  bool result = c.compile( d->isStandardLibrary, d->source.source(), d->name, d->m_moduleData, nameSpace, d->parameters );

  if(result)
  {
    d->compiled = true;
    // Register the compiled module in the virtual machine
    llvm::sys::DynamicLibrary::LoadLibraryPermanently( _OPENSHIVA_LIB_, 0 ); // This needed because when OpenShiva is used in a plugins, OpenShiva symbols aren't loaded globally and then llvm can't find them (FIXME move that somewhere else)
    d->m_moduleData->doLink();
    d->moduleProvider = new llvm::ExistingModuleProvider( d->m_moduleData->llvmModule() );
    GTLCore::VirtualMachine::instance()->registerModule( d->moduleProvider );
    d->name = nameSpace;
    postCompilation();
    SHIVA_ASSERT( d->isStandardLibrary or d->name != "" );
  } else {
    d->compiled = false;
    cleanup();
    d->compilationErrors = c.errorMessages();
  }
  
}

GTLCore::String Library::compilationErrorsMessage() const
{
  std::ostringstream os;
  for( std::list<GTLCore::ErrorMessage>::iterator it = d->compilationErrors.begin();
       it != d->compilationErrors.end(); ++it)
  {
    os << it->fileName() << " at " << it->line() << " : " << it->errorMessage()  << std::endl;
  }
  return os.str();
}


bool Library::isCompiled() const
{
  return d->compiled;
}

const std::list<GTLCore::ErrorMessage>& Library::compilationErrors() const
{
  return d->compilationErrors;
}

GTLCore::String Library::asmSourceCode() const
{
  std::ostringstream os;
  os << *d->m_moduleData->llvmModule() << std::endl;
  return os.str();
}

const GTLCore::ModuleData* Library::data() const
{
  return d->m_moduleData;
}

std::list<GTLCore::Function*> Library::functions()
{
  return d->m_moduleData->functions();
}

const Metadata* Library::metadata() const
{
  return d->source.metadata();
}

void Library::setParameter( const GTLCore::String& _name, const GTLCore::Value& _value )
{
  d->parameters[ _name ] = _value;
}

void Library::setParameters( const std::map< GTLCore::String, GTLCore::Value>&  _values )
{
  for( std::map< GTLCore::String, GTLCore::Value>::const_iterator it = _values.begin();
       it != _values.end(); ++it)
  {
    d->parameters[ it->first ] = it->second;
  }
}
