/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Source.h"

using namespace OpenShiva;

#include <fstream>

#include <GTLCore/String.h>

#include "Debug.h"
#include "Lexer_p.h"
#include "LightParser_p.h"
#include "MetadataLexer_p.h"
#include "MetadataParser_p.h"
#include "Metadata.h"
#include "GTLCore/ErrorMessage.h"

#include "GTLCore/Metadata/Factory_p.h"

struct Source::Private
{
  Private() : metadata(0), metadataCompilationFailed(false), uptodate(false), type( InvalidSource ) {}
  void compileMetaData();
  void update();
  GTLCore::String name;
  GTLCore::String source;
  Metadata* metadata;
  bool metadataCompilationFailed;
  std::list<GTLCore::ErrorMessage> compilationErrors;
  bool uptodate; ///< Hold wether name and type have been extracted from source
  SourceType type;
  ImageType outputImageType;
  std::vector<ImageType> inputImageTypes;
};


// ---- Library::Private ---- //

void Source::Private::compileMetaData()
{
  metadata = 0;
  compilationErrors.clear();
  std::istringstream iss(source);
  MetadataLexer* lexer = new MetadataLexer(&iss);
  MetadataParser parser( lexer, "" );
  metadata = parser.parse();
  compilationErrors = parser.errorMessages();
  metadataCompilationFailed = not compilationErrors.empty();
  GTL_DEBUG( compilationErrors.size() );
  GTL_DEBUG( parser.errorMessages().size() );
  if( compilationErrors.size() != 0 )
  {
    GTLCore::Metadata::Factory::deleteEntry( metadata );
    metadata = 0;
  }
}

void Source::Private::update()
{
  if(uptodate) return;
  std::istringstream iss(source);
  Lexer lexer( &iss );
  LightParser parser(&lexer);
  parser.parse();
  name = parser.name();
  type = parser.sourceType();
  outputImageType = parser.outputImageType();
  inputImageTypes = parser.inputImageTypes();
  if( type != Library)
  {
    switch( inputImageTypes.size() )
    {
      case 0:
        type = GeneratorKernel;
        break;
      case 1:
        type = FilterKernel;
        break;
      default:
        type = CompositionKernel;
        break;
    }
  }
  uptodate = true;
}

Source::Source() : d(new Private)
{
  
}

Source::Source(const Source& _rhs) : d(new Private(*_rhs.d))
{
  d->metadata = 0;
}

Source& Source::operator=(const Source& _rhs)
{
  *d = *_rhs.d;
  return *this;
}

Source::~Source()
{
  delete d;
}

const GTLCore::String& Source::name() const
{
  d->update();
  return d->name;
}
const GTLCore::String& Source::source() const
{
  return d->source;
}

void Source::setSource( const GTLCore::String& _source)
{
  GTLCore::Metadata::Factory::deleteEntry( d->metadata );
  d->metadata = 0;
  d->metadataCompilationFailed = false;
  d->uptodate = false;
  d->source = _source;
}

void Source::loadFromFile(const GTLCore::String& _fileName)
{
  GTLCore::String source = "";
  std::ifstream in;
  in.open(_fileName.c_str() );
  if(not in)
  {
    SHIVA_DEBUG( "Impossible to open file " << _fileName );
    return;
  }
  GTLCore::String str;
  std::getline(in,str);
  while ( in ) {
    source += str;
    source += "\n";
    std::getline(in,str);
  }
  setSource(source);
}

const Metadata* Source::metadata() const
{
  if(not d->metadata and not d->metadataCompilationFailed)
  {
    d->compileMetaData();
  }
  return d->metadata;
}

const std::list<GTLCore::ErrorMessage>& Source::metadataCompilationErrors() const
{
  return d->compilationErrors;
}

Source::SourceType Source::sourceType() const
{
  d->update();
  return d->type;
}

Source::ImageType Source::outputImageType() const
{
  return d->outputImageType;
}

Source::ImageType Source::inputImageType(int idx) const
{
  if( idx >= 0 and (std::size_t)idx < d->inputImageTypes.size())
  {
    return d->inputImageTypes[idx];
  }
  return InvalidImage;
}

int Source::countInputImages() const
{
  return d->inputImageTypes.size();
}
