/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Wrapper_p.h"

#include <map>
#include <vector>

#include <llvm/DerivedTypes.h>
#include <llvm/Function.h>
#include <llvm/Module.h>

#include "GTLCore/Function_p.h"
#include "GTLCore/Macros.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/Parameter.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/TypesManager_p.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"

#include "GTLCore/AbstractImage.h"
#include "CodeGenerator_p.h"
#include "Debug.h"
#include "PixelVisitor_p.h"
#include "PixelConvertExpressionFactory_p.h"

#include "wrappers/ImageWrap_p.h"

using namespace OpenShiva;

struct WrappedFunctions {
  void* memToVec;
  void* vecToMem;
};

typedef std::map< GTLCore::PixelDescription, WrappedFunctions > pdToWF;
typedef pdToWF::iterator pdToWF_it;
typedef pdToWF::const_iterator pdToWF_cit;

struct Wrapper::Private {
  pdToWF imageFunctions;
  Kernel* kernel;
  GTLCore::ModuleData* moduleData;
};

Wrapper::Wrapper(Kernel* _kernel, GTLCore::ModuleData* _moduleData) : d(new Private)
{
  d->kernel = _kernel;
  d->moduleData = _moduleData;
}

Wrapper::~Wrapper()
{
  delete d;
}

ImageWrap* Wrapper::wrapImage(GTLCore::AbstractImage* _abstractImage)
{
  SHIVA_DEBUG("wrapImage");
  ImageWrap* owrap = new ImageWrap;
  owrap->image = _abstractImage;
  pdToWF_it it = d->imageFunctions.find( _abstractImage->pixelDescription() );
  if( it == d->imageFunctions.end() )
  {
    // Wrap functions doesn't exist yet, generate them
    WrappedFunctions wf;
    wf.memToVec = GTLCore::VirtualMachine::instance()->getPointerToFunction(
        CodeGenerator::generateMemToVec( d->moduleData, _abstractImage->pixelDescription()  ) );
    wf.vecToMem = GTLCore::VirtualMachine::instance()->getPointerToFunction(
        CodeGenerator::generateVecToMem( d->moduleData, _abstractImage->pixelDescription()  ) );
    d->imageFunctions[ _abstractImage->pixelDescription() ] = wf;
    owrap->memToVec = wf.memToVec;
    owrap->vecToMem = wf.vecToMem;
  } else {
    // Use allready existing wrap function
    owrap->memToVec = it->second.memToVec;
    owrap->vecToMem = it->second.vecToMem;
  }
  SHIVA_DEBUG("Image wrapped");
  return owrap;
}

void Wrapper::fillTypesManager( GTLCore::ModuleData* _module, GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter, int _channels )
{
  GTL_ASSERT(_channels > 0 );
  SHIVA_DEBUG("fillTypesManager");
  _convertCenter->addConvertExpressionFactory(new PixelConvertExpressionFactory);
  for( int i = 1; i <= 5; ++i)
  {
    // Create Image type
    const GTLCore::Type* pixelType = createPixelType( _module, _typesManager, _convertCenter, GTLCore::String::number( i ), i );
    createImageType( _module, _typesManager, GTLCore::String::number( i ), i, pixelType );
  }
  const GTLCore::Type* pixelType = createPixelType( _module, _typesManager, _convertCenter, "", _channels );
  createImageType( _module, _typesManager, "", _channels, pixelType );
  createRegionType( _module, _typesManager );
}

const GTLCore::Type* Wrapper::createPixelType( GTLCore::ModuleData* _module, GTLCore::TypesManager* _typesManager, GTLCore::ConvertCenter* _convertCenter, const GTLCore::String& _suffix, int _channels )
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // it's llvm declaration must be changed too in        //
  // struct PixelWrap !                                  //
  //---------------------- WARNING ----------------------//
  
  const GTLCore::Type* vecType = vectorType( _typesManager, _channels );
  std::vector<GTLCore::Type::StructDataMember> pixelDataMembers;
  pixelDataMembers.push_back( GTLCore::Type::StructDataMember( "data", vecType ) );
  
  pixelDataMembers.push_back( GTLCore::Type::StructDataMember( "coord", _typesManager->getVector( GTLCore::Type::Float, 2 ) ) );
  
  const GTLCore::Type* type = _typesManager->d->createStructure( "pixel" + _suffix, pixelDataMembers );
  type->d->setVisitor( PixelVisitor::instance() );
  
  _convertCenter->addAutoConversion( type, vecType );
  
  return type;
}

void Wrapper::createRegionType( GTLCore::ModuleData* _module, GTLCore::TypesManager* _typesManager )
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // it's llvm declaration must be changed too in        //
  // struct RegionWrap !                                 //
  //---------------------- WARNING ----------------------//
  std::vector<GTLCore::Type::StructDataMember> regionDataMembers;
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "x", GTLCore::Type::Float ) );
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "y", GTLCore::Type::Float) );
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "width", GTLCore::Type::Float ) );
  regionDataMembers.push_back( GTLCore::Type::StructDataMember( "height", GTLCore::Type::Float ) );
  const GTLCore::Type* type = _typesManager->d->createStructure( "region", regionDataMembers);
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "left", "region_wrap_left", GTLCore::Type::Float, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "right", "region_wrap_right", GTLCore::Type::Float, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "bottom", "region_wrap_bottom", GTLCore::Type::Float, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "top", "region_wrap_top", GTLCore::Type::Float, 1,
                type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "intersect", "region_wrap_intersect", GTLCore::Type::Void, 2,
                type, type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "union", "region_wrap_union", GTLCore::Type::Void, 2,
                type, type ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "outset", "region_wrap_outset", GTLCore::Type::Void, 2,
                type, GTLCore::Type::Float ) ) );
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createExternalFunction(
                _module, "inset", "region_wrap_inset", GTLCore::Type::Void, 2,
                type, GTLCore::Type::Float ) ) );
}

llvm::Function* Wrapper::image_wrap_dataFunction( llvm::Module* _module, const GTLCore::Type* _imageType )
{
  GTL_ASSERT( _imageType );
  std::vector<const llvm::Type*> functionArgs;
  functionArgs.push_back(llvm::PointerType::get( _imageType->d->type(), 0));
  functionArgs.push_back(llvm::IntegerType::get(32));
  functionArgs.push_back(llvm::IntegerType::get(32));
  llvm::FunctionType* functionTy = llvm::FunctionType::get( llvm::PointerType::get(llvm::IntegerType::get(8), 0), functionArgs, false);
  return (llvm::Function*)_module->getOrInsertFunction( "image_wrap_data", functionTy);
}

llvm::FunctionType* Wrapper::image_wrap_sample_nearest_type( GTLCore::TypesManager* _typesManager, const GTLCore::Type* _imageType, const GTLCore::Type* _pixelType )
{
  std::vector<const llvm::Type*> functionArgs;
  functionArgs.push_back( _imageType->d->pointerType() );
  functionArgs.push_back( _typesManager->getVector( GTLCore::Type::Float, 2 )->d->type() );
  return llvm::FunctionType::get( _pixelType->d->pointerType() , functionArgs, false);
}

llvm::FunctionType* Wrapper::image_wrap_mem_to_vec_float_type( GTLCore::TypesManager* _typesManager, int _channels )
{
  std::vector<const llvm::Type*> functionArgs;
  functionArgs.push_back( llvm::PointerType::get( vectorType( _typesManager, _channels)->d->type(), 0) );
  functionArgs.push_back( llvm::PointerType::get( llvm::IntegerType::get(8), 0));
  return llvm::FunctionType::get( llvm::Type::VoidTy, functionArgs, false);
}

llvm::FunctionType* Wrapper::image_wrap_vec_float_to_mem_type( GTLCore::TypesManager* _typesManager, int _channels )
{
  std::vector<const llvm::Type*> functionArgs;
  functionArgs.push_back( llvm::PointerType::get( llvm::IntegerType::get(8), 0));
  functionArgs.push_back( llvm::PointerType::get( vectorType( _typesManager, _channels)->d->type(), 0) );
  return llvm::FunctionType::get( llvm::Type::VoidTy, functionArgs, false);
}

void Wrapper::createImageType( GTLCore::ModuleData* _moduleData, GTLCore::TypesManager* _typesManager, const GTLCore::String& _suffix, int _channels, const GTLCore::Type* _pixelType )
{
  //---------------------- WARNING ----------------------//
  // Whenever the following structure is edited,         //
  // it's llvm declaration must be changed too in        //
  // struct ImageWrap !                                  //
  //---------------------- WARNING ----------------------//
  std::vector<GTLCore::Type::StructDataMember> imageDataMembers;
  imageDataMembers.push_back( GTLCore::Type::StructDataMember( "image", GTLCore::Type::Pointer) );
  imageDataMembers.push_back( GTLCore::Type::StructDataMember(
        "memToVec",
        GTLCore::Type::Private::createArbitraryType(
            llvm::PointerType::get( image_wrap_mem_to_vec_float_type( _typesManager, _channels ), 0 ) ) ) ); // FIXME arbitraty type are leaking
  imageDataMembers.push_back( GTLCore::Type::StructDataMember(
        "vecToMem",
        GTLCore::Type::Private::createArbitraryType(
            llvm::PointerType::get( image_wrap_vec_float_to_mem_type( _typesManager, _channels ), 0 ) ) ) ); // FIXME arbitraty type are leaking
  const GTLCore::Type* type = _typesManager->d->createStructure( "image" + _suffix, imageDataMembers );
  
  type->d->addFunctionMember( GTLCore::Type::StructFunctionMember(
          GTLCore::Function::Private::createInternalFunction(
                _moduleData, "sampleNearest",
                CodeGenerator::generateImageSampleNearest( _moduleData, type, _pixelType ),
                _pixelType, 2,
                type, _moduleData->typesManager()->getVector( GTLCore::Type::Float, 2 ) ) ) );
}

const GTLCore::Type* Wrapper::vectorType( GTLCore::TypesManager* _typesManager, int _channels )
{
  if( _channels == 1)
  {
    return GTLCore::Type::Float;
  }
  return _typesManager->getVector( GTLCore::Type::Float, _channels);
}
