/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "OpenShiva/Lexer_p.h"
#include "GTLCore/Token_p.h"

#define TEST_FOR_TOKEN( toktype ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
  }

#define TEST_FOR_TOKEN_AND_POS( toktype, _line_, _column_ ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
    GTLTEST_CHECK_EQUAL(tok.line, _line_ ); \
    GTLTEST_CHECK_EQUAL(tok.column, _column_ ); \
  }

#define TEST_FOR_TOKEN_AND_VALUE( toktype, field, value ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
    GTLTEST_CHECK_NEAR_EQUAL(tok.field, value); \
  }

#define TEST_FOR_TOKEN_AND_VALUE_STRICT( toktype, field, value ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
    GTLTEST_CHECK_EQUAL(tok.field, value); \
  }

class TestLexerKeywords : public GTLTest::Case {
  public:
    TestLexerKeywords() : GTLTest::Case("Keywords") {}
    virtual void runTest()
    {
      std::istringstream iss(" bool bool2 bool3 bool4 const float float2 float3 float4 else for if import int int2 int3 int4 long return size struct void while kernel input output print true false");
      OpenShiva::Lexer lng( &iss);
      TEST_FOR_TOKEN( BOOL );
      TEST_FOR_TOKEN( BOOL2 );
      TEST_FOR_TOKEN( BOOL3 );
      TEST_FOR_TOKEN( BOOL4 );
      TEST_FOR_TOKEN( CONST );
      TEST_FOR_TOKEN( FLOAT );
      TEST_FOR_TOKEN( FLOAT2 );
      TEST_FOR_TOKEN( FLOAT3 );
      TEST_FOR_TOKEN( FLOAT4 );
      TEST_FOR_TOKEN( ELSE );
      TEST_FOR_TOKEN( FOR );
      TEST_FOR_TOKEN( IF );
      TEST_FOR_TOKEN( IMPORT );
      TEST_FOR_TOKEN( INT );
      TEST_FOR_TOKEN( INT2 );
      TEST_FOR_TOKEN( INT3 );
      TEST_FOR_TOKEN( INT4 );
      TEST_FOR_TOKEN( LONG );
      TEST_FOR_TOKEN( RETURN );
      TEST_FOR_TOKEN( SIZE );
      TEST_FOR_TOKEN( STRUCT );
      TEST_FOR_TOKEN( VOID );
      TEST_FOR_TOKEN( WHILE );
      TEST_FOR_TOKEN( KERNEL );
      TEST_FOR_TOKEN( INPUT );
      TEST_FOR_TOKEN( OUTPUT );
      TEST_FOR_TOKEN( PRINT );
      TEST_FOR_TOKEN( TTRUE );
      TEST_FOR_TOKEN( TFALSE );
    }
};

class TestLexerConstants : public GTLTest::Case {
  public:
    TestLexerConstants() : GTLTest::Case("Constants") {}
    virtual void runTest()
    {
      std::istringstream iss(" 10 10.0 10e-7 20 30. 42.0 \"hello\" 8 \"escape \\\"me\\\"\"");
      OpenShiva::Lexer lng( &iss);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( INTEGER_CONSTANT, i, 10);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 10.0);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 10e-7);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( INTEGER_CONSTANT, i, 20);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 30.);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 42.0);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( STRING_CONSTANT, string, "hello");
      TEST_FOR_TOKEN_AND_VALUE_STRICT( INTEGER_CONSTANT, i, 8);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( STRING_CONSTANT, string, "escape \\\"me\\\"");
  }
};

class TestLexerSpecialCharacters : public GTLTest::Case {
  public:
    TestLexerSpecialCharacters() : GTLTest::Case("SpecialCharacters") {}
    virtual void runTest()
    {
      std::istringstream iss(" ; :: , . { } ( ) [ ] = == ! != & && || | ^ < <= << > >= >> + ++ - -- * / % ~");
      OpenShiva::Lexer lng( &iss);
      TEST_FOR_TOKEN( SEMI );
      TEST_FOR_TOKEN( COLONCOLON );
      TEST_FOR_TOKEN( COMA );
      TEST_FOR_TOKEN( DOT );
      TEST_FOR_TOKEN( STARTBRACE );
      TEST_FOR_TOKEN( ENDBRACE );
      TEST_FOR_TOKEN( STARTBRACKET );
      TEST_FOR_TOKEN( ENDBRACKET );
      TEST_FOR_TOKEN( STARTBOXBRACKET );
      TEST_FOR_TOKEN( ENDBOXBRACKET );
      TEST_FOR_TOKEN( EQUAL );
      TEST_FOR_TOKEN( EQUALEQUAL );
      TEST_FOR_TOKEN( NOT );
      TEST_FOR_TOKEN( DIFFERENT );
      TEST_FOR_TOKEN( BITAND );
      TEST_FOR_TOKEN( AND );
      TEST_FOR_TOKEN( OR );
      TEST_FOR_TOKEN( BITOR );
      TEST_FOR_TOKEN( BITXOR );
      TEST_FOR_TOKEN( INFERIOR );
      TEST_FOR_TOKEN( INFERIOREQUAL );
      TEST_FOR_TOKEN( LEFTSHIFT );
      TEST_FOR_TOKEN( SUPPERIOR );
      TEST_FOR_TOKEN( SUPPERIOREQUAL );
      TEST_FOR_TOKEN( RIGHTSHIFT );
      TEST_FOR_TOKEN( PLUS );
      TEST_FOR_TOKEN( PLUSPLUS );
      TEST_FOR_TOKEN( MINUS );
      TEST_FOR_TOKEN( MINUSMINUS );
      TEST_FOR_TOKEN( MULTIPLY );
      TEST_FOR_TOKEN( DIVIDE );
      TEST_FOR_TOKEN( MODULO );
      TEST_FOR_TOKEN( TILDE );
    }
};

class TestLexerNoSpace : public GTLTest::Case {
  public:
    TestLexerNoSpace() : GTLTest::Case("NoSpace") {}
    virtual void runTest()
    {
      std::istringstream iss("32)=(<<<3<==");
      OpenShiva::Lexer lng( &iss);
      TEST_FOR_TOKEN( INTEGER_CONSTANT );
      TEST_FOR_TOKEN( ENDBRACKET );
      TEST_FOR_TOKEN( EQUAL );
      TEST_FOR_TOKEN( STARTBRACKET );
      TEST_FOR_TOKEN( LEFTSHIFT );
      TEST_FOR_TOKEN( INFERIOR );
      TEST_FOR_TOKEN( INTEGER_CONSTANT );
      TEST_FOR_TOKEN( INFERIOREQUAL );
      TEST_FOR_TOKEN( EQUAL );
  }
};

class TestLexerIdentifier : public GTLTest::Case {
  public:
    TestLexerIdentifier() : GTLTest::Case("Identifier") {}
    virtual void runTest()
    {
      std::istringstream iss(" HOHOH BOUH23 GLOB_");
      OpenShiva::Lexer lng(&iss);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( IDENTIFIER, string, "HOHOH");
      TEST_FOR_TOKEN_AND_VALUE_STRICT( IDENTIFIER, string, "BOUH23");
      TEST_FOR_TOKEN_AND_VALUE_STRICT( IDENTIFIER, string, "GLOB_");
    }
};

class TestLexerPos : public GTLTest::Case {
  public:
    TestLexerPos() : GTLTest::Case("Pos") {}
    virtual void runTest()
    {
      std::istringstream iss(" HELLO\nWORLD !");
      OpenShiva::Lexer lng(&iss);
      TEST_FOR_TOKEN_AND_POS( IDENTIFIER, 1, 2);
      TEST_FOR_TOKEN_AND_POS( IDENTIFIER, 2, 1);
      TEST_FOR_TOKEN_AND_POS( NOT, 2, 7);
    }
};


class TestLexerComments : public GTLTest::Case {
  public:
    TestLexerComments() : GTLTest::Case("Comments") {}
    virtual void runTest()
    {
      std::istringstream iss(" /* HELLO\nWORLD ! */ + // BOUH \n -");
      OpenShiva::Lexer lng(&iss);
      TEST_FOR_TOKEN( PLUS );
      TEST_FOR_TOKEN( MINUS );
    }
};

class TestLexer : public GTLTest::Suite {
  public:
    TestLexer() : GTLTest::Suite("Lexer")
    {
      addCase(new TestLexerConstants );
      addCase(new TestLexerKeywords );
      addCase(new TestLexerSpecialCharacters );
      addCase(new TestLexerNoSpace );
      addCase(new TestLexerPos );
      addCase(new TestLexerIdentifier );
      addCase(new TestLexerComments );
    }
};
