# agent-broadcast.tcl --
#
#       Derived from VideoAgent, responsible for broadcasting onto MBone the
#       director selected streams.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import VideoAgent

#-----------------------------------------------------------------------------
# Class:
#   DcBroadcastAgent
#
# Description:
#   Derived from VideoAgent.  DcBroadcastAgent takes video packets from
#   the Studio session, pass it through a switcher, and send it out to
#   out to a broadcast session.  
#
# Members:
#   switcher_ -- the Module/RTPSwitcher object.
#-----------------------------------------------------------------------------
Class DcBroadcastAgent -superclass VideoAgent
DcBroadcastAgent set src_counter_ 0


#-----------------------------------------------------------------------------
# Method:
#   DcBroadcastAgent init
#
# Description:
#   Create a new switcher and set the target of the switcher to 
#   the transmitter.
#
# Arguments:
#   app -- the Module/RTPSwitcher object.
#   addr,port,ttl -- the IP address, port number and TTL of the output 
#                    session.
#-----------------------------------------------------------------------------
DcBroadcastAgent instproc init { app addr port ttl} {
    $self next $app "$addr/$port:$port//$ttl"

    $self instvar switcher_

    set switcher_ [new Module/RTPSwitcher]

    # set the local transmitter as the transmitter for this object
    $switcher_ target [$self get_transmitter]
}


#-----------------------------------------------------------------------------
# Method:
#   DcBroadcastAgent get_switcher
#
# Description:
#   Return the switcher object.
#-----------------------------------------------------------------------------
DcBroadcastAgent public get_switcher { } {
    return [$self set switcher_]
}


#-----------------------------------------------------------------------------
# Method:
#   DcBroadcastAgent map_source
#
# Description:
#   Ask the switcher to edit packet source to $osrcid everytime it sees
#   a packet with srcid $isrcid.  If $osrcid is empty, then it is set to
#   a generated srcid.
#-----------------------------------------------------------------------------
DcBroadcastAgent public map_source { isrcid { osrcid "" } } {
    $self instvar switcher_
    DcBroadcastAgent instvar src_counter_

    if { $osrcid == "" } {
	set osrcid [expr [$self get_local_srcid] + $src_counter_]
	incr src_counter_
    }
    $switcher_ MapSource $isrcid $osrcid

    return $osrcid
}


#-----------------------------------------------------------------------------
# Method:
#   DcBroadcastAgent unmap_source
#
# Description:
#   Ask the switcher to forget about the mapping set by map_source.  If
#   no such mapping exists, switcher ignores the request.
#-----------------------------------------------------------------------------
DcBroadcastAgent public unmap_source { isrcid osrcid } {
    $self instvar switcher_
    $switcher_ UnmapSource $isrcid $osrcid
}

