# application-replay.tcl --
#
#       Stuff for building the window that is popped up when the director
#       activates a Replay.  It creates a replayproxy service to talk to MARS
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Application
import SSAC_Client
import CSdsService
import CServiceManager
import CService

Class CReplayApplication -superclass Application

##############################################################################
#
# CReplayApplication instproc InitArgs { options } {
#
# Input:
# options - the options object
#
# Output:
# none
#
# Description:
# Registers all options before the command line argument is parsed
#
##############################################################################
CReplayApplication instproc InitArgs { options } {
    # for the service discovery service
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL

    # for the service discovery service
    $options register_option -rp optComPort

    # for the active services
    $options register_option -sas optSpecAS
}

##############################################################################
#
# CReplayApplication instproc InitResources { options } {
#
# Input:
# options - the options object
#
# Output:
# none
#
# Description:
# Gives defaults for options
#
##############################################################################
CReplayApplication instproc InitResources { options } {

    # for the service discovery service
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"

    # for the service discovery service
    $options add_default optComPort "11408"

    # for the active services
    $options add_default optSpecAS 224.4.5.24/50000/31
}


##############################################################################
#
# CReplayApplication instproc init { argv } {
#
# Input:
# argv - command line input
#
# Output:
# none
#
# Description:
# Constructor for the object.
#
##############################################################################
CReplayApplication instproc init { argv } {
    $self next Replay


    # Initiailization of options
    set options [$self options]
    $self InitArgs $options
    $self InitResources $options
    $options load_preferences "archival replay"
    set argv [$options parse_args $argv]


    # create the sds session object
    set inetServiceAddr [$options get_option optServiceAddress]
    set iServicePort [$options get_option optServicePort]
    set iServiceTTL [$options get_option optServiceTTL]

    $self instvar m_sdsService
    set m_sdsService [new CSdsService $self $inetServiceAddr $iServicePort \
	    $iServicePort $iServiceTTL]

    # setup the service objects
    $self instvar m_serviceManager
    $self instvar m_service

    set iPort [$options get_option optComPort]
    set m_serviceManager [new CServiceManager "ServiceApp" "Replay" $iPort]
    $m_serviceManager Attach $self

}

##############################################################################
#
# CReplayApplication public GetSdsServiceData { } {
#
# Input:
# none
#
# Output:
# the data that the will go out to the sds system
#
# Description:
# This is a callback function for the service discovery system.  Need to
# return the data that will go out to the sds system.  So far there are
# three fields with their values
#
##############################################################################
CReplayApplication public GetSdsServiceData { } {
    set options [$self options]
    set iComPort [$options get_option optComPort]
    set hostname [exec hostname]

    set location "Locationless"
    set type "Archive Replay"

    set data [list "UPDATE:" [concat $hostname $iComPort] \
	    [list [list "LOCATION:" $location] \
	    	  [list "TYPE:" $type]]]

    return $data
}

##############################################################################
#
# CReplayApplication instproc NewConnection { service }
#
# Input:
# service - the new service that got connected
#
# Output:
# 1 - if handling this service
# 0 - otherwise
#
# Description:
# A call back function for the service manager.  This function will be called
# when a new connection has just been noticed by the service manager
#
##############################################################################
CReplayApplication instproc NewConnection { service } {
    $self instvar m_service

    set m_service $service

    # Add the service calls
    $m_service MapMessage "SYN_SERVICE_IMAGE" $self SynServiceImage
    $m_service MapMessage "GET_UI_WINDOW" $self GetUIWindow
    $m_service MapMessage "START_REPLAY" $self StartReplay
    $m_service MapMessage "CLOSE_LINK" $self CloseService

    return 1
}

##############################################################################
#
# CReplayApplication public SynServiceImage { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the image or text
# to display the service.  The simple protocol goes as such:
# type type_dependent_arguments.  The type could be text and it's argument
# the ascii of title.  It could also be image and be a bitmap of the image
#
##############################################################################
CReplayApplication public SynServiceImage { service arguments } {
    $service Send "SYN_SERVICE_IMAGE" "text Replay"
}


##############################################################################
#
# CReplayApplication public GetUIWindow { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the code
# to make the window.  It should send back the code to build the window.
# There are two things that are assumed to be set by the client.  One is
# winFrame the window to draw into.  Another is service, which is the
# m_service object to send back to yourself.
#
##############################################################################
CReplayApplication public GetUIWindow { service arguments } {
    set cmd ""

    append cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    append cmd "wm title \$winFrame \"Replay Manager\" \n"

    append cmd "frame \$winFrame.catname\n"
    append cmd "pack \$winFrame.catname -side top -fill x -expand 1\n"

    append cmd "label \$winFrame.catname.label -text \"Archive Name:\" \n"
    append cmd "pack \$winFrame.catname.label -side left \n"
    append cmd "global catnameEntry\$__name \n"
    append cmd "set catnameEntry\$__name \
	    \"/n/vid22/mars/cs298-5/opening/cat.ctg\" \n"

    append cmd "entry \$winFrame.catname.entry \
	    -textvariable catnameEntry\$__name -relief sunken \n"
    append cmd "pack \$winFrame.catname.entry -side right -fill x -expand 1 \n"

    append cmd "frame \$winFrame.audioSession\n"
    append cmd "pack \$winFrame.audioSession -side top -fill x -expand 1\n"

    append cmd "label \$winFrame.audioSession.label -text \
	    \"Audio Session:\" \n"
    append cmd "pack \$winFrame.audioSession.label -side left \n"
    append cmd "global audioSessionEntry\$__name \n"
    append cmd "set audioSessionEntry\$__name 233.0.25.02/22446\n"
    append cmd "entry \$winFrame.audioSession.entry \
	    -textvariable audioSessionEntry\$__name -relief sunken \n"
    append cmd "pack \$winFrame.audioSession.entry -side right \
	    -fill x -expand 1 \n"

    # need to get the contact info and also options
    append cmd "frame \$winFrame.command \n"
    append cmd "pack \$winFrame.command -side top -fill x -expand 1\n"

    append cmd "button \$winFrame.command.closeButton -text Close \
	    -command \"destroy \$winFrame\" \n"
    append cmd "pack \$winFrame.command.closeButton -side right\n"

    append cmd "button \$winFrame.command.startButton -text Start \
	    -command \"global catnameEntry\$__name; global audioSessionEntry\__name; \$service Send START_REPLAY \\\"\$inetMStudioAddr \$iMStudioPort \$inetClientAddr \$iClientPort \\\$catnameEntry\$__name \\\$audioSessionEntry\$__name \\\" \"\n"
    append cmd "pack \$winFrame.command.startButton -side right\n"

    $service Send "GET_UI_WINDOW" $cmd
}


##############################################################################
#
# CReplayApplication public StartReplay { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the start the
# replay.
#
##############################################################################
CReplayApplication public StartReplay { service arguments } {

    # set up the arguments and launch the replay proxy
    set options [$self options]
    set specAS [$options get_option optSpecAS]

#    set strExec "dc-replayproxy-5.0b3"
    set strExec "dc-replayproxy"

    append strExec " -sas $specAS $arguments"

#    puts "strExec - $strExec"

    if { [catch "eval exec $strExec &" pid] != 0 } {
	puts "replay got an error starting the proxy: check the path"
	exit 0
    }
}


##############################################################################
#
# CReplayApplication public CloseService { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client closes the service link.
#
##############################################################################
CReplayApplication public CloseService { service arguments } {
    exit
}

