# application-hm.tcl --
#
#       Application object for Host Manager (hm)
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/hm/application-hm.tcl,v 1.52 2002/02/03 04:27:05 lim Exp $


import Application HMAgent

#-
# Class:
#   HMApplication
# Description:
#   Main application object for host manager (hm).  It is responsible
#   for parsing command lines arguments, initializing user-defined
#   resources and commands, and creating an HMAgent.  The intelligence
#   of the application lies in HMAgent.
# See Also:
#   HMAgent
#-
Class HMApplication -superclass Application

#-
# Method:
#   HMApplication init
# Description:
#   Initialize default values for command line arguments, parse
#   command line arguments, and load user hooks.  Finally, create
#   an HM agent and run user hooks.
#-
HMApplication public init argv {
	$self next hm
	set o [$self options]
	$self init_args $o
	$self init_resources $o
	$o parse_args $argv

	# Source the user's hook file if it exists. The function
	# user_hook, which may be defined in this file, will be
	# called at the end of init.
	if {[$o get_option userhookFile] != ""} {
		if {[file isfile [$o get_option userhookFile]] && \
			[file readable [$o get_option userhookFile]]} {
			source [$o get_option userhookFile]
		} else {
			puts stderr "Unable to source \"[$o get_option userhookFile]\". Not a file or not readable."
		}
	}

	$self init_local

	# FIXME closing stdout causes any puts statements to crash
	# hm if it's started in fork mode.
	if { [$self get_option doFork] != "" } {
		close stdout
		close stdin
		close stderr
		fork
		set logfd [open [$self get_option logFile] a+]
	} else {
		set logfd stdout
	}
	$self add_option execArgs $argv
	new HMAgent $self $logfd

	$self user_hook
}


HMApplication private init_args o {
	$o register_option -u userhookFile

	$o register_boolean_option -lb loadBalance
	$o register_boolean_option -fork doFork
	$o register_boolean_option -glunix glunix
	$o register_boolean_option -noload noLoad
	$o register_boolean_option -nolog noLog

	$o register_option -target targetNum
	$o register_option -int checkInterval
	$o register_option -path execPath
	$o register_option -log logFile
	$o register_option -rcmd execCmd
	$o register_option -conffile megaConfFile
	$o register_option -megactrl megaCtrl
	$o register_option -scriptdir scriptDir
}


HMApplication private init_resources o {

	$o add_default megaCtrl 224.4.5.24/50000/31
	$o add_default megaCtrlBW 20000
	# 60 seconds max startup wait for first message
	$o add_default megaStartupWait 60
	$o add_default scriptDir /var/tmp

	$o add_default unicastOutput 1

	$o add_default lambda 5.0
	$o add_default maxWait 15000
	$o add_default maxPending 5

	$o add_default minPort 10000
	$o add_default maxPorts 10000

	$o add_default checkFactor 2

	$o add_default checkInterval 20000
	$o add_default deathInterval 10000
	$o add_default minDeathWait 60000

	$o add_default highLoad 1.00
	$o add_default lowLoad 0.05
	$o add_default noLoad yes
	$o add_default execPath [pwd]
	$o add_default execCmd "ssh -n"
	$o add_default minHmNum 1
	$o add_default loadSamples 2
	$o add_default logFile /tmp/hmlog
	$o add_default megaConfFile mega.conf
	$o add_default link no

	$o add_default allow_distrib no
	$o add_default glob_chan 224.4.5.28/55000/15
}
