# ui-edit.tcl --
#
#       Creates the UI for editing a session announcement.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import TextEntry SDPMediaEditWindow SDPTimeEditWindow mashutils

#FIXME
import SDPParser

# This class constructs and manages a toplevel window that can be
# used to edit the contents of a session announcement described
# with SDP.  It is used by nsdr for editing existing announcements
# as well as for creating new announcement.
Class SDPEditWindow

# Creates a new window for editing the message in the SDPMessage
# object <i>msg</i>.  If <i>msg</i> is an empty string, the window
# is used to construct a new session announcement and being
# announcing it.  <i>src</i> is a ProgramSource object that is does
# the actual advertisement.
SDPEditWindow public init {msg src args} {
    eval $self next $args

    $self instvar win_ msg_ src_
    set win_ .new$self
    set msg_ $msg
    set src_ $src

    $self buildwin $win_
}

# Destroys this window.  Invoked when the Cancel button is pressed.
SDPEditWindow public destroy {} {
    $self instvar mediawin_ timewin_ win_
    $self instvar title_ url_ email_ phone_

    foreach o "$mediawin_ $timewin_ $title_ $url_ $email_ $phone_" {
	delete $o
    }
    destroy $win_
}

# Constructs the user interface elements for a new session announcement
# editing window.
SDPEditWindow private buildwin w {
    $self instvar msg_ src_ switchbutton_ mediawin_ timewin_ simple_

    toplevel $w
    wm title $w "nsdr: Edit Program"

    $self instvar notebook_
    set notebook_ [notebook $w.n -width 500 -height 200 \
		       -tabheight 27 -tabpadx 2 -bd 2 \
		       -font [$self get_option smallfont]]
    pack $notebook_ -side bottom -fill both -expand yes -padx 2 -pady 2

    set f $w.buttons
    frame $f
    set switchbutton_ [button $f.switch -command "$self toggle-advanced"]
    button $f.cancel -text "Cancel" -command "delete $self"
    pack $f.switch $f.cancel -side left
    if {$msg_ == ""} {
	button $f.create -text "Create" -command "$self create"
	pack $f.create -side left
    } else {
	button $f.change -text "Change" -command "$self change"
	pack $f.change -side left
    }
    pack $f -side bottom -before $notebook_

    $notebook_ addTab "Session Information"
    set f [frame $notebook_.info]
    $notebook_ addFrame $f 0 -expand yes -fill both -padx 2 -pady 2
    $self build-info $f

    $notebook_ addTab "Media Streams"
    set f [frame $notebook_.streams]
    $notebook_ addFrame $f 1 -expand yes -fill both -padx 2 -pady 2
    set mediawin_ [new SDPMediaEditWindow $f $msg_ $src_]

    $notebook_ addTab "Times"
    set f [frame $notebook_.times]
    $notebook_ addFrame $f 2 -expand yes -fill both -padx 2 -pady 2
    set timewin_ [new SDPTimeEditWindow $f $msg_]

    #FIXME encryption tab

    set simple_ [$self yesno simpleInterface]
    if {$simple_ == 0} {
	$switchbutton_ configure -text "Simple"
    } else {
	$switchbutton_ configure -text "Advanced"
    }

    $notebook_ tabPress 0
}

SDPEditWindow private toggle-advanced {} {
    $self instvar simple_ switchbutton_ mediawin_ timewin_
    if {$simple_ == 0} {
	set simple_ 1
	$switchbutton_ configure -text "Advanced"
	$mediawin_ build-simple
	$timewin_ build-simple
    } else {
	set simple_ 0
	$switchbutton_ configure -text "Simple"
	$mediawin_ build-advanced
	$timewin_ build-advanced
    }
}

# Constructs the portion of the SDP editing window that is used to
# edit session-wide information (e.g., title, description, etc...)
SDPEditWindow private build-info w {
    $self instvar msg_

    $self instvar title_ desc_ url_ email_ phone_
    if {$msg_ == ""} {
	set title "New Session"
	set desc ""
	set url ""

	set name [$self get_option nsdrName]
	set addr [$self get_option rtpEmail]
	if {$addr == "" } {
	    set addr [email_heuristic]
	}
	set email "$name <$addr>"
	set phone "$name [$self get_option nsdrPhone]"
    } else {
	set title [$msg_ field_value s]
	set desc [$msg_ field_value i]
	set url [$msg_ field_value u]
	set email [$msg_ field_value e]
	set phone [$msg_ field_value p]
    }

    # title
    set f $w.title
    frame $f
    label $f.l -text "Session Title:"
    pack $f.l -side left
    set title_ [new TextEntry "$self notnull" $f.entry $title]
    pack $f.entry -side left -fill x -expand yes
    pack $f -side top -fill x

    # session description
    set f $w.desc
    frame $f
    label $f.l -text "Session Description:"
    pack $f.l -side top
    set desc_ $f.t
    text $f.t -bg white -height 4 -font [$self get_option entryFont]
    $f.t insert 0.0 $desc
    pack $f.t -fill both
    pack $f -fill x

    # web page
    set f $w.web
    frame $f
    label $f.l -text "Web Page:" -anchor e
    pack $f.l -side left
    set url_ [new TextEntry "1" $f.e $url]
    pack $f.e -side left -fill x -expand yes
    button $f.b -text "Test" \
	-command "[Application instance] gourl \[$url_ entry-value\]"
    pack $f.b
    pack $f -fill x

    # e-mail address
    set f $w.mail
    frame $f
    label $f.l -text "E-Mail:" -anchor e
    pack $f.l -side left
    set email_ [new TextEntry "1" $f.e $email]
    pack $f.e -side left -fill x -expand yes
    pack $f -fill x

    # phone
    set f $w.phone
    frame $f
    label $f.l -text "Phone:" -anchor e
    pack $f.l -side left
    set phone_ [new TextEntry "1" $f.e $phone]
    pack $f.e -side left -fill x -expand yes
    pack $f -fill x
}

# Used as a callback for a TextEntry object, rejects an empty string.
SDPEditWindow private notnull s {
    if {$s == ""} {
	return 1
    }
    return 0
}

# Called when the Create button is pressed.  Creates a new SDPMessage
# object from what has been entered in the user interface and hands
# it to the ProgramSource to start advertising.
SDPEditWindow private create {} {
    $self instvar title_ timewin_ mediawin_ src_

    set base "v=0\n"
    append base "o=[$self generate_o]\n"
    append base "s=[string trim [$title_ entry-value]]\n"

    set time [$timewin_ buildmsg]

    # b=, z=, k= fields? (bandwidth, time zone adjustment, encryption key)
    set media [$mediawin_ buildmsgs]
    if {[llength $media] == 0} {
	puts "returning coz media is '$media'"
	return
    }

    # base announcement contains all info
    set text $base
    append text [$self build_details]
    append text $time
    append text "a=tool:Nsdr-[version]\n"
    #FIXME other global attrs?
    append text [lindex $media 0]

    set msgs [list $text]

    #further announcements don't contain details
    foreach m [lrange $media 1 end] {
	set text $base
	append text $time
	append text "a=Nsdr-[version]\n"
	append text $m
	lappend msgs $text
    }

    #FIXME why does SDPParser need to be instantiated?
    set parser [new SDPParser]
    set objs {}
    foreach msg $msgs {
	lappend objs [$parser parse $msg]
    }
    destroy $parser
    set prog [eval new Program $objs]

    #puts "\n\nbuilt messages:\n[join $msgs "\n"]"
    set announcer [[Application instance] set announcer_]
    $announcer announce $prog
}

#
SDPEditWindow private build_details {} {
    $self instvar desc_ url_ email_ phone_

    set text ""
    set desc [$desc_ get 0.0 end]
    regsub -all "\n|\r" $desc " " desc
    set desc [string trim $desc]
    if {$desc != ""} {
	append text "i=$desc\n"
    }

    set url [string trim [$url_ entry-value]]
    if {$url != ""} {
	append text "u=$url\n"
    }

    set email [string trim [$email_ entry-value]]
    if {$email != ""} {
	append text "e=$email\n"
    }

    set phone [string trim [$phone_ entry-value]]
    if {$phone != ""} {
	append text "p=$phone\n"
    }

    #FIXME
    after 1 "delete $self"

    return $text
}

#
SDPEditWindow private generate_o {} {
    set user [user_heuristic]
    # ntp time is recommended for these; is local time okay?
    set sid [clock seconds]
    set version [clock seconds]
    set addr "IN IP4 [localaddr]"
    return "$user $sid $version $addr"
}

# Called when the Change button is pressed.  Modifies the SDPMessage
# object that describes this session and then notifies the ProgramSource
# to modify what is being announced.
SDPEditWindow private change {} {
    #FIXME
    puts "change"
}

