# ui-program.tcl --
#
#       A single program being announced in the session directory user
#       interface.  Builds and manages the window that is created when the
#       program is selected in the main window of the user interface.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# A single program being announced in the session directory user
# interface.  Builds and manages the window that is created when
# the program is selected in the main window of the user interface.
Class ProgramWindow

# Creates a new ProgramWindow object to display the program in the
# Program <i>p</i>
ProgramWindow public init {p args} {
	eval $self next $args
	$self set prog_ $p

	global tcl_platform
	if {$tcl_platform(platform) == "unix" || $tcl_platform(platform) == "windows"} {
		$self set offset_ 2208988800
	} else {
		#FIXME
		self fatal "sorry, net yet ported to $tcl_platform(platform)"
	}
}

#
ProgramWindow public destroy {} {
	$self instvar win_
	catch {destroy $win_}
	$self next
}

#
ProgramWindow public title {} {
    return [[$self set prog_] field_value s]
}

# Maps the window with the information for this program if it is
# not currently mapped and unmaps it if it is.  Creates the window
# if necessary by calling <i>buildwin</i>.
ProgramWindow public toggle-window {} {
    $self instvar win_
    if ![info exists win_] {
	set win_ .prog$self
	$self buildwin $win_
	return
    }
    if [winfo ismapped $win_] {
	wm withdraw $win_
    } else {
	wm deiconify $win_
    }
}

# Sets the (private) instvar <i>apps_</i> to a list of user
# applications that might be started for this program.  See
# UserApplication::get_apps for details about how this is done.
ProgramWindow private set_apps {} {
    $self set apps_ [UserApplication get_apps [$self set prog_]]
}

# Creates a toplevel window <i>w</i> and instantiates the user
# interface elements for the description of this ProgramWindow.
ProgramWindow private buildwin w {
    $self instvar prog_ advanced_

    set advanced_ 0

    toplevel $w
    wm title $w "nsdr: Program Info"

    set mfont [$self get_option medfont]

    label $w.title -text [$self title] -font $mfont
    pack $w.title -side top -fill x -expand no

    frame $w.description
    set t $w.description.t
    set s $w.description.s
    text $t -state normal -relief ridge -bd 2 -height 5 \
	-font $mfont -wrap word -yscroll "$s set"
    if [$prog_ have_field i] {
	$t insert 0.0 [$prog_ field_value i]
    } else {
	$t insert 0.0 "No description provided."
    }
    $t configure -state disabled
    pack $t -fill both -expand yes
    scrollbar $w.description.s -command "$t yview"
    bind $w.description <Configure> "$self fix-scrollbar"
    pack $w.description -side top -fill both -expand yes

    if ![$self yesno simpleInterface] {
	$self build-advanced
	return
    }

    frame $w.bottom

    frame $w.bottom.f

    $self build-apps $w.bottom.f.apps 1
    pack $w.bottom.f.apps -side left -fill both -expand yes -padx 2 -pady 2

    $self build-times $w.bottom.f.times
    pack $w.bottom.f.times -side left -fill both -padx 2 -pady 2

    pack $w.bottom.f -side top -fill both -expand yes

    set f $w.bottom.buttons
    frame $f
    button $f.advanced -text "Advanced" -command "$self build-advanced"
    button $f.dismiss -text "Dismiss" -command "wm withdraw $w"
    pack $f.advanced $f.dismiss -side left
    if [$prog_ have_field u] {
	button $f.web -text "View Web Page" \
	    -command "[Application instance] gourl [$prog_ field_value u]"
	pack $f.web -side left -before $f.dismiss
    }
    pack $f -side top

    pack $w.bottom -fill both -expand yes

    wm protocol $w WM_DELETE_WINDOW "wm withdraw $w"
}

ProgramWindow private build-advanced {} {
    $self instvar win_ prog_ offset_ advanced_

    set advanced_ 1

    catch {destroy $win_.bottom}

    frame $win_.info -relief flat
    if [$prog_ have_field u] {
	set url [$prog_ field_value u]
	button $win_.info.webb \
	    -text "Web Page:" -command "[Application instance] gourl $url"
	grid $win_.info.webb -row 0 -column 0 -sticky ew
	label $win_.info.webl -text $url -relief sunken \
	    -bg white -bd 2 -anchor w
	grid $win_.info.webl -row 0 -column 1 -sticky ew -padx 5
    }

    if [$prog_ have_field e] {
	set email [$prog_ field_value e]
	label $win_.info.maill -text "E-Mail Contact:"
	grid $win_.info.maill -row 1 -column 0 -sticky ew
	label $win_.info.mailv -bg white -relief sunken -bd 2 \
	    -text $email -anchor w
	grid $win_.info.mailv -row 1 -column 1 -sticky ew -padx 5
    }

    if [$prog_ have_field p] {
	set phone [$prog_ field_value p]
	label $win_.info.phonel -text "Telephone Contact:"
	grid $win_.info.phonel -row 2 -column 0 -sticky ew
	label $win_.info.phonev -bg white -relief sunken -bd 2 \
	    -text $phone -anchor w
	grid $win_.info.phonev -row 2 -column 1 -sticky ew -padx 5
    }
    pack $win_.info -fill both -expand yes
    grid columnconfigure $win_.info 0 -minsize 100 -weight 0
    grid columnconfigure $win_.info 1 -weight 1

    frame $win_.bottom

    # applications box
    $self build-apps $win_.bottom.apps 0
    pack $win_.bottom.apps -side left -fill both -expand yes \
	-padx 2 -pady 2

    # times box
    $self build-times $win_.bottom.times
    pack $win_.bottom.times -side left -fill both -padx 2 -pady 2

    # media streams box
    frame $win_.bottom.media -relief flat
    label $win_.bottom.media.l -relief flat -text "Media Streams"
    pack $win_.bottom.media.l -side top -fill x -expand no
    set f $win_.bottom.media.f
    frame $f -relief sunken -bd 2

    set i 0
    #FIXME
    foreach media [[$prog_ base] set allmedia_] {
	label $f.type$i -relief flat -anchor e -text "[$media set mediatype_]:"
	grid $f.type$i -row $i -column 0 -sticky ew

	if [catch {set addr [$media set caddr_]}] {set addr "???" }
	set n [string first / $addr]
	if {$n != -1} { set addr [string range $addr 0 [expr $n-1]] }
	if [catch {set port [$media set port_]}] {set port "???" }
	label $f.info$i -relief sunken -bd 1 -bg white \
	    -anchor w -text "$addr/$port"
	grid $f.info$i -row $i -column 1 -sticky ew -padx 5

	incr i
    }
    pack $f -fill both -expand yes
    pack $win_.bottom.media -side left -fill both -padx 2 -pady 2

    pack $win_.bottom -fill both -expand yes

    frame $win_.buttons
    button $win_.buttons.source -text "SDP Source" \
	-command "$self toggle-srcwin"
    button $win_.buttons.quit -text "Dismiss" \
	-command "wm withdraw $win_"
    pack $win_.buttons.source $win_.buttons.quit -side left

    set o [split [$prog_ field_value o]]
    if {[lindex $o 0] == [user_heuristic] && [lindex $o 5] == [localaddr]} {
	#FIXME
	set src [[[Application instance] set ui_] current-source]
	button $win_.buttons.edit -text "Edit" \
	    -command "new SDPEditWindow $prog_ $src" -state disabled
	button $win_.buttons.delete -text "Delete Program" \
	    -command "$self stop-announcing"
	pack $win_.buttons.edit $win_.buttons.delete \
	    -side left -before $win_.buttons.quit
    }

    pack $win_.buttons

    wm protocol $win_ WM_DELETE_WINDOW "wm withdraw $win_"
}

#
ProgramWindow private build-times w {
    $self instvar prog_ offset_

    if ![winfo exists $w] {
	frame $w
    }
    label $w.l -relief flat -text "Times"
    pack $w.l -side top -fill x -expand no
    set f $w.f
    frame $f -relief sunken -bd 2

    #FIXME needs to be way improved
    set times [split [$prog_ field_value t]]
    if {[lindex $times 1] == 0} {
	set permanent 1
	set text "Session is always present."
    } else {
	set permanent 0
	set text "Session will be active"
	if [$prog_ have_field r] {
	    set l [split [$prog_ field_value r]]
	    set secs [lindex $l 0]
	    if {[string first d $secs] > 0} {
		set i [string first d $secs]
		set days [string range $secs 0 [expr $i-1]]
		if {$days == 1} {
		    append text " every day"
		} else {
		    append text " every $days days"
		}
	    } elseif {[string first h $secs] > 0} {
		set i [string first h $secs]
		set hrs [string range $secs 0 [expr $i-1]]
		if {$hrs == 1} {
		    append text " every hour"
		} else {
		    append text " every $hrs hours"
		}
	    } elseif {[string first m $secs] > 0} {
		set i [string first m $secs]
		set mins [string range $secs 0 [expr $i-1]]
		append text " every $mins minutes"
	    } elseif {$secs % 604800 == 0} {
		set weeks [expr $secs / 604800]
		if {$weeks == 1} {
		    append text " every week"
		} else {
		    append text " every $weeks weeks"
		}
	    } elseif {$secs % 86400 == 0} {
		set days [expr $secs / 86400]
		if {$days == 1} {
		    append text " every day"
		} else {
		    append text "every $days days"
		}
	    } elseif {$secs % 3600 == 0} {
		set hrs [expr $secs / 3600]
		if {$hrs == 1} {
		    append text " every hour"
		} else {
		    append text "every $hrs hours"
		}
	    } elseif {$secs % 60 == 0} {
		set mins [expr $secs / 60]
		append text " every $mins minutes"
	    }

	    #FIXME need to handle other parts of r= field (offsets)
	}
	set start [clock format [expr [lindex $times 0] - $offset_] \
		       -format "%H:%M %b %d"]
	set end [clock format [expr [lindex $times 1] - $offset_] \
		     -format "%H:%M %b %d"]
	set text "$text from $start to $end"
    }
    message $f.m -relief flat -width 150 -text $text
    pack $f.m -side top

    pack $f -fill both -expand yes
}

#
ProgramWindow private build-apps {w simple} {
    if ![winfo exists $w] {
	frame $w -relief flat
    }
    label $w.l -relief flat -text "Applications"
    pack $w.l -side top -fill x -expand no
    set f $w.f
    frame $f -relief sunken -bd 2

    set i 0
    $self set_apps
    set all_apps {}
    foreach a [$self set apps_] {
	set name [lindex $a 0]
	set cmd [lindex $a 1]

	button $f.r$i -text "Run:" -command "$self run $f.e$i"
	grid $f.r$i -row $i -column 0 -sticky ew

	entry $f.e$i -bg white -width 20
	grid $f.e$i -row $i -column 1 -sticky ew

	if {$simple} {
	    $f.e$i insert 0 $name
	    $f.e$i configure -state disabled
	    lappend all_apps $cmd

	    $f.r$i configure -command "$self run \{$cmd\}"
	} else {
	    $f.e$i insert 0 $cmd
	    lappend all_apps $f.e$i

	    $f.r$i configure -command "$self run $f.e$i"
	}

	incr i
    }
    grid columnconfigure $f 1 -weight 1
    button $f.runall -text "Run All Applications" \
	-command [concat $self run $all_apps]
    grid $f.runall -row $i -column 0 -columnspan 2
    pack $f -fill both -expand yes
}

# Decides whether a scrollbar is needed for the text widget containing
# the program description and maps it if necessary (or unmaps it if it
# is not needed).
ProgramWindow private fix-scrollbar {} {
    $self instvar win_
    set t "$win_.description.t"
    set s "$win_.description.s"

    set l [$t yview]
    if {[lindex $l 0] != 0 || [lindex $l 1] != 1} {
	pack $s -side right -before $t -fill y
    } else {
	pack forget $s
    }
}

#
ProgramWindow public updateprog {p} {
    $self instvar win_ prog_ advanced_

    if ![info exists win_] {
	# window hasn't been built yet -- nothing to update
	return
    }

    # update title
    $win_.title configure -text [$self title]

    # update description
    set t $win_.description.t
    $t configure -state normal
    $t delete 0.0 end
    if [$prog_ have_field i] {
	$t insert 0.0 [$prog_ field_value i]
    } else {
	$t insert 0.0 "No description provided."
    }
    $t configure -state disabled

    if $advanced_ {
	# update url
	if [$prog_ have_field u] {
	    set url [$prog_ field_value u]
	    if ![winfo exists $win_.info.webb] {
		#FIXME duplicated in buildwin
		button $win_.info.webb -text "Web page:"
		grid $win_.info.webb -row 0 -column 0 -sticky ew
		label $win_.info.webl -relief sunken -bg white -bd 2 -anchor w
		grid $win_.info.webl -row 0 -column 1 -sticky ew -padx 5
	    }
	    $win_.info.webb configure \
		-command "[Application instance] gourl $url"
	    $win_.info.webl configure -text $url
	}

	# update email
	if [$prog_ have_field e] {
	    if ![winfo exists $win_.info.maill] {
		#FIXME duplicated in buildwin
		label $win_.info.maill -text "E-Mail Contact:"
		grid $win_.info.maill -row 1 -column 0 -sticky ew
		label  $win_.info.mailv -bg white -relief sunken -bd 2 \
		    -anchor
		grid $win_.info.mailv -row 1 -column 1 -sticky ew -padx 5
	    }
	    $win_.info.mailv configure -text [$prog_ field_value e]
	}

	# update phone
	if [$prog_ have_field p] {
	    if ![winfo exists $win_.info.phonel] {
		#FIXME duplicated in buildwin
		label $win_.info.phonel -text "Telephone Contact:"
		grid $win_.info.phonel -row 2 -column 0 -sticky ew
		label  $win_.info.phonev -bg white -relief sunken -bd 2 \
		    -anchor
		grid $win_.info.phonev -row 2 -column 1 -sticky ew -padx 5
	    }
	    $win_.info.phonev configure -text [$prog_ field_value p]
	}

	# update media streams
	set media [[$prog_ base] set allmedia_]
	set f $win_.bottom.media.f
	set i 0
	foreach m $media {
	    if ![winfo exists $f.type$i] {
		label $f.type$i -relief flat -anchor e
		label $f.info$i -relief sunken -bd 1 -bg white -anchor w
		grid $f.type$i -row $i -column 0 -sticky ew
		grid $f.info$i -row $i -column 1 -sticky ew -padx 5
	    }
	    if [catch {set addr [$m set caddr_]}] { set addr "???" }
	    set n [string first / $addr]
	    if {$n != -1} { set addr [string range $addr 0 [expr $n-1]] }
	    if [catch {set port [$m set port_]}] { set port "???" }

	    $f.type$i config -text "[$m set mediatype_]:"
	    $f.info$i config -text "$addr/$port"

	    incr i
	}
	set old [llength [winfo children $f]]
	set new [llength $media]
	for {set i $old} {$i < $new} {incr i} {
	    destroy $f.type$i $f.info$i
	}
    }


    if $advanced_ {
	set appwin $win_.bottom.apps
	set timewin $win_.bottom.times
    } else {
	set appwin $win_.bottom.f.apps
	set timewin $win_.bottom.f.times
    }

    # update apps window
    foreach child [winfo children $appwin] { destroy $child }
    $self build-apps $appwin $advanced_

    # update times
    foreach child [winfo children $timewin] { destroy $child }
    $self build-times $timewin
}

# Maps the window with the SDP source for this program if it is
# not currently mapped and unmaps it if it is.  Creates the window
# if necessary by calling <i>build-srcwin</i>.
ProgramWindow private toggle-srcwin {} {
    $self instvar sw_ win_
    if ![info exists sw_] {
	set sw_ "$win_.src"
	$self build-srcwin $sw_
	return
    }
    if [winfo ismapped $sw_] {
	wm withdraw $sw_
    } else {
	wm deiconify $sw_
    }
}

# Creates a toplevel window <i>w</i> and displays the SDP source
# for this program within.
ProgramWindow private build-srcwin w {
    toplevel $w
    wm title $w "nsdr: SDP Source"

    #FIXME
    $self instvar prog_
    set msg [$prog_ base]

    set t $w.t
    set sx $w.sx
    set sy $w.sy

    text $t -font [$self get_option smallfont] -wrap none \
	-xscroll "$sx set" -yscroll "$sy set"
    $t insert 0.0 [$msg set msgtext_]
    grid $t -row 0 -column 0 -sticky nsew

    scrollbar $sx -orient horizontal -command "$t xview" -width 10
    grid $sx -row 1 -column 0 -sticky ew
    scrollbar $sy -orient vertical -command "$t yview" -width 10
    grid $sy -row 0 -column 1 -sticky ns

    button $w.b -text "Dismiss" -command "wm withdraw $w"
    grid $w.b -row 2 -column 0 -columnspan 2

    wm protocol $w WM_DELETE_WINDOW "wm withdraw $w"
}

# Given an SDPTime object <i>t</i> and an NTP time <i>now</i>,
# finds the first time after <i>now</i> when this program will
# be active.  Returns that time or an empty string if there is
# no such time (i.e., if all active times have passed or if the
# session is unbounded)
#
# FIXME maybe this should be moved to SDPTime
ProgramWindow private nexttime {t now} {
    $self instvar offset_

    set first [$t set starttime_]
    if {$first == 0} { return "" }

    # if start time hasn't come yet, it is the next time
    # return times in local time
    if {$now < $first} {
	return [format %u [expr $first - $offset_]]
    }

    # at this point, starttime has passed.  if there are no
    # repeats, this time field is hopeless
    if ![$t have_field "r"] { return "" }

    # try all repeats
    set end [$t set endtime_]
    set offs [$t set offlist_]
    set int [$t set repeat_interval_]

    while {$first < $end} {
	foreach o $offs {
	    set time [format %u [expr $first + $o]]
	    if {$now < $time} {
		# found one
		return [format %u [expr $time - $offset_]]
	    }
	}
	set first [format %u [expr $first + $int]]
    }

    # nothing worked...
    return ""
}

# Calculates how long (in milliseconds) to wait before posting an
# alarm or starting an application for this program.  Uses nexttime
# to find the next time, subtracts the lead time from the
# <i>alarmLead</i> resource, and converts to milliseconds.
ProgramWindow private getwait {} {
    $self instvar prog_ offset_

    set now [clock seconds]
    set ntpnow [format %u [expr $now + $offset_]]
    foreach time [[$prog_ base] set alltimedes_] {
	set t [$self nexttime $time $ntpnow]
	if {$t == ""} { continue }
	if ![info exists mintime] {
	    set mintime $t
	} elseif {$t < $mintime} {
	    set mintime t
	}
    }
    if ![info exists mintime] { return "" }
    set lead [$self get_option alarmLead]
    set wait [expr 1000*($mintime - $now - $lead)]

#    set sec [expr $wait/1000]
#    puts "waiting $sec seconds (until [clock format [expr $now+$sec]])"

    return $wait
}

# Launch an application for this program.
ProgramWindow public run {args} {
    foreach cmd $args {
        #FIXME
        if [winfo exists $cmd] {
            set cmd [$cmd get]
        }

        # Escape special Tcl characters so exec works ok.

        set escapedCmd $cmd
        regsub -all {\\} $escapedCmd {\\\\} escapedCmd
        regsub -all {"} $escapedCmd {\"} escapedCmd
        set i [expr [string first {\"} $escapedCmd] - 1]
        set j [expr $i + 2]
        set escapedCmd \
            [string range $escapedCmd 0 $i][string range $escapedCmd $j end]
        set i [expr [string last {\"} $escapedCmd] - 1]
        set j [expr $i + 2]
        set escapedCmd \
            [string range $escapedCmd 0 $i][string range $escapedCmd $j end]
        regsub -all {\[} $escapedCmd {\[} escapedCmd
        regsub -all {\]} $escapedCmd {\]} escapedCmd
        regsub -all {\$} $escapedCmd {\$} escapedCmd

        # FIXME
        # The <<null is a hack to get nsdr working on Windows.  Without it, the
        # exec fails in TclpCreateProcess when it tries to duplicate the handle
        # to stdin.  Tcl 8.4a2 does not appear to fix this problem because the
        # stdin handle passed to us is a pipe.  --LL
        #
        if [catch {eval exec $escapedCmd <<null &} m] {
            $self warn "couldn't run \"$cmd\": $m"
        }
    }
}

#
ProgramWindow private stop-announcing {} {
    # FIXME
    set src [[[Application instance] set ui_] current-source]
    $src stop-announce [$self set prog_]

    #FIXME need to delete
}
