# camera-sonyEVID30.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/serial/camera-sonyEVID30.tcl,v 1.12 2002/02/03 04:29:47 lim Exp $


import SerialChannel

#
# The extension of SerialChannel specialized to the Sony EVID30
# Remote-Controllable Camera
#
Class SerialChannel/Camera-Sony-EVID30 -superclass SerialChannel

#
# init and open cam
#
SerialChannel/Camera-Sony-EVID30 public init { args } {
	eval [list $self] next $args

	$self instvar panSpeedMax_ tiltSpeedMax_ zoomSpeedMax_ \
			panPosition_ zoomPosition_ \
			presets_

	set panSpeedMax_ 0x18
	set tiltSpeedMax_ 0x14
	set zoomSpeedMax_ 0x0f

	$self device "/dev/cuac02"
	$self stopBits 1
	$self parity none
	$self open

	$self set_pan_speed 10
	$self set_tilt_speed 10
	$self set_zoom_speed 35
}


# set preset entry number <num>
# <br>
# internal format of presets is "<8 pan/tilt-coords> <4 zoom coords>",
# making it a list of of 12 integers.
#
SerialChannel/Camera-Sony-EVID30 public set_preset { num } {
	$self instvar presets_
	set presets_($num) [$self get_coords]
	if {[llength $presets_($num)] != 8} {
		puts "Error: presets not working..."
		unset presets_($num)
		return
	}
	append presets_($num) " "
	append presets_($num) [$self get_zoom]
	puts "set_preset($num) = $presets_($num)"
}

#
# export preset for saving in external agents.
# <br>
# The *external* format converts
# from the internal space-separated to dot-separated to allow
# it to be passed as a single list item
#
SerialChannel/Camera-Sony-EVID30 public get_preset { num } {
	$self instvar presets_
	if {[array names presets_ $num] == ""} {
		return ""
	} else {
		if {[llength $presets_($num)] != 12} {
			puts "Error: preset has only [llength $presets_($num)] members"
		}
		regsub -all " " $presets_($num) "." dottedResult
		#puts "get_preset($num) = $dottedResult"
		return $dottedResult
	}
}

#
# overloaded function takes either a integer preset number
# or dotted list (external preset format).  We infer the
# correct data type manually.
#
SerialChannel/Camera-Sony-EVID30 public goto_preset { numOrPreset } {
	$self instvar presets_

	if {[string first "." $numOrPreset] == -1} {
		# assume internal preset number
		if {[array names presets_ $numOrPreset] == ""} {
			puts "no preset $numOrPreset set."
			return
		}
		if {[llength $presets_($numOrPreset)] != 12} {
			puts "Error: preset has only \
					[llength $presets_($numOrPreset)] members"
		}
		set g [lrange $presets_($numOrPreset) 0 7]
		set z [lrange $presets_($numOrPreset) 8 11]
	} else {
		# assume actual preset sent
		if {[llength [split $numOrPreset "."]] != 12} {
			puts "Error: sent preset has wrong \# of members: $numOrPreset"
		}
		set g [lrange [split $numOrPreset "."] 0 7]
		set z [lrange [split $numOrPreset "."] 8 11]
	}
	eval $self goto_coords $g
	eval $self goto_zoom $z
}

#
# returns current pan/zoom/tilt settings in external preset format
#
SerialChannel/Camera-Sony-EVID30 public get_settings {} {
	set reply [$self get_coords]
	append reply " "
	append reply [$self get_zoom]
	if {[llength $reply] != 12} {
		puts "get_settings not working..."
		return ""
	}
	regsub -all " " $reply "." dotRep
	puts $dotRep
	return $dotRep
}


# set pan speed ; arg is percent from 1..100
#
SerialChannel/Camera-Sony-EVID30 public set_pan_speed {percent} {
	$self instvar panSpeed_ panSpeedMax_ panPerc_

	if {$percent < 1 || $percent > 100} {
		return "set speed by percentage from 1..100"
	}
	set panPerc_ $percent
	set panSpeed_ [expr (($percent * $panSpeedMax_) / 100) + 1]
	if {$panSpeed_ > $panSpeedMax_} {
		incr panSpeed_ -1
	}
	set panSpeed_ [format "0x%.2x" $panSpeed_]
}


# set tilt speed; arg is percentage from 1..100
#
SerialChannel/Camera-Sony-EVID30 public set_tilt_speed {percent} {
	$self instvar tiltSpeed_ tiltSpeedMax_ tiltPerc_

	if {$percent < 1 || $percent > 100} {
		return "set speed by percentage from 1..100"
	}
	set tiltPerc_ $percent
	set tiltSpeed_ [expr (($percent * $tiltSpeedMax_) / 100) + 1]
	if {$tiltSpeed_ > $tiltSpeedMax_} {
		incr tiltSpeed_ -1
	}
	set tiltSpeed_ [format "0x%.2x" $tiltSpeed_]
}

# set zoomSpeed; arg is percentage from 1..100
#
SerialChannel/Camera-Sony-EVID30 public set_zoom_speed {percent} {
	$self instvar zoomSpeed_ zoomSpeedMax_ zoomPerc_

	if {$percent < 1 || $percent > 100} {
		return "set speed by percentage from 1..100"
	}
	set zoomPerc_ $percent
	set zoomSpeed_ [expr (($percent * $zoomSpeedMax_) / 100) + 1]
	if {$zoomSpeed_ > $zoomSpeedMax_} {
		incr zoomSpeed_ -1
	}
	set zoomSpeed_ [format "0x%.2x" $zoomSpeed_]
}


#
# ------------------  These are all from the VISCA command set
# (many have been left out .... FIXME)
#
#  -- input:
#
#  movement
#
SerialChannel/Camera-Sony-EVID30 public reset {} {
	$self send "0x81 0x01 0x06 0x05 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_left {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x01 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_right {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x02 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_stop {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x03 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_up {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x03 0x01 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_down {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x03 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_upleft {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x01 0x01 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_upright {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x02 0x01 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_downleft {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x01 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_downright {} {
	$self instvar panSpeed_ tiltSpeed_
	$self send "0x81 0x01 0x06 0x01 $panSpeed_ $tiltSpeed_ 0x02 0x02 0xff"
}

#
# go to absolute position setting (pan + tile ... not zoom)
# j k r s  are pan  ; t u v w  are tilt
# but the best way to get these numbers is from a getCoords method call
#
SerialChannel/Camera-Sony-EVID30 public goto_coords {j k r s t u v w} {
	$self instvar panSpeed_ tiltSpeed_
	set j [format "0x%.2x" $j]
	set k [format "0x%.2x" $k]
	set r [format "0x%.2x" $r]
	set s [format "0x%.2x" $s]
	set t [format "0x%.2x" $t]
	set u [format "0x%.2x" $u]
	set v [format "0x%.2x" $v]
	set w [format "0x%.2x" $w]
	$self send "0x81 0x01 0x06 0x02 $panSpeed_ $tiltSpeed_\
			$j $k $r $s $t $u $v $w 0xff"
}

SerialChannel/Camera-Sony-EVID30 public move_home {} {
	$self goto_coords 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00
}

#
# -- input:
#
#  zooming
#

SerialChannel/Camera-Sony-EVID30 public zoom_in {} {
	$self instvar zoomSpeed_
	set val [format "0x%.2X" [expr $zoomSpeed_ + 0x20]]
	$self send "0x81 0x01 0x04 0x07 $val 0xff"
}

SerialChannel/Camera-Sony-EVID30 public zoom_out {} {
	$self instvar zoomSpeed_
	set val [format "0x%.2X" [expr $zoomSpeed_ + 0x30]]
	$self send "0x81 0x01 0x04 0x07 $val 0xff"
}

SerialChannel/Camera-Sony-EVID30 public zoom_stop {} {
	$self send "0x81 0x01 0x04 0x07 0x00 0xff"
}

#
# goto absolute zoom coordinates; a b c d inputs are best
# derived from calls to get_zoom method
#
SerialChannel/Camera-Sony-EVID30 public goto_zoom {a b c d} {
	$self instvar zoomSpeed_
	set a [format "0x%.2x" $a]
	set b [format "0x%.2x" $b]
	set c [format "0x%.2x" $c]
	set d [format "0x%.2x" $d]
	$self send "0x81 0x01 0x04 0x47 $a $b $c $d 0xff"
}

#
# -- input:
#
#  settings
#

SerialChannel/Camera-Sony-EVID30 public power_on {} {
	$self send "0x81 0x01 0x04 0x00 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public power_off {} {
	$self send "0x81 0x01 0x04 0x00 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public focus_manual {} {
	$self send "0x81 0x01 0x04 0x38 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public focus_auto {} {
	$self send "0x81 0x01 0x04 0x38 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public focus_near {} {
	$self send "0x81 0x01 0x04 0x08 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public focus_far {} {
	$self send "0x81 0x01 0x04 0x08 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public focus_stop {} {
	$self send "0x81 0x01 0x04 0x08 0x00 0xff"
}

SerialChannel/Camera-Sony-EVID30 public goto_focus {a b c d} {
	set a [format "0x%.2x" $a]
	set b [format "0x%.2x" $b]
	set c [format "0x%.2x" $c]
	set d [format "0x%.2x" $d]
	$self send "0x81 0x01 0x04 0x48 $a $b $c $d 0xff"
}

SerialChannel/Camera-Sony-EVID30 public whiteBalance_auto {} {
	$self send "0x81 0x01 0x04 0x35 0x00 0xff"
}

SerialChannel/Camera-Sony-EVID30 public whiteBalance_indoor {} {
	$self send "0x81 0x01 0x04 0x35 0x01 0xff"
}

SerialChannel/Camera-Sony-EVID30 public whiteBalance_outdoor {} {
	$self send "0x81 0x01 0x04 0x35 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public whiteBalance_onePushMode {} {
	$self send "0x81 0x01 0x04 0x35 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public whiteBalance_onePushTrigger {} {
	$self send "0x81 0x01 0x04 0x10 0x05 0xff"
}

SerialChannel/Camera-Sony-EVID30 public brightness_reset {} {
	$self send "0x81 0x01 0x04 0x0d 0x00 0xff"
}

SerialChannel/Camera-Sony-EVID30 public brightness_up {} {
	$self send "0x81 0x01 0x04 0x0d 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public brightness_down {} {
	$self send "0x81 0x01 0x04 0x0d 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public shutter_reset {} {
	$self send "0x81 0x01 0x04 0x0a 0x00 0xff"
}

SerialChannel/Camera-Sony-EVID30 public shutter_up {} {
	$self send "0x81 0x01 0x04 0x0a 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public shutter_down {} {
	$self send "0x81 0x01 0x04 0x0a 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public iris_reset {} {
	$self send "0x81 0x01 0x04 0x0b 0x00 0xff"
}

SerialChannel/Camera-Sony-EVID30 public iris_up {} {
	$self send "0x81 0x01 0x04 0x0b 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public iris_down {} {
	$self send "0x81 0x01 0x04 0x0b 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public gain_reset {} {
	$self send "0x81 0x01 0x04 0x0c 0x00 0xff"
}

SerialChannel/Camera-Sony-EVID30 public gain_up {} {
	$self send "0x81 0x01 0x04 0x0c 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public gain_down {} {
	$self send "0x81 0x01 0x04 0x0c 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public backlight_on {} {
	$self send "0x81 0x01 0x04 0x33 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public backlight_off {} {
	$self send "0x81 0x01 0x04 0x33 0x03 0xff"
}

SerialChannel/Camera-Sony-EVID30 public autoZoom_on {} {
	$self send "0x81 0x01 0x07 0x03 0x02 0xff"
}

SerialChannel/Camera-Sony-EVID30 public autoZoom_off {} {
	$self send "0x81 0x01 0x07 0x02 0x03 0xff"
}

#
# ----- VISCA Inquiry commmands
#

SerialChannel/Camera-Sony-EVID30 public get_coords {} {
	$self clearReply
	$self send "0x81 0x09 0x06 0x12 0xff"
	after 200
	set reply [$self getReply]
	if {$reply == ""} {return "Error: no reply"}
	return [lrange $reply 2 9]
}

SerialChannel/Camera-Sony-EVID30 public get_zoom {} {
	$self clearReply
	$self send "0x81 0x09 0x04 0x47 0xff"
	after 200
	set reply [$self getReply]
	if {$reply == ""} {return "Error: no reply"}
	return [lrange $reply 2 5]
}

SerialChannel/Camera-Sony-EVID30 public get_power {} {
	$self clearReply
	$self send "0x81 0x09 0x04 0x00 0xff"
	after 200
	set reply [$self getReply]
	if {$reply == ""} {return "Error: no reply"}
	switch [lindex $reply 2] {
		2 {return "On"}
		3 {return "Off"}
		default {return "Error! -- bad getPower Reply: $reply"}
	}
}


#
# ----- These are for interfaces that can't time the move/stop
#       combinations
#

SerialChannel/Camera-Sony-EVID30 public bump_left {{wait 100}} {
    $self move_left
    after $wait
    $self move_stop
}

SerialChannel/Camera-Sony-EVID30 public bump_right {{wait 100}} {
    $self move_right
    after $wait
    $self move_stop
}

SerialChannel/Camera-Sony-EVID30 public bump_up {{wait 100}} {
    $self move_up
    after $wait
    $self move_stop
}

SerialChannel/Camera-Sony-EVID30 public bump_down {{wait 100}} {
    $self move_down
    after $wait
    $self move_stop
}

SerialChannel/Camera-Sony-EVID30 public bump_in {{wait 100}} {
    $self zoom_in
    after $wait
    $self zoom_stop
}

SerialChannel/Camera-Sony-EVID30 public bump_out {{wait 100}} {
    $self zoom_out
    after $wait
    $self zoom_stop
}

