;;;-*- Mode: LISP; Package: CCL -*-


(in-package "CCL")

(eval-when (:compile-toplevel :execute)
  (setq *readtable* *objc-readtable*))

(declaim
  (special *open-editor-documents*)
  (type list *open-editor-documents*))

(defstruct cocoa-editor-info
  (document nil)			; the NSDocument
  (controller nil)			; the NSWindowController (maybe).
  (listener nil)			; True (a lisp process) if a listener
  (modeline-plist nil)			; info from attribute line
)

(defparameter *default-font-name* "Courier")
(defparameter *default-font-size* 12.0)

;;; Try to find the specified font.  If it doesn't exist (or isn't
;;; fixed-pitch), try to find a fixed-pitch font of the indicated size.
(defun default-font (&key (name *default-font-name*)
			  (size *default-font-size*))
  (setq size (float size 0.0f0))
  (with-cstrs ((name name))
    (let* ((fontname [(@class "NSString") "stringWithCString:" (* :char) name])
	   (font [(@class "NSFont") "fontWithName:size:"
		   :id fontname :single-float size]))
      (if (or (%null-ptr-p font)
	      (and 
	       (eql #$NO [font "isFixedPitch" :<BOOL>])
	       (eql #$NO [font "_isFakeFixedPitch" :<BOOL>])))
	(setq font [(@class "NSFont") "userFixedPitchFontOfSize:"
		    :single-float size]))
      font)))

(defun get-size-for-textview (font nrows ncols)
  (let* ((sf [font "screenFont"]))
    (values (fceiling (* nrows
			 (+ [sf "defaultLineHeightForFont" :single-float]
			    (if (eql
				 #$NO
				 [font "_wantsToHaveLeadingBelow" :<BOOL>])
			      1.0f0
			      0.0f0))))
	    (fceiling (* ncols
			 [sf "widthOfString:" :id #@" " :single-float])))))


(defun size-textview-containers (tv height width)
  (let* ((scrollview [[tv "superview"] "superview"])
	 (window [scrollview "window"]))
    (rlet ((tv-size :<NSS>ize :height height :width width)
	   (sv-size :<NSS>ize))
      [(@class "NSScrollView")
       "frameSizeForContentSize:hasHorizontalScroller:hasVerticalScroller:borderType:"
       :<NSS>ize tv-size
       :<BOOL> [scrollview "hasHorizontalScroller" :<BOOL>]
       :<BOOL> [scrollview "hasVerticalScroller" :<BOOL>]
       :<NSB>order<T>ype [scrollview "borderType" :<NSB>order<T>ype]
       (:-> sv-size)]
      (incf (pref sv-size :<NSS>ize.height) 20)
      (incf (pref sv-size :<NSS>ize.width) 12)
      [window "setContentSize:" :<NSS>ize sv-size])))

      
(defun info-from-document (doc)
  (find doc *open-editor-documents* :key #'cocoa-editor-info-document))

(defun info-from-controller (controller)
  (find controller *open-editor-documents* :key #'cocoa-editor-info-controller))


(defparameter *tab-width* 8)

;;; Create a paragraph style, mostly so that we can set tabs reasonably.
(defun create-paragraph-style (font line-break-mode)
  (let* ((p [[(@class "NSMutableParagraphStyle") "alloc"] "init"])
	 (charwidth [[font "screenFont"]
		     "widthOfString:" :id #@" " :single-float]))
    [p "setLineBreakMode:" :<NSL>ine<B>reak<M>ode
       (ecase line-break-mode
	 (:char #$NSLineBreakByCharWrapping)
	 (:word #$NSLineBreakByWordWrapping)
	 (nil #$NSLineBreakByClipping))]
    ;; Clear existing tab stops.
    [p "setTabStops:" :id [(@class "NSArray") "array"]]
    (do* ((i 1 (1+ i)))
	 ((= i 100) p)
      (let* ((tabstop [[(@class "NSTextTab") "alloc"]
		       "initWithType:location:"
		       :unsigned #$NSLeftTabStopType
		       :single-float (* (* i *tab-width*)
					charwidth)]))
	[p "addTabStop:" :id tabstop]
	[tabstop "release"]))))
    
(defun create-text-attributes (&key (font (default-font))
				    (line-break-mode :char)
				    (color nil))
  (let* ((dict [[[(@class "NSMutableDictionary") "alloc"]
		 "initWithCapacity:" :unsigned (if color 3 2)]
		"retain"]))
    [dict "setObject:forKey:"
	  :id (create-paragraph-style font line-break-mode)
	  :id #@"NSParagraphStyle"]
    [dict "setObject:forKey:" :id font :id  #@"NSFont"]
    (when color
      [dict "setObject:forKey:" :id color :id #@"NSColor"])
    dict))
				    
  
(def-objc-class "lispeditorwindowcontroller" "NSWindowController"
  textview				;The (primary) textview
  packagename				;Textfield for package name display
  echoarea				;Textfield for message display.
  ((history-count "histcount") :int)	;current history count (for prev/next)
  ((prev-history-count "prevhist") :int) ;value of history-count before last cmd
  )

(define-objc-method ("displayEchoArea:" "lispeditorwindowcontroller")
    (:id contents :void)
  [echoarea "setStringValue:" :address contents])

(define-objc-method ("clearEchoArea" "lispeditorwindowcontroller")
    (:void)
  [echoarea "setStringValue:" :id #@""])

(define-objc-method ("displayPackageName:" "lispeditorwindowcontroller")
     (:id name :void)
  [packagename "setStringValue:" :id name])

(defun shortest-package-name (package)
  (let* ((shortest (package-name package))
	 (shortest-len (length shortest)))
    (declare (fixnum shortest-len))
    (dolist (nick (package-nicknames package) shortest)
      (let* ((nicklen (length nick)))
	(declare (fixnum nicklen))
	(if (< nicklen shortest-len)
	  (setq shortest-len nicklen shortest nick))))))
	     
(define-objc-method ("updatePackageName" "lispeditorwindowcontroller")
     (:void)
  (let* ((info (info-from-controller self))
	 (package (and info (getf (cocoa-editor-info-modeline-plist info)
				  :package)))
	 (name (if (and package (typep package 'package))
		 (shortest-package-name package)
		 "#<PACKAGE unset>")))
    (with-cstrs ((name name))
      (let* ((string [(@class "NSString")
		      "stringWithCString:" (* :char) name]))
	[self "displayPackageName:" :id string]))))
    
;;; The lispeditorwindowcontroller is the textview's "delegate": it
;;; gets consulted before certain actions are performed, and can
;;; perform actions on behalf of the textview.

;;; Action methods implemented by the controller (in its role as the
;;; textview's delegate).

;;; If the first line of the buffer contains text between a pair of
;;; "-*-"s, treat the line as an attribute line. 
(define-objc-method ("_rangeForModeline" "lispeditorwindowcontroller")
    (:id tv (* :<NSR>ange) r :void)
  (let* ((textstring [tv "string"]))
    (rlet ((linerange :<NSR>ange)
	   (startrange :<NSR>ange :location 0 :length 0))
      [textstring "lineRangeForRange:"
		  :<NSR>ange startrange (:-> linerange)]
      (when (> (pref linerange :<NSR>ange.length) 0)
	(decf (pref linerange :<NSR>ange.length)))
      (rlet ((matchrange1 :<NSR>ange)
	     (matchrange2 :<NSR>ange))
	[textstring "rangeOfString:options:range:"
		 :address #@"-*-"
		 :unsigned-fullword 0
		 :<NSR>ange linerange
		 (:-> matchrange1)]
	(if (and (> (pref matchrange1 :<NSR>ange.length) 0)
		 (progn
		   (incf (pref matchrange1 :<NSR>ange.location)
			 (pref matchrange1 :<NSR>ange.length))
		   (setf (pref matchrange1 :<NSR>ange.length)
			 (- (pref linerange :<NSR>ange.length)
			    (pref matchrange1 :<NSR>ange.location)))
		   [textstring
		    "rangeOfString:options:range:"
		    :address #@"-*-"
		    :unsigned-fullword 0
		    :<NSR>ange matchrange1
		    (:-> matchrange2)]
		 (> (pref matchrange1 :<NSR>ange.length) 0)))  
	  (setf (pref r :<NSR>ange.location)
		(pref matchrange1 :<NSR>ange.location)
		(pref r :<NSR>ange.length)
		(- (pref matchrange2 :<NSR>ange.location)
		   (pref r :<NSR>ange.location)))
	  (setf (pref r :<NSR>ange.location) 0
		(pref r :<NSR>ange.length) 0))))))

;;; Return a list whose elements are of the form:
;;;  (opt-name-keyword . (opt-value-start . opt-value-end))
;;;  for each option.  Options are separated colons semicolons;
;;;  option names are separated from option values by colons.
(defun extract-modeline-components (string)
  (let* ((start 0)
	 (end (length string))
	 (options ()))
    (if (find #\: string)
      (block parse-options
	(do* ((opt-start start (1+ semi))
	      semi
	      colon)
	     (nil)
	  (setq colon (position #\: string :start opt-start :end end))
	  (unless colon
	    (return nil))
	  (setq semi (or (position #\; string :start colon :end end) end))
	  (push
	   (cons
	    (intern
	     (nstring-upcase (string-trim '(#\space #\tab)
					  (subseq string opt-start colon)))
	     *keyword-package*)	    
	    (cons
	     (do* ((i (1+ colon) (1+ i)))
		  ((= i semi) (return-from parse-options nil))
	       (unless (whitespacep (schar string i))
		 (return i)))
	     (do* ((i semi j)
		   (j (1- i) (1- j)))
		  (())
	       (unless (whitespacep (schar string j))
		 (return i)))))
	   options)
	  (when (= semi end) (return options)))))))

(defun process-modeline-components (components info)
  (let* ((plist ()))
    (dolist (c components (setf (cocoa-editor-info-modeline-plist info) plist))
      (let* ((indicator (car c))
	     (value (cdr c)))
	(case indicator
	  (:package (let* ((spec (let* ((*package* *keyword-package*))
				   (ignore-errors (read-from-string value)))))
		      (when spec
			(let* ((pkg (ignore-errors (find-package
						    (if (atom spec)
						      spec
						      (car spec))))))
			  (if pkg
			    (setf (getf plist indicator) pkg))))))
	  (t (setf (getf plist indicator) value)))))))

(define-objc-method ("reparseModeline:" "lispeditorwindowcontroller") (:id tv)
  (unless (%null-ptr-p tv)
    (let* ((info (info-from-controller self)))
      (when info
	(let* ((textstring [tv "string"]))
	  (rlet ((modelinerange :<NSR>ange))
	    [self "_rangeForModeline" :address tv :address modelinerange]
	    (unless (zerop (pref modelinerange :<NSR>ange.length))
	      (let* ((string (lisp-string-from-nsstring
			      [textstring "substringWithRange:"
					  :<NSR>ange modelinerange]))
		     (components
		      (mapcar #'(lambda (x)
				  (destructuring-bind (name start . end) x
				    (cons name
					  (subseq string start end))))
			      (extract-modeline-components string))))
		(process-modeline-components components info)
		[self "updatePackageName"]))))))))
  

(define-objc-method ("addModeline:" "lispeditorwindowcontroller")
     (:id tv)
  (let* ((textstring [tv "string"]))
    (rlet ((modelinerange :<NSR>ange)
	   (startrange :<NSR>ange :location 0 :length 0)
	   (selrange :<NSR>ange))
      [self "_rangeForModeline" :address tv :address modelinerange]
      (when (= (pref modelinerange :<NSR>ange.length) 0)
	(let* ((template #@";;;-*- Mode: LISP; Package: %@ -*-
")
	       (info (info-from-document self))
	       (package (or (if info
			      (getf
			       :package
			       (cocoa-editor-info-modeline-plist info)))
			    (symbol-value-in-top-listener-process
			     '*package*)
			    *package*))
	       (package-name (package-name package))
	       (namelen (length package-name)))
	  (with-cstrs ((pname package-name))
	    (with-nsstr (nsstr pname namelen)
	      (let* ((proto [(@class "NSString")
			     "stringWithFormat:"
			     :id template
			     :id nsstr]))
		[tv "setSelectedRange:" :<NSR>ange startrange]
		[tv "insertText:" :id proto]
		(setf (pref modelinerange :<NSR>ange.location)
		      6
		      (pref modelinerange :<NSR>ange.length)
		      (- [proto "length" :unsigned-fullword] (+ 6 1 3))))))))
      (let* ((components (extract-modeline-components
			  (lisp-string-from-nsstring
			   [textstring "substringWithRange:"
				       :<NSR>ange modelinerange])))
	     (package-component (assoc :PACKAGE components)))
	(if package-component
	  (destructuring-bind (start . end) (cdr package-component)
	    (setf (pref selrange :<NSR>ange.location)
		  (+ start (pref modelinerange :<NSR>ange.location))
		  (pref selrange :<NSR>ange.length)
		  (- end start)))
	  (setf (pref selrange :<NSR>ange.location)
		(pref modelinerange :<NSR>ange.location)
		(pref selrange :<NSR>ange.length)
		0))
	[tv "setSelectedRange:" :<NSR>ange selrange]
	[tv "scrollRangeToVisible:" :<NSR>ange selrange]
	[tv "display"])))
  self)

;;; Interrupt/abort something.  When that means something ...
(define-objc-method ("interrupt:" "lispeditorwindowcontroller") (:id tv)
  (declare (ignore tv))
  self)


(define-objc-method ("evalDefun:" "lispeditorwindowcontroller") (:id tv)
  (rlet ((defunrange :<NSR>ange)
	 (workrange :<NSR>ange))
    (let* ((textbuf [tv "string"])
	   (textlen [textbuf "length" :unsigned-fullword]))
	[tv "selectedRange" (:-> defunrange)]
	(let* ((pointpos (pref defunrange :<NSR>ange.location)))
	  (if (> (pref defunrange :<NSR>ange.length) 0)
	    (progn
	      (setf (pref workrange :<NSR>ange.location)
		    (pref defunrange :<NSR>ange.location)
		    (pref workrange :<NSR>ange.length)
		    (pref defunrange :<NSR>ange.length))
	      (multiple-value-bind (ok non-wsp)
		  (balanced-expressions-in-range-forward workrange textbuf)
		(unless (and ok non-wsp)
		  (setf (pref defunrange :<NSR>ange.length) 0))))
	    (let* ((defun-start (previous-start-of-defun textbuf pointpos)))
	      (when defun-start
		(setf (pref workrange :<NSR>ange.location) defun-start
		      (pref workrange :<NSR>ange.length) (- textlen defun-start))
		(if (forward-over-list workrange textbuf)
		  (setf (pref defunrange :<NSR>ange.location)
			defun-start
			(pref defunrange :<NSR>ange.length)
			(- (1+ (pref workrange :<NSR>ange.location))
			   defun-start))
		  (setf (pref defunrange :<NSR>ange.length)
			0)))))
	  (if (and (> (pref defunrange :<NSR>ange.length) 0)
		   #|(> pointpos (+ (pref defunrange :<NSR>ange.location)
				  (pref defunrange :<NSR>ange.length)))|#)
	    (send-to-top-listener
	     (info-from-controller self)
	     [textbuf "substringWithRange:" :<NSR>ange defunrange])
	    (#_NSBeep))))))

#|
(def-objc-method ("beginningOfDefun:" "lispeditorwindowcontroller")
    (:id tv))
  (rlet ((selection :<NSR>ange))
    (let* ((textbuf [tv "string"])
	   (textlen [textbuf "length" :unsigned-fullword]))
      [tv "selectedRange" (:-> selection)]
      (let* ((pointpos (pref section :<NSR>ange.location))
	     (defun-start
    [
|#

;;; Also a delegate method
(define-objc-method ("textView:doCommandBySelector:"
		     "lispeditorwindowcontroller")
     (:id tv :<SEL> selector :<BOOL>)
  ;(#_NSLog #@"selector = %s, self = %@" :<SEL> selector :id self)
  (setq prev-history-count history-count
	history-count 0)
  (if (= [self "respondsToSelector:" :address selector :unsigned-byte] #$NO)
    #$NO
    (progn
      [self "performSelector:withObject:"
	    :address selector
	    :address tv]
      #$YES)))


;;; The lispeditordocument class.


(def-objc-class "lispeditordocument" "NSDocument"
  ((textview "textView") :id)
  filedata
  packagename
  echoarea)

(define-objc-method ("windowNibName" "lispeditordocument")  (:id)
  #@"lispeditor")

(define-objc-method ("makeWindowControllers" "lispeditordocument")
    (:void)
  (let* ((nibname [self "windowNibName"])
	 (controller [[(@class "lispeditorwindowcontroller") "alloc"]
		      "initWithWindowNibName:owner:"
		      :address nibname
		      :address self]))
    [self "addWindowController:" :address controller]
    [controller "release"]))


(define-objc-method ("dataRepresentationOfType:" "lispeditordocument")
    ((* :char) type :id)
  (declare (ignorable type))
  ;(#_NSLog #@"dataRepresentationOfType: %s" :address type)
  [[textview "string"]
   "dataUsingEncoding:allowLossyConversion:"
   :unsigned-fullword #$NSASCIIStringEncoding
   :unsigned-byte #$YES])

'(define-objc-method ("readFromFile:ofType:" "lispeditordocument")
    (:id path :id type :bool)
  (declare (ignorable type))
  (let* ((ts [textview "textStorage"])
	 (options [(@class "NSMutableDictionary") "dictionary"])
	 (url [(@class "NSURL" ) "fileURLWithPath:" :id path]))
    (#_NSLog #@"URL= %@" :id url)
    [[ts "mutableString"] "setString:" :id #@""]
    [ts "beginEditing"]
    (rlet ((attrs (* :id) (%null-ptr)))
      (let* ((win [ts "readFromURL:options:documentAttributes:"
		      :id url :id options :id attrs :<BOOL>]))
	[ts "endEditing"]
	win))))
    
    
	 
(define-objc-method ("loadDataRepresentation:ofType:" "lispeditordocument")
    (:id data :id type :<BOOL>)
  (declare (ignorable type))
  ;(#_NSLog #@"loadDataRepresentation:ofType (listener) type = %@" :address type)
  (setq filedata data)
  (if (%null-ptr-p data)
    #$NO
    #$YES))

(define-objc-method ("windowControllerDidLoadNib:" "lispeditordocument")
    (:id acontroller :void)
  ;(#_NSLog #@"windowControllerDidLoadNib (editor document)")
  [:super "windowControllerDidLoadNib:" :address acontroller]
  ;; Apple/NeXT thinks that adding extra whitespace around cut & pasted
  ;; text is "smart".  Really, really smart insertion and deletion
  ;; would alphabetize the selection for you (byChars: or byWords:);
  ;; sadly, if you want that behavior you'll have to do it yourself.
  ;; Likewise with the extra spaces.
  [textview "setAlignment:" :<NST>ext<A>lignment #$NSNaturalTextAlignment]
  [textview "setSmartInsertDeleteEnabled:" :<BOOL> #$NO]
  [textview "setRichText:" :<BOOL> #$NO]
  [textview "setUsesFontPanel:" :<BOOL> #$YES]
  [textview "setUsesRuler:" :<BOOL> #$NO]
  (push (make-cocoa-editor-info
	   :document (%setf-macptr (%null-ptr) self)
	   :controller (%setf-macptr (%null-ptr) acontroller)
	   :listener nil)
	  *open-editor-documents*)
  (set-objc-instance-variable acontroller "textview" textview)
  (set-objc-instance-variable acontroller "echoarea" echoarea)
  (set-objc-instance-variable acontroller "packagename" packagename)
  [textview "setDelegate:" :address acontroller]
  (let* ((font (default-font)))
    (multiple-value-bind (height width)
      (get-size-for-textview  font 24 80)
      (size-textview-containers textview height width))
    [textview "setTypingAttributes:"
	      :id (create-text-attributes
		   :font font
		   :color [(@class "NSColor") "blackColor"])]
    (unless (%null-ptr-p filedata)
      (rlet ((emptyrange :<NSR>ange :location 0 :length 0))
	[textview "replaceCharactersInRange:withString:"
		  :<NSR>ange emptyrange
		  :id [[(@class "NSString") "alloc"]
		       "initWithData:encoding:"
		       :id filedata
		       :unsigned
		       #$NSASCIIStringEncoding]]
	[acontroller "reparseModeline:" :id textview]))))

(define-objc-method ("close" "lispeditordocument") (:void)
  [:super "close"]
  (let* ((info (info-from-document self)))
    (when info
      (let* ((proc (cocoa-editor-info-listener info)))
        (when proc
	      (setf (cocoa-editor-info-listener info) nil)
	      (process-kill proc)))
      (without-interrupts (setq *open-editor-documents*
				 (delete info *open-editor-documents*))))))

;;; Syntax utilities

;;; If range is non-empty, return the current char without affecting range.
(defun current-char-in-range (rangeptr textbuf)
  (let* ((width (pref rangeptr :<NSR>ange.length)))
    (declare (ingeger width))
    (if (zerop width)
      nil
      (code-char
       [textbuf "characterAtIndex:"
		:unsigned-fullword (pref rangeptr :<NSR>ange.location)
		:unsigned-byte]))))

(defun next-char-in-range (rangeptr textbuf)
  (let* ((width (pref rangeptr :<NSR>ange.length)))
    (declare (integer width))
    (unless (zerop width)
      (setf (pref rangeptr :<NSR>ange.length) (1- width)
	    (pref rangeptr :<NSR>ange.location)
	    (1+ (pref rangeptr :<NSR>ange.location)))
      (current-char-in-range rangeptr textbuf))))

;;; Try to extend the range backward, unless its location is
;;; already at (or below) limit.
(defun prev-char-in-range (rangeptr textbuf &optional (limit 0))
  (let* ((pos (pref rangeptr :<NSR>ange.location)))
    (when (> pos limit)
      (setf (pref rangeptr :<NSR>ange.location)
	    (1- (pref rangeptr :<NSR>ange.location))
	    (pref rangeptr :<NSR>ange.length)
	    (1+ (pref rangeptr :<NSR>ange.length)))
      (current-char-in-range rangeptr textbuf))))

(defun forward-over-#-comment (rangeptr textbuf)
  ;; We've just read a "#|" : the range points to the |.  Return
  ;; T if the number of open #| comments reaches 0 (with the range
  ;; pointing to the outermost closing #), NIL if we hit EOF first.
  (do* ((count 1)
	(pending-open nil)
	(pending-close nil))
       ((zerop count) t)
    (declare (fixnum count))		; Pretty unlikely not to be.
    (case (next-char-in-range rangeptr textbuf)
      ((nil) (return))
      (#\| (if pending-open
	     (progn (incf count) (setq pending-open nil))
	     (setq pending-close t)))
      (#\# (if pending-close
	     (progn (decf count) (setq pending-close nil))
	     (setq pending-open t))))))

(defun backward-over-#-comment (rangeptr textbuf &optional (limit 0))
  ;; We've just read a trailing "|#" : the range points to the |.  Return
  ;; T if the number of open #| comments reaches 0 (with the range
  ;; pointing to the outermost closing #), NIL if we hit EOF first.
  (do* ((count 1)
	(pending-open nil)
	(pending-close nil))
       ((zerop count) t)
    (declare (fixnum count))		; Pretty unlikely not to be.
    (case (prev-char-in-range rangeptr textbuf limit)
      ((nil) (return))
      (#\| (if pending-open
	     (progn (incf count) (setq pending-open nil))
	     (setq pending-close t)))
      (#\# (if pending-close
	     (progn (decf count) (setq pending-close nil))
	     (setq pending-open t))))))

(defun forward-until-match (rangeptr textbuf matchchar)
  (do* ((ch (next-char-in-range rangeptr textbuf)
	    (next-char-in-range rangeptr textbuf)))
       ((eql ch matchchar) t)
    (when (null ch)
      (return nil))))

;;; Range points to #\; .  Win if we find a newline before EOF; leave
;;; range pointing to newline on success.
(defun forward-over-semi-comment (rangeptr textbuf)
  (forward-until-match rangeptr textbuf #\Newline))

;;; (Harder to find semi-comments backward ...)

;;; Range points to #\|; find match & leave range pointing there.
(defun forward-over-multi-escape (rangeptr textbuf)
  (forward-until-match rangeptr textbuf #\|))

;;; Advance over a string.  The range points to a leading (unescaped)
;;; #\".  If we find a trailing unescaped #\", return T with the
;;; range pointing to it, else return NIL.
(defun forward-over-string (rangeptr textbuf)
  (do* ((ch (next-char-in-range rangeptr textbuf)
	    (next-char-in-range rangeptr textbuf)))
       ((null ch))
    (if (eql ch #\")
      (return t)
      (if (eql ch #\\)
	(when (null (next-char-in-range rangeptr textbuf))
	  (return nil))))))

;;; The range points to the trailing unescaped #\".  Back up until
;;; we find a matching unescaped #\".  (We have to back up an extra
;;; char, then move forward if the extra char wasn't a #\\.)  Return
;;; T (with the range pointing at the leading #\"), else NIL.
(defun backward-over-string (rangeptr textbuf &optional (limit 0))
  (do* ((ch (prev-char-in-range rangeptr textbuf limit)
	    (prev-char-in-range rangeptr textbuf limit)))
       ((null ch) nil)
    (when (eql ch #\")
      (setq ch (prev-char-in-range rangeptr textbuf limit))
      (if (null ch)
	(return)
	(unless (eql ch #\\)
	  (next-char-in-range rangeptr textbuf)
	  (return t))))))

;;; Point the range to the first non-whitespace character.
(defun forward-skip-whitespace (rangeptr textbuf)
  (do* ((ch (current-char-in-range rangeptr textbuf)
	    (next-char-in-range rangeptr textbuf)))
       ((null ch))
    (unless (whitespacep ch)
      (return t))))

;;; Range points to list-open character (e.g., open-paren.)  Return
;;; T if we can advance so that range points to list-close char,
;;; seeing nothing but balanced expressions along the way.
(defun forward-over-list (rangeptr textbuf &optional (close #\)))
  (loop
      (let* ((ch (next-char-in-range rangeptr textbuf)))
	(if (eql ch close)
	  (return t)
	  (case ch
	    ((nil #\) #\] #\}) (return nil))
	    ;; I suppose that this could be made non-recursive.
	    ;; Anything nested more than a dozen or two levels
	    ;; deep probably means that the cat fell asleep
	    ;; on the keyboard ...
	    (#\( (unless (forward-over-list rangeptr textbuf #\))
		 (return nil)))
	    (#\[ (unless (forward-over-list rangeptr textbuf #\])
		   (return nil)))
	    (#\{ (unless (forward-over-list rangeptr textbuf #\})
		   (return nil)))

	    (#\# (setq ch (next-char-in-range rangeptr textbuf))
		 (if (or (null ch)
			 (and (eql ch #\|)
			      (not (forward-over-#-comment rangeptr textbuf))))
		   (return nil)))
	    (#\" (unless (forward-over-string rangeptr textbuf)
		   (return nil)))
	    (#\| (unless (forward-over-multi-escape rangeptr textbuf))
		 (return nil))
	    (#\\ (if (null (next-char-in-range rangeptr textbuf))
		   (return nil)))
	    (#\; (unless (forward-over-semi-comment rangeptr textbuf)
		   (return nil))))))))

;;; Return (values T T) if all expressions in range are properly
;;; balanced and something other than semantic whitespace was
;;; seen, else return (values T NIL) if all expressions are
;;; balanced, else return (values NIL NIL) if some expression
;;; is unterminated but nothing's prematurely terminated, else
;;; return (values NIL T)
(defun balanced-expressions-in-range-forward (rangeptr textbuf)
  (do* ((ch (current-char-in-range rangeptr textbuf)
	    (next-char-in-range rangeptr textbuf))
	(seen-something-interesting nil))
       ((null ch) (return (values t seen-something-interesting)))
    (case ch
      ((#\) #\] #\}) (return (values nil t)))
      (#\( (if (forward-over-list rangeptr textbuf #\))
	     (setq seen-something-interesting t)
	     (return (values nil nil))))
      (#\[ (if (forward-over-list rangeptr textbuf #\])
	     (setq seen-something-interesting t)
	     (return (values nil nil))))
      (#\{ (if (forward-over-list rangeptr textbuf #\})
	     (setq seen-something-interesting t)
	     (return (values nil nil))))
      (#\" (if (forward-over-string rangeptr textbuf)
	     (setq seen-something-interesting t)
	     (return (values nil nil))))
      (#\| (if (forward-over-multi-escape rangeptr textbuf)
	     (setq seen-something-interesting t)
	     (return (values nil nil))))
      (#\; (unless (forward-over-semi-comment rangeptr textbuf)
	     (return (values nil nil))))
      (#\# (let* ((nextch (next-char-in-range rangeptr textbuf)))
	     (if (null nextch)
	       (return (values nil nil))
	       (if (eql nextch #\|)
		 (unless (forward-over-#-comment rangeptr textbuf)
		   (return (values nil nil)))))))
      (t
       (unless seen-something-interesting
	 (unless (whitespacep ch)
	   (setq seen-something-interesting t)))))))
  
(defun previous-start-of-defun (textbuf startpos)
  (rlet ((linerange :<NSR>ange)
	 (posrange :<NSR>ange :length 0))
    (do* ((pos startpos (1- (pref linerange :<NSR>ange.location))))
	 ((< pos 0))
      (setf (pref posrange :<NSR>ange.location) pos)
      [textbuf "lineRangeForRange:" :<NSR>ange posrange (:-> linerange)]
      (if (eql (current-char-in-range linerange textbuf) #\()
	(return (pref linerange :<NSR>ange.location))))))

;;; This is almost completely wrong: we need to ensure that the form
;;; is read in the correct package, etc.
(defun send-to-top-listener (sender-info nsstring &optional (append-newline t))
  (declare (ignorable sender-info))
  ;(#_NSLog #@"sending string \"%@\"" :address nsstring)
  (let* ((listener
	  (info-from-document [(@class "lisplistenerdocument")
			       "topListener"])))
    (when listener
      (let* ((controller (cocoa-editor-info-controller listener)))
	[controller "sendString:" :address nsstring]
	(when append-newline
	  [controller "sendString:" :address #@"
"]
	  )))))
