;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html


(defvar *inhibit-abort* nil)

#+ppc-target
(defppclapfunction break-event-pending-p ()
  (ref-global arg_z arch::intflag)
  (set-global rzero arch::intflag)
  (cmpwi arg_z 0)
  (mr arg_z rnil)
  (beqlr)
  (la arg_z arch::t-offset rnil)
  (blr))

#+sparc-target
(defsparclapfunction break-event-pending-p ()
  (ref-global %arg_z arch::intflag)
  (set-global %rzero arch::intflag)
  (tst %arg_z)
  (mov %rnil %arg_z)
  (bne.a @1)
   (inc arch::t-offset %arg_z)
  @1
  (retl)
   (nop))


; If any bits in the *periodic-task-mask* are set in the ptaskstate.flags word of
; a periodic task, it will not be run
(defvar *periodic-task-mask* 0)

(defmethod print-object ((p periodic-task) stream)
  (print-unreadable-object (p stream :type t :identity t)
    (format stream "~s ~d"
	    (ptask.name p)
	    (pref (ptask.state p) :ptaskstate.interval))))

(defun find-named-periodic-task (name)
  (dolist (task *%periodic-tasks%*)
    (when (eq name (ptask.name task))
      (return task))))

(defun %install-periodic-task (name function interval &optional 
                                    (flags 0)
                                    (privatedata (%null-ptr)))
  (without-interrupts
   (let* ((already (find-named-periodic-task name))
          (state (if already (ptask.state already) 
		     (malloc (record-length ptaskstate))))
          (task (or already (%istruct 'periodic-task state name nil))))
     (setf (ptask.function task) function)
     (setf (rref state ptaskstate.interval) interval
           (rref state ptaskstate.flags) flags
           (rref state ptaskstate.private) privatedata
           (rref state ptaskstate.nexttick) (%tick-sum (get-tick-count) interval))
     (unless already (push task *%periodic-tasks%*))
     task)))

(defmacro with-periodic-task-mask ((mask &optional disable-gc-polling) &body body)
  (let ((thunk (gensym)))
    `(let ((,thunk #'(lambda () ,@body)))
       (funcall-with-periodic-task-mask ,mask ,disable-gc-polling ,thunk))))

(defvar *periodic-task-masks* nil)

(defvar *gc-polling-disable-count* 0)
(declaim (fixnum *gc-polling-disable-count*))

; All this hair is so that multiple processes can vote on the *periodic-task-mask*
(defun funcall-with-periodic-task-mask (mask disable-gc-polling thunk)
  (let* ((cell (list mask)))
    (declare (dynamic-extent cell))
    (flet ((logior-list (list)
             (declare (type list list))
             (let ((res 0))
               (declare (fixnum res))
               (loop
                 (when (null list) (return res))
                 (setq res (%ilogior res (pop list)))))))
      (declare (inline logior-list))
      (unwind-protect
        (progn
          (without-interrupts
           (setf (cdr cell) *periodic-task-masks*
                 *periodic-task-masks* cell)
           (setq *periodic-task-mask* (logior-list *periodic-task-masks*))
           (when disable-gc-polling
             (incf *gc-polling-disable-count*)
             (setf *gc-event-status-bits* 
                   (%ilogior (lsh 1 $gc-polling-enabled-bit) *gc-event-status-bits*))
             ; making this a bignum (on 68k) is in very poor taste
             ;(bitsetf $gc-polling-enabled-bit (the fixnum *gc-event-status-bits*))
             ))
          (funcall thunk))
        (without-interrupts
         (let* ((first *periodic-task-masks*)
                (this first)
                (last nil))
           (declare (type cons first this last))
           (loop
             (when (eq this cell)
               (if last
                 (setf (cdr last) (cdr this))
                 (pop first))
               (return (setq *periodic-task-masks* first)))
             (setq last this
                   this (cdr this))))
         (setq *periodic-task-mask* (logior-list *periodic-task-masks*))
         (when disable-gc-polling
           (when (eql 0 (decf *gc-polling-disable-count*))
             ; as far as I can tell PPC gc ignores all this event stuff
             (setf *gc-event-status-bits* 
                   (%ilogand (%ilognot (lsh 1 $gc-polling-enabled-bit)) *gc-event-status-bits*))
             ;(bitclrf $gc-polling-enabled-bit (the fixnum *gc-event-status-bits*))
             )))))))


(defun force-break-in-listener (p)
  (process-interrupt p
		     #'(lambda () (let* ((*interrupt-level* 0))
				    (break)
				    (clear-input *terminal-io*)))))

; We only let one process at a time run periodic tasks.
; Normally, they will run to completion since they run without-interrupts,
; but if one of them does process-wait (e.g. with-focused-view), some
; other process might be scheduled. This flag prevents deadlock by
; letting that other process do periodic tasks as well.
(defglobal *running-periodic-tasks* nil)
(defun cmain (&optional preempt)
  (if preempt
    #+proxy-scheduler
    (%proxy-preemption-handler)
    #-proxy-scheduler
    nil
    (progn
      (unless *inhibit-abort*
	(when (break-event-pending-p)
	  (force-break-in-listener *interactive-abort-process*)))
      (unless  *in-scheduler*
	(let-globally ((*in-scheduler* t))
		      (let* ((c *current-process*))
			(when (and c (> (%tick-difference (get-tick-count) (process.nexttick c))
					0))
			  (suspend-current-process)))))
      (flet ((maybe-run-periodic-task (task)
	       (let ((now (get-tick-count))
		     (state (ptask.state task)))
		 (when (and (>= (%tick-difference now (rref state ptaskstate.nexttick))
				0)
			    (eql 0 (logand (the fixnum (rref state ptaskstate.flags))
					   (the fixnum *periodic-task-mask*))))
		   (setf (rref state ptaskstate.nexttick) (+ now (rref state ptaskstate.interval)))
		   (funcall (ptask.function task))))))
	(let ((event-dispatch-task *event-dispatch-task*))
	  (maybe-run-periodic-task event-dispatch-task)
	  (without-interrupts
	   (bitclrf $gc-allow-stack-overflows-bit *gc-event-status-bits*)
	   (unless *running-periodic-tasks*
	     (let-globally ((*running-periodic-tasks* t))
			   (dolist (task *%periodic-tasks%*)
			     (unless (eq task event-dispatch-task)
			       (maybe-run-periodic-task task)))))))))))


(defun %remove-periodic-task (name)
  (without-interrupts
   (let ((task (find-named-periodic-task name)))
     (when task (setq *%periodic-tasks%* (delete task *%periodic-tasks%*)))
     task)))


(defun event-poll ()
  (without-interrupts
   (force-output *terminal-io*))
)


; Is it really necessary to keep this guy in a special variable ?
(defloadvar *event-dispatch-task* 
  (%install-periodic-task 
   'event-poll
   'event-poll
   20
   (+ $ptask_draw-flag $ptask_event-dispatch-flag)))


(defun event-ticks ()
  (let ((task *event-dispatch-task*))
    (when task (rref (ptask.state task) ptaskstate.interval))))

(defun set-event-ticks (n)
  (setq n (require-type n '(integer 0 3767)))   ;  Why this weird limit ?
  (let ((task *event-dispatch-task*))
    (when task (setf (rref (ptask.state task) ptaskstate.interval) n))))

;; Making the *initial-process* quit will cause an exit(),
;; though it might be nicer if all processes were shut down
;; in an orderly manner first.  This is the not-so-nice way
;; of quitting ...
(defun %quit ()
  (quit))



; end of L1-events.lisp

