;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html

(in-package "CCL")

(eval-when (:compile-toplevel :execute)
  (require "SOLARIS-RECORDS")
  (require "SPARC-ARCH")
  (require "SPARCENV")
  (require "NUMBER-MACROS")

(defparameter *sparc-instruction-fields*
  `((:op . ,(byte 2 30))
    (:disp30 . ,(byte 30 0))
    (:rd . ,(byte 5 25))
    (:a . , (byte 1 29))
    (:cond . ,(byte 4 25))
    (:op2 . ,(byte 3 22))
    (:imm22 . ,(byte 22 0))
    (:disp22 . ,(byte 22 0))
    (:op3 . ,(byte 6 19))
    (:rs1 . ,(byte 5 14))
    (:i . ,(byte 1 13))
    (:simm13 . ,(byte 13 0))
    (:opf . ,(byte 8 5))
    (:rs2 . ,(byte 5 0))))

(defun sparc-instruction-field (field-name)
  (or (cdr (assoc field-name *sparc-instruction-fields*))
      (error "Unknown SPARC instruction field: ~s" field-name)))

(defun sparc-instruction-field-mask (field-spec)
  (let* ((name (if (atom field-spec) field-spec (car field-spec)))
	 (value (if (atom field-spec) -1 (cadr field-spec))))
    (dpb value (sparc-instruction-field name) 0)))



(defmacro with-sparc-xp-registers-and-offset ((xp register-number) (registers offset) &body body)
  `(with-macptrs ((,registers))
    (let* ((,offset (%sparc-xp-register-offset ,xp ,registers ,register-number)))
      ,@body)))

(defmacro rs1-field (i)
  `(ldb (byte 5 14) ,i))

(defmacro rs2-field (i)
  `(ldb (byte 5 0) ,i))

(defmacro rd-field (i)
  `(ldb (byte 5 25) ,i))

(defmacro simm13-field (i)
  `(%sparc-simm13 ,i))

(defmacro sparc-lisp-reg-p (reg)
  `(logbitp ,reg sparc-node-regs))

(defmacro sparc-lap-word (instruction-form)
  (uvref (uvref (compile-named-function
		 `(lambda (&lap 0)
		   (sparc-lap-function () ((?? 0) (r? %rzero))
		    ,instruction-form (nop)))
		 nil nil nil nil nil nil :sparc) 0) 0))

(defmacro sparc-instruction-mask (&rest fields)
  `(logior ,@(mapcar #'sparc-instruction-field-mask (cons :op fields))))
)

(defun %sparc-simm13 (i)
  (let* ((val (ldb (byte 13 0) i)))
    (if (logbitp 12 val)
       (- val (ash 1 13))
       val)))

;; Destructively modifies p.  An error to call this on %g0.
(defun %sparc-xp-register-offset (xp p reg)
  (cond ((member reg '(:psr :pc :npc :y))
	 (%setf-macptr p (%inc-ptr xp #.(get-field-offset :ucontext.uc-mcontext.gregs)))
	 (case reg
	   (:psr 0)
	   (:pc 4)
	   (:npc 8)
	   (:y 12)))
	((or (< reg 1) (> reg 31))
	 (error "Invalid register: ~d" reg))
	((< reg 16)
	 (%setf-macptr p (%inc-ptr xp #.(get-field-offset :ucontext.uc-mcontext.gregs)))
	 (ash (+ 3 reg) 2))
	(t
	 (%setf-macptr p (raref xp :ucontext.uc-mcontext.gregs (+ 3 sparc::%sp)))
	 (ash (- reg 16) 2))))

(defun sparc-xp-gpr-lisp (xp register-number)
  (if (eql register-number 0)
      0
      (with-sparc-xp-registers-and-offset (xp register-number) (registers offset)
					  (values (%get-object registers offset)))))

(defun (setf sparc-xp-gpr-lisp) (value xp register-number)
  (if (eql register-number 0)
      (if (eql value 0)
	  0
	  (error "Can't set GPR 0!"))
      (with-sparc-xp-registers-and-offset (xp register-number) (registers offset)
					  (%set-object registers offset value))))

(defun sparc-xp-gpr-signed-long (xp register-number)
  (if (eql register-number 0)
      0
      (with-sparc-xp-registers-and-offset (xp register-number) (registers offset)
					  (values (%get-signed-long registers offset)))))

(defun sparc-xp-gpr-macptr (xp register-number)
  (if (eql register-number 0)
      (%null-ptr)
      (with-sparc-xp-registers-and-offset (xp register-number) (registers offset)
					  (values (%get-ptr registers offset)))))

(defun sparc-xp-argument-list (xp)
  (let* ((nargs (sparc-xp-gpr-lisp xp sparc::%nargs)) ; tagged as a fixnum
	 (arg-x (sparc-xp-gpr-lisp xp sparc::%arg_x))
	 (arg-y (sparc-xp-gpr-lisp xp sparc::%arg_y))
	 (arg-z (sparc-xp-gpr-lisp xp sparc::%arg_z)))
    (cond ((eql nargs 0) nil)
	  ((eql nargs 1) (list arg-z))
	  ((eql nargs 2) (list arg-y arg-z))
	  (t (let* ((args (list arg-x arg-y arg-z))
		    (vsp (xp-gpr-macptr xp sparc::%vsp)))
	       (dotimes (i (- nargs 3) args)
		 (push (%get-object vsp (* i 4)) args)))))))
(defparameter *trap-lookup-tries* 5)



(defun %scan-for-instr (mask opcode fn pc-index tries)
  (let ((code-vector (and fn (uvref fn 0)))
        (offset 0))
    (declare (fixnum offset))
    (flet ((get-instr ()
             (if code-vector
               (let ((index (+ pc-index offset)))
                 (unless (< index 0)
		   (uvref code-vector index)))
               (%get-long pc-index (the fixnum (* 4 offset))))))
      (declare (dynamic-extent #'get-instr))
      (dotimes (i tries)
        (decf offset)
        (let ((instr (get-instr)))
	  (unless instr (return))
          (when (match-instr instr mask opcode)
            (return instr))
          (when (codevec-header-p instr)
            (return nil)))))))

(defvar *error-reentry-count* 0)

(defun funcall-with-error-reentry-detection (thunk)
  (let* ((count *error-reentry-count*)
         (*error-reentry-count* (1+ count)))
    (cond ((eql count 0) (funcall thunk))
          ((eql count 1) (error "Error reporting error"))
          (t (bug "Error reporting error")))))

(defun sparc-return-address-offset (xp fn reg)
  (with-sparc-xp-registers-and-offset (xp reg) (regs reg-offset)
    (if (typep fn 'function)
      (without-interrupts
       (let* ((lr (%get-object regs reg-offset))
	      (code-vector (uvref fn 0))
	      (fn-base (%uvector-data-fixnum code-vector))
	      (offset (- lr fn-base)))
	 (declare (fixnum lr fn-base offset))
	 (if (and (>= offset 0)
		  (< offset (uvsize code-vector)))
	     (ash offset arch::fixnum-shift)
	     (%get-ptr regs reg-offset))))
      (%get-ptr regs reg-offset))))
    				      
  
; When a trap happens, we may have not yet created control
; stack frames for the functions containing PC & RA0.
; If that is the case, we add fake-stack-frame's to *fake-stack-frames*
; There are 4 cases:
;
; PC in FN
;   Push 1 stack frame: PC/FN
;   This might miss one recursive call, but it won't miss any variables
; PC in NFN
;   Push 2 stack frames:
;   1) PC/NFN/VSP
;   2) RA0/FN/VSP
;   This might think some of NFN's variables are part of FN's stack frame,
;   but that's the best we can do.
; RA0 in FN
;   Push 1 stack frame: RA0/FN
; None of the above
;   Push no new stack frames
;
; The backtrace support functions in "ccl:l1;sparc-stack-groups.lisp" know how
; to find the fake stack frames and handle them as arguments.
(defun funcall-with-xp-stack-frames (xp trap-function thunk)
  (cond ((null trap-function)
         ; Maybe inside a subprim from a lisp function
         (let* ((fn (sparc-xp-gpr-lisp xp sparc::%fn))
                (ra0 (sparc-return-address-offset xp fn sparc::%ra0)))
           (if (fixnump ra0)
             (let* ((lsp (sparc-xp-gpr-lisp xp sparc::%lsp))
                    (vsp (sparc-xp-gpr-lisp xp sparc::%vsp))
                    (frame (%cons-fake-stack-frame lsp lsp fn ra0 vsp *fake-stack-frames*))
                    (*fake-stack-frames* frame))
               (declare (dynamic-extent frame))
               (funcall thunk frame))
             (funcall thunk (sparc-xp-gpr-lisp xp sparc::%lsp)))))
        ((eq trap-function (sparc-xp-gpr-lisp xp sparc::%fn))
         (let* ((sp (sparc-xp-gpr-lisp xp sparc::%lsp))
                (fn trap-function)
                (ra0 (sparc-return-address-offset xp fn :pc))
                (vsp (sparc-xp-gpr-lisp xp sparc::%vsp))
                (frame (%cons-fake-stack-frame sp sp fn ra0 vsp *fake-stack-frames*))
                (*fake-stack-frames* frame))
           (declare (dynamic-extent frame))
           (funcall thunk frame)))
        ((eq trap-function (sparc-xp-gpr-lisp xp sparc::%nfn))
         (let* ((sp (sparc-xp-gpr-lisp xp sparc::%lsp))
                (fn (sparc-xp-gpr-lisp xp sparc::%fn))
                (ra0 (sparc-return-address-offset xp fn sparc::%ra0))
                (vsp (sparc-xp-gpr-lisp xp sparc::%vsp))
                (lr-frame (%cons-fake-stack-frame sp sp fn ra0 vsp))
                (pc-fn trap-function)
                (pc-ra0 (sparc-return-address-offset xp pc-fn :pc))
                (pc-frame (%cons-fake-stack-frame sp lr-frame pc-fn pc-ra0 vsp *fake-stack-frames*))
                (*fake-stack-frames* pc-frame))
           (declare (dynamic-extent lr-frame pc-frame))
           (funcall thunk pc-frame)))
        (t (funcall thunk (sparc-xp-gpr-lisp xp sparc::%lsp)))))


;; Enter here from handle-trap in "lisp-exceptions.c".
;; xp is a pointer to an ExceptionInformationPowerPC record.
;; the-trap is the trap instruction that got us here.
;; fn-reg is either fn, nfn or 0. If it is fn or nfn, then
;; the trap occcurred in that register's code vector.
;; If it is 0, then the trap occurred somewhere else.
;; pc-index is either the index in fn-reg's code vector
;; or, if fn-reg is 0, the address of the PC at the trap instruction.
;; This code parallels the trap decoding code in
;; "lisp-exceptions.c" that runs if (symbol-value 'cmain)
;; is not a macptr.
;; Some of these could probably call %err-disp instead of error,
;; but I was too lazy to look them up.

(defcallback cmain (:address xp 
                             :unsigned-fullword fn-reg 
                             :address pc-or-index 
                             :unsigned-fullword trap-hi-16
                             :unsigned-fullword trap-lo-16
                             :signed-fullword ignore-1)
  (declare (ignore ignore-1))
  (let* ((fn (unless (eql fn-reg 0) (sparc-xp-gpr-lisp xp fn-reg)))
	 (trap-number (ldb (byte 7 0) trap-lo-16))
	 (trap-cond (ldb (byte 4 (- 25 16)) trap-hi-16)))
    (with-xp-stack-frames (xp fn frame-ptr)
      (if (eql trap-number sparc::trap-event-poll)
        (unwind-protect
          (progn
            (handle-gc-hooks)
            (setq *interrupt-level* 0)
            (cmain))
          ; Set the first binding of *interrupt-level* to >= 0
          ; This is the one saved by the without-interrupts in %pascal-functions%
          (do-db-links (db var val)
            (when (eq var '*interrupt-level*)
              (unless (>= (the fixnum val) 0)
                (setf (%fixnum-ref db 8) 0))
              (return))))
        (with-error-reentry-detection
          (let* ((pc-index (if (eql fn-reg 0) pc-or-index (%ptr-to-int pc-or-index)))
                instr cmp-instr rs1 condition)
            (cond
             ((eql trap-number sparc::trap-nargs)
	      (%error "Wrong number of arguments" nil frame-ptr))
	       
	     ((and (eql trap-number sparc::trap-fulltag-check)
		   (eql trap-cond sparc::condne)
		   (setq instr (scan-for-instr
				(sparc-instruction-mask :op3 :i :simm13 )
				(sparc-lap-word (ldub (r? arch::misc-subtag-offset) r?))
				fn pc-index))
		   (setq cmp-instr (scan-for-instr
				    (sparc-instruction-mask :op3 :i :rd)
				    (sparc-lap-word (cmp r? ??))
				    fn pc-index))
                   (sparc-lisp-reg-p (setq rs1 (rs1-field instr))))
              (let* ((typecode (simm13-field cmp-instr))
                     (type-tag (logand typecode arch::fulltagmask))
                     (type-name (svref (if (eql type-tag ppc::fulltag-nodeheader)
					   *nodeheader-types*
					   *immheader-types*)
                                       (ldb (byte (- arch::num-subtag-bits ppc::ntagbits) arch::ntagbits) typecode))))
                (%error (make-condition 'type-error
                                        :format-control (%rsc-string $XWRONGTYPE)
                                        :datum (sparc-xp-GPR-lisp xp rs1)
                                        :expected-type type-name)
                        nil
                        frame-ptr)))
	     ((and (eql trap-number sparc::trap-lowbyte-check)
		   (eql trap-cond sparc::condne)
		   (setq instr (scan-for-instr (sparc-instruction-mask :op3 :i :simm13)
					       (sparc-lap-word
						(and r? arch::subtag-mask r?))
					       fn pc-index))
		   (setq cmp-instr (scan-for-instr
				    (sparc-instruction-mask :op3 :rd :i :simm13)
				    (sparc-lap-word (cmp r? arch::subtag-character))
				    fn pc-index))
		   (sparc-lisp-reg-p (setq rs1 (rs1-field instr))))
              (%error (make-condition 'type-error
                                        :datum (sparc-xp-GPR-lisp xp rs1)
                                        :expected-type 'character)
                        nil
                        frame-ptr))
	     ((and (or (eql trap-number sparc::trap-fulltag-check)
		       (eql trap-number sparc::trap-lisptag-check))
		   (eql trap-cond sparc::condne)
                   (setq instr (scan-for-instr (sparc-instruction-mask :op3 :i)
                                               (sparc-lap-word (and r? ?? r?))
					       fn pc-index))
		   (setq cmp-instr (scan-for-instr (sparc-instruction-mask :op3 :rd)
						   (sparc-lap-word (cmp r? ??))
						   fn pc-index))
                   (sparc-lisp-reg-p (setq rs1 (RS1-field instr))))
              (let* ((tag (logand cmp-instr arch::tagmask))
                     (type-name 
                      (case tag
                        (#.ppc::tag-fixnum 'fixnum)
                        (#.ppc::tag-list (if (eql trap-number sparc::trap-fulltag-check) 'cons 'list))
                        (#.ppc::tag-misc 'uvector)
                        (#.ppc::tag-imm 'immediate))))                                      
                (%error (make-condition 'type-error
                                        :datum (sparc-xp-GPR-lisp xp rs1)
                                        :expected-type type-name)
                        nil
                        frame-ptr)))
             ((and (eq trap-number sparc::trap-unbound-variable)
		   (setq instr (scan-for-instr (sparc-instruction-mask :op3 :i :simm13)
					       (sparc-lap-word (ld (r? arch::symbol.vcell) r?))
					       fn pc-index))
		   (setq cmp-instr (scan-for-instr (sparc-instruction-mask :op3 :rd :i :simm13)
						   (sparc-lap-word (cmp r? arch::unbound-marker))
						   fn pc-index))
		   (sparc-lisp-reg-p (setq rs1 (rs1-field instr))))
              (setf (sparc-xp-GPR-lisp xp (Rs1-field cmp-instr))
                    (%kernel-restart-internal $xvunbnd (list (sparc-xp-GPR-lisp xp rs1)) frame-ptr)))
	     ((and (eq trap-number sparc::trap-bounds-check)
		   (setq instr (scan-for-instr (sparc-instruction-mask :op3 :i)
					       (sparc-lap-word (ld (r? ??) r?))
					       fn pc-index))
		   (setq cmp-instr (scan-for-instr (sparc-instruction-mask :op3 :rd)
						   (sparc-lap-word (cmp r? r?))
						   fn pc-index))
		   (sparc-lisp-reg-p (setq rs1 (rs1-field instr))))
              (%error (%rsc-string $xarroob)
                      (list (sparc-xp-GPR-lisp xp (Rs1-field cmp-instr))
                            (sparc-xp-GPR-lisp xp rs1))
                      frame-ptr))
             ;; twi 27 ra d - array header rank check
	     ((and (eql trap-number sparc::trap-rank-check)
		   (setq instr (scan-for-instr (sparc-instruction-mask :op3 :i :simm13)
					       (sparc-lap-word (ld (r? arch::arrayH.rank) r?))
                                               fn pc-index))
		   (setq cmp-instr (scan-for-instr (sparc-instruction-mask :op3 :rd :i)
						   (sparc-lap-word (cmp r? ??))
						   fn pc-index))
		   (sparc-lisp-reg-p (setq rs1 (Rs1-field instr))))
	      (%error (%rsc-string $xndims)
		      (list (sparc-xp-gpr-lisp xp rs1)
			    (ash (simm13-field cmp-instr) (- arch::fixnumshift)))
		      frame-ptr))
	     ((and (eql trap-number sparc::trap-array-flags-check)
		   (setq instr (scan-for-instr (sparc-instruction-mask :op3 :i :simm13)
                                               (sparc-lap-word (ld (r? arch::arrayH.flags) r?))
                                               fn pc-index))
		   (setq cmp-instr (scan-for-instr (sparc-instruction-mask :op3 :rd)
						   (sparc-lap-word (cmp r? r?))
						   fn pc-index))		   
		   (sparc-lisp-reg-p (setq rs1 (Rs1-field instr)))
		   (let* ((expected (sparc-xp-gpr-lisp xp (Rs2-field cmp-instr)))
			  (expected-subtype (ldb
					     arch::arrayH.flags-cell-subtag-byte
					     expected))
			  (expect-simple (=
					  (ldb arch::arrayH.flags-cell-bits-byte
					       expected)
					  (ash 1 $arh_simple_bit)))
			  (type-name
			   (case expected-subtype
			     (#.arch::subtag-double-float-vector 'double-float))))

		     (and type-name expect-simple
			  (setq condition
				(make-condition 'type-error
						:datum (sparc-xp-gpr-lisp xp rs1)
						:expected-type
						`(simple-array ,type-name))))))
	      (%error condition nil frame-ptr))
			       
             ;; Unknown trap
             (t (%error "Unknown trap: #x~x~%xp: ~s, fn: ~s, pc: #x~x"
                        (list (logior (ash trap-hi-16 16) trap-lo-16)
			      xp fn (ash pc-index arch::fixnumshift))
                        frame-ptr)))))))))


(defun handle-gc-hooks ()
  (let ((bits *gc-event-status-bits*))
    (declare (fixnum bits))
    (cond ((logbitp $gc-postgc-pending-bit bits)
           (setq *gc-event-status-bits*
                 (logand (lognot (+ (ash 1 $gc-pregc-pending-bit)
                                    (ash 1 $gc-postgc-pending-bit)))
                         bits))
           (let ((f *post-gc-hook*))
             (when (functionp f) (funcall f))))
          ((logbitp $gc-pregc-pending-bit bits)
           (setq *gc-event-status-bits* (bitclr $gc-pregc-pending-bit bits))
           (let ((f *pre-gc-hook*))
             (when (functionp f) (funcall f)))))))
