/*************************************************************************
 *
 *  $RCSfile: xline.cxx,v $
 *
 *  $Revision: 1.7.206.1 $
 *
 *  last change: $Author: vg $ $Date: 2004/10/04 11:30:28 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#pragma hdrstop

#include <stdlib.h>
#include <math.h>
#include <tools/bigint.hxx>
#include <vcl/poly.hxx>
#include <vcl/svapp.hxx>
#include <vcl/virdev.hxx>
#include <vcl/metaact.hxx>
#include <vcl/gdimtf.hxx>
#include <vcl/gradient.hxx>
#include "xpoly.hxx"
#include "xoutx.hxx"

#ifndef INCLUDED_SVTOOLS_COLORCFG_HXX
#include <svtools/colorcfg.hxx>
#endif

#include <algorithm>

#define GLOBALOVERFLOW

/*************************************************************************
|*
|* eine der Haelften einer Bezierkurve berechnen
|*
\************************************************************************/

void SplitBezier(const XPolygon& rBez, XPolygon& rSplit, BOOL bFirstPart)
{
	if ( bFirstPart )
	{
		rSplit[0] = rBez[0];
		rSplit[1] = (rBez[0] + rBez[1]) / 2;
		rSplit[2] = (rBez[0] + rBez[1] * 2 + rBez[2]) / 4;
		rSplit[3] = (rBez[0] + rBez[1] * 3 + rBez[2] * 3 + rBez[3]) / 8;
	}
	else
	{
		rSplit[1] = (rBez[3] + rBez[2] * 2 + rBez[1]) / 4;
		rSplit[2] = (rBez[3] + rBez[2]) / 2;
		rSplit[3] = rBez[3];
	}
}

/*************************************************************************
|*
|* pruefen, ob die uebergebene Bezierkurve eine gerade Linie ist
|*
\************************************************************************/

BOOL IsBezierStraight(const XPolygon& rBez)
{
	// Point-on-Line-Test nach Graphics Gems I, S. 49f.
	long X0 = rBez[0].X(), Y0 = rBez[0].Y();
	long X1 = rBez[1].X(), Y1 = rBez[1].Y();
	long X2 = rBez[2].X(), Y2 = rBez[2].Y();
	long X3 = rBez[3].X(), Y3 = rBez[3].Y();
	BigInt DX(X3 - X0);
	BigInt DY(Y3 - Y0);
	BigInt nAbsDX(DX), nAbsDY(DY);
	nAbsDX.Abs();
	nAbsDY.Abs();
	BigInt nMax(nAbsDX > nAbsDY ? nAbsDX : nAbsDY);
	BigInt nCompare;

	// Kontrollpunkte auf der (unendlichen) Linie durch P3 und P0?
	nCompare = DY * BigInt(X1-X0) - DX * BigInt(Y1-Y0);
	nCompare.Abs();
	if ( nCompare >= nMax )
		return FALSE;

	nCompare = DY * BigInt(X2-X0) - DX * BigInt(Y2-Y0);
	nCompare.Abs();
	if ( nCompare >= nMax )
		return FALSE;

	//                                                  ____
	// wenn ja, dann pruefen, ob ausserhalb der Strecke P3P0
	if ( (X3 < X0 && X0 < X1) || (Y3 < Y0 && Y0 < Y1) ) return FALSE;
	if ( (X1 < X0 && X0 < X3) || (Y1 < Y0 && Y0 < Y3) ) return FALSE;
	if ( (X0 < X3 && X3 < X1) || (Y0 < Y3 && Y3 < Y1) ) return FALSE;
	if ( (X1 < X3 && X3 < X0) || (Y1 < Y3 && Y3 < Y3) ) return FALSE;

	if ( (X3 < X0 && X0 < X2) || (Y3 < Y0 && Y0 < Y2) ) return FALSE;
	if ( (X2 < X0 && X0 < X3) || (Y2 < Y0 && Y0 < Y3) ) return FALSE;
	if ( (X0 < X3 && X3 < X2) || (Y0 < Y3 && Y3 < Y2) ) return FALSE;
	if ( (X2 < X3 && X3 < X0) || (Y2 < Y3 && Y3 < Y3) ) return FALSE;

	return TRUE;
}

/*************************************************************************
|*
|* Bezierkurve durch Iteration bestimmen
|*
\************************************************************************/

void XOutIterateBezier(const XPolygon& rBez, Rectangle& rRect, USHORT nMaxDepth)
{
	// Das Folgende Statement ist Optimierung
	if (rRect.IsInside(rBez[0]) && rRect.IsInside(rBez[1]) &&
		rRect.IsInside(rBez[2]) && rRect.IsInside(rBez[3])) return;

	if ( nMaxDepth == 0 || IsBezierStraight(rBez) )
	{
		long nX = rBez[3].X(),
			 nY = rBez[3].Y();
		rRect.Left()    = Min(nX, rRect.Left());
		rRect.Right()   = Max(nX, rRect.Right());
		rRect.Top()     = Min(nY, rRect.Top());
		rRect.Bottom()  = Max(nY, rRect.Bottom());
	}
	else
	{
		XPolygon aSplitBez(4);

		nMaxDepth--;
		SplitBezier(rBez, aSplitBez, TRUE);
		XOutIterateBezier(aSplitBez, rRect, nMaxDepth);
		aSplitBez[0] = aSplitBez[3];
		SplitBezier(rBez, aSplitBez, FALSE);
		XOutIterateBezier(aSplitBez, rRect, nMaxDepth);
	}
}

/*************************************************************************
|*
|*    XOutputDevice::XOutCalcXPolyExtent()
|*
|*    Beschreibung
|*    Ersterstellung    14.08.95 ESO
|*    Letzte Aenderung  15.08.95 ESO
|*
*************************************************************************/

Rectangle XOutCalcXPolyExtent(const XPolygon& rXPoly, OutputDevice* pOut)
{
	if ( rXPoly.GetPointCount() == 0 )
		return Rectangle();

	USHORT i;
	USHORT nPntMax=rXPoly.GetPointCount()-1;

	FASTBOOL bHasBezier=FALSE;
	Rectangle aRect(rXPoly[0],rXPoly[0]);
	// zunaechst das Rect der Stuetzstellen (ohne Kontrollpunkte) bestimmen
	Point aPt;
	for (i=nPntMax; i>0; i--) {
		if (!rXPoly.IsControl(i)) {
			aPt=rXPoly[i]; // lokal kopieren fuer bessere Performance
			if (aPt.X()<aRect.Left  ()) aRect.Left  ()=aPt.X();
			if (aPt.X()>aRect.Right ()) aRect.Right ()=aPt.X();
			if (aPt.Y()<aRect.Top   ()) aRect.Top   ()=aPt.Y();
			if (aPt.Y()>aRect.Bottom()) aRect.Bottom()=aPt.Y();
		} else bHasBezier=TRUE;
	}
	if (!bHasBezier) return aRect;

	if (pOut!=NULL)
		aRect = pOut->LogicToPixel(aRect);
	i=0;
	while ( i < nPntMax )
	{
		if ( i <= nPntMax - 3 && rXPoly.GetFlags(i+1) == XPOLY_CONTROL )
		{
			XPolygon aBez(4);

			if ( pOut )
			{
				aBez[0] = pOut->LogicToPixel(rXPoly[i]);
				aBez[1] = pOut->LogicToPixel(rXPoly[i+1]);
				aBez[2] = pOut->LogicToPixel(rXPoly[i+2]);
				aBez[3] = pOut->LogicToPixel(rXPoly[i+3]);
			}
			else
			{
				aBez[0] = rXPoly[i];
				aBez[1] = rXPoly[i+1];
				aBez[2] = rXPoly[i+2];
				aBez[3] = rXPoly[i+3];
			}
			XOutIterateBezier(aBez, aRect, 8);
			i += 3;
		}
		else
		{
			Point aPnt(rXPoly[++i]);
			if ( pOut )
				aPnt = pOut->LogicToPixel(aPnt);

			aRect.Left()    = Min(aPnt.X(), aRect.Left());
			aRect.Right()   = Max(aPnt.X(), aRect.Right());
			aRect.Top()     = Min(aPnt.Y(), aRect.Top());
			aRect.Bottom()  = Max(aPnt.Y(), aRect.Bottom());
		}
	}
	if ( pOut )
		aRect = pOut->PixelToLogic(aRect);
	return aRect;
}

/*************************************************************************
|*
|*    XOutputDevice::CalcBezierStepCount()
|*
|*    Beschreibung
|*    Ersterstellung    08.11.94
|*    Letzte Aenderung  14.12.94 ESO
|*
*************************************************************************/

long XOutCalcBezierStepCount( const XPolygon& rXPoly, USHORT nIndex,
							  OutputDevice* pOut, USHORT nRough )
{
	long nSteps;

	if( pOut || nRough )
	{
		const Point& aPt = rXPoly[ nIndex++ ];
		const Point& aPt1 = rXPoly[ nIndex++ ];
		const Point& aPt2 = rXPoly[ nIndex++ ];
		const Point& aPt3 = rXPoly[ nIndex ];
		long nDx1 = Abs( aPt1.X() - aPt.X() ) * 2;
		long nDy1 = Abs( aPt1.Y() - aPt.Y() ) * 2;
		long nDx2 = Abs( aPt3.X() - aPt2.X() ) * 2;
		long nDy2 = Abs( aPt3.Y() - aPt2.Y() ) * 2;
		long nDxHndl = Abs( aPt2.X() - aPt1.X() );
		long nDyHndl = Abs( aPt2.Y() - aPt1.Y() );

		long nDelta = Max(Max(nDx1, nDy1), Max(nDx2, nDy2));
		nDelta = Max(nDelta, Max(nDxHndl, nDyHndl));
		if( pOut )
			nDelta = pOut->LogicToPixel(Size(nDelta, 0)).Width();
		if( nRough )
			nSteps = nDelta / nRough + 4;
		else
			nSteps = nDelta / 25 + 4;
	}
	else nSteps = 10;

    // #i19674# For nSteps^3 in the vicinity of LONG_MAX, the
    // XOutCalcBezier algorithm below will overflow.
    return ::std::min( 1024L, nSteps );
}


/*************************************************************************
|*
|*    XOutputDevice::CalcBezier()
|*
|*    Beschreibung
|*    Ersterstellung    08.11.94
|*    Letzte Aenderung  14.12.94 ESO
|*
*************************************************************************/

void XOutCalcBezier( const XPolygon& rXPoly, USHORT nBezIndex, Polygon& rPoly,
					 USHORT nPolyIndex, long nSteps )
{
	if ( nPolyIndex > XPOLY_MAXPOINTS )
	{
		return;
	}

	double nX;
	double nY;
	double nX0, nX1, nX2, nX3;
	double nY0, nY1, nY2, nY3;

	long nDiff = nSteps - 1;
	long nDiv = nSteps * nSteps * nSteps;
	rPoly[nPolyIndex++] = rXPoly[nBezIndex];

	nX0 = rXPoly[nBezIndex  ].X();
	nY0 = rXPoly[nBezIndex++].Y();
	nX1 = rXPoly[nBezIndex  ].X() * 3;
	nY1 = rXPoly[nBezIndex++].Y() * 3;
	nX2 = rXPoly[nBezIndex  ].X() * 3;
	nY2 = rXPoly[nBezIndex++].Y() * 3;
	nX3 = rXPoly[nBezIndex  ].X();
	nY3 = rXPoly[nBezIndex  ].Y();

	for (long nStep = 1; nStep < nSteps; nStep++, nDiff--, nPolyIndex++)
	{
		long nAcc = nDiff * nDiff * nDiff;

		nX = nX0 * nAcc;
		nY = nY0 * nAcc;
		nAcc = nAcc / nDiff * nStep;
		nX += nX1 * nAcc;
		nY += nY1 * nAcc;
		nAcc = nAcc / nDiff * nStep;
		nX += nX2 * nAcc;
		nY += nY2 * nAcc;
		nAcc = nAcc / nDiff * nStep;
		nX += nX3 * nAcc;
		nY += nY3 * nAcc;

		rPoly[nPolyIndex].X() = (long) (nX / nDiv);
		rPoly[nPolyIndex].Y() = (long) (nY / nDiv);
	}
	rPoly[nPolyIndex] = rXPoly[nBezIndex];
}

/*************************************************************************
|*
|*    XOutputDevice::CreatePolygon()
|*
|*    Beschreibung
|*    Ersterstellung    09.11.94
|*    Letzte Aenderung  14.12.94
|*
*************************************************************************/

Polygon XOutCreatePolygon( const XPolygon& rXPoly, OutputDevice* pOut,
						   USHORT nRough )
{
	if ( rXPoly.GetPointCount() == 0 )
		return Polygon(0);

	USHORT i = 0;
	ULONG  nPolySize = 1;
	USHORT nPntMax = rXPoly.GetPointCount()-1;

	while (i<nPntMax) {
		if (i+3<=nPntMax && rXPoly.IsControl(i+1)) {
#ifdef DGB_UTIL
			// am 14.2.1997 von Joe
			if (!rXPoly.IsControl(i+2)) {
				ByteString aMsg("XOutCreatePolygon(): Fehlender Kontrollpunkt an Position ");
				aMsg += i+2;
				aMsg += ". Stattdessen ist dort eine normale Stuetzstelle.";
				DGB_ERROR(aMsg.GetBuffer());
			}
			if (rXPoly.IsControl(i+3)) {
				ByteString aMsg("XOutCreatePolygon(): Fehlende Stuetzstelle an Position ");
				aMsg += i+3;
				aMsg += ". Stattdessen ist dort ein Kontrollpunkt.";
				DGB_ERROR(aMsg.GetBuffer());
			}
#endif
			nPolySize+=(USHORT)XOutCalcBezierStepCount(rXPoly,i,pOut,nRough);
			i+=3;
		} else {
			nPolySize++;
			i++;
		}
	}

	if ( nPolySize > XPOLY_MAXPOINTS )
		nPolySize = XPOLY_MAXPOINTS;

	Polygon aPoly( (USHORT) nPolySize );
	USHORT  nPolyPos = 0, nPolyPosMax = nPolySize - 1;
    
    aPoly[ 0 ] = rXPoly[ i = 0 ];

	while( i < nPntMax && nPolyPos < nPolySize )
	{
		if( i + 3 <= nPntMax && rXPoly.GetFlags( i + 1 ) == XPOLY_CONTROL )
		{
			USHORT nSteps = (USHORT) XOutCalcBezierStepCount( rXPoly, i, pOut, nRough );
			
            if( nPolyPos + nSteps >= (USHORT) nPolySize )
				nSteps = (USHORT)( nPolySize - nPolyPos - 1 );
			
            XOutCalcBezier( rXPoly, i, aPoly, nPolyPos, nSteps );
			nPolyPos += (USHORT) nSteps;
			i += 3;
		}
		else if( nPolyPos < nPolyPosMax )
			aPoly[ ++nPolyPos ] = rXPoly[ ++i ];
	}

	return aPoly;
}

Polygon XOutCreatePolygonBezier( const XPolygon& rXPoly, OutputDevice* pOut )
{
	sal_uInt16 i, nPtCount = rXPoly.GetPointCount();
	Polygon aPoly( nPtCount );
	for ( i = 0; i < nPtCount; i++ )
	{
		aPoly[ i ] = rXPoly[ i ];
        aPoly.SetFlags( i, (PolyFlags)( rXPoly.GetFlags( i ) ) );
	}
	return aPoly;
}

PolyPolygon XOutCreatePolyPolygonBezier( const XPolyPolygon& rXPolyPoly, OutputDevice* pOut )
{
	PolyPolygon aPolyPoly;

	USHORT	nCount = rXPolyPoly.Count(), i;

	for( i = 0; i < nCount; i++ )
	{
		if( rXPolyPoly[i].GetPointCount() > 0 )
			aPolyPoly.Insert(XOutCreatePolygonBezier(rXPolyPoly[i], pOut));
	}

	return aPolyPoly;
}

/*************************************************************************
|*
|*    Parameterklassse fuer dicke Linien mit Linienmuster
|*
\************************************************************************/

class XLineParam
{
 public:
	long    nPatSeg;
	long    nPatRemain;
	BOOL    bHasJoin, bUseJoin3;
	Point   aJoin1, aJoin2, aJoin3;
	double  fLength;
	long    nLineDx, nLineDy;
	long    nDxW, nDyW;

	XLineParam() {}

	void Init(const Point& rP1, const Point& rP2, long nWidth);
};

void XLineParam::Init(const Point& rP1, const Point& rP2, long nWidth)
{
	double fWidth, fRound;

	nPatSeg = -1;
	nPatRemain = 0;
	bHasJoin = FALSE;
	bUseJoin3 = FALSE;
	nLineDx = rP2.X() - rP1.X();
	nLineDy = rP2.Y() - rP1.Y();
	fLength = sqrt((double) nLineDx * nLineDx + (double) nLineDy * nLineDy);
	if ( fLength > 0 )
		fWidth = nWidth / fLength;
	if ( nLineDy >= 0 ) fRound =  0.5;
	else                fRound = -0.5;
	nDxW =   (long) (fWidth * nLineDy + fRound);
	if ( nLineDx >= 0 ) fRound =  0.5;
	else                fRound = -0.5;
	nDyW = - (long) (fWidth * nLineDx + fRound);

	aJoin1.X() = rP2.X() + nDxW / 2;
	aJoin1.Y() = rP2.Y() + nDyW / 2;
	aJoin2 = aJoin1;
	aJoin2.X() -= nDxW;
	aJoin2.Y() -= nDyW;
	aJoin3 = aJoin1;
}

/*************************************************************************
|*
|*    XOutputDevice::CalcFatLineJoin()
|*
|*    Beschreibung      Uebergang zwischen zwei Linien eines Polygons
|*                      berechnen
|*    Ersterstellung    02.12.94
|*    Letzte Aenderung  09.01.95
|*
\************************************************************************/

void XOutputDevice::CalcFatLineJoin(const Point& rEnd, const Point& rNext,
									XLineParam& rParam)
{
	long nNextDx = rNext.X() - rEnd.X();
	long nNextDy = rNext.Y() - rEnd.Y();

	if ( nNextDx || nNextDy )
	{
		double fRound;
		long nJoinDx, nJoinDy;
		BOOL bFlatJoin = FALSE;
		BOOL bResetJoin = FALSE;

		double fNextLen = sqrt((double) nNextDx * nNextDx +
							   (double) nNextDy * nNextDy);

		// Berechnung des Anschlussuebergangs durch Auswertung
		// der Seitenverhaeltnisse in den Uebergangsdreiecken
		long nPrevDxW = rParam.nDxW;
		long nPrevDyW = rParam.nDyW;

		double fWidth = (double) nLineWidth / fNextLen;
		if ( nNextDy >= 0 ) fRound =  0.5;
		else                fRound = -0.5;
		long nNextDxW =   (long) (fWidth * nNextDy + fRound);
		if ( nNextDx >= 0 ) fRound =  0.5;
		else                fRound = -0.5;
		long nNextDyW = - (long) (fWidth * nNextDx + fRound);
		long nDxA = nNextDxW - nPrevDxW;
		long nDyA = nNextDyW - nPrevDyW;
		long nDxU = nPrevDxW + nNextDxW;
		long nDyU = nPrevDyW + nNextDyW;
		double fJoin = nDxA * nDxA + nDyA * nDyA;
		double fULen = nDxU * nDxU + nDyU * nDyU;

		if ( fULen > 0 )    fJoin = sqrt(fJoin / fULen) / 2;
		else                fJoin = 0;

		if ( fJoin > 0.7 )
		{
			double fJoinLen = fJoin * nLineWidth;
			double fLen = rParam.fLength < fNextLen ? rParam.fLength : fNextLen;

			if ( fJoinLen > fLen )
			{
				fJoin = 0;
				if ( fLen == rParam.fLength )
					bResetJoin = TRUE;
			}
			bFlatJoin = TRUE;
		}

		if ( nPrevDyW >= 0 )    fRound =  0.5;
		else                    fRound = -0.5;
		nJoinDx = -(long) (fJoin * nPrevDyW + fRound);

		if ( nPrevDxW >= 0 )    fRound =  0.5;
		else                    fRound = -0.5;
		nJoinDy = (long) (fJoin * nPrevDxW + fRound);

		// mit Vektorprodukt feststellen, ob Anschlusslinie links oder rechts
		// von der ersten Linie verlaeuft; < 0 -> links
		BOOL bLeftTurn =( (rParam.nLineDx * nNextDy -
						   rParam.nLineDy * nNextDx) < 0 );

		if ( bLeftTurn )
		{
			nJoinDx = - nJoinDx;
			nJoinDy = - nJoinDy;
		}
		rParam.bUseJoin3 = FALSE;

		if ( bFlatJoin )
		{
			if ( bLeftTurn )
			{
				rParam.aJoin3 = rParam.aJoin1;
				rParam.aJoin1 = rParam.aJoin2;
				if ( bResetJoin )
					rParam.aJoin3 = rParam.aJoin2;
				else
				{
					rParam.aJoin3.X() += nJoinDx;
					rParam.aJoin3.Y() += nJoinDy;
				}
				rParam.aJoin2.X() = rEnd.X() - nNextDxW / 2;
				rParam.aJoin2.Y() = rEnd.Y() - nNextDyW / 2;
				rParam.bUseJoin3 = TRUE;
			}
			else
			{
				if ( bResetJoin )
					rParam.aJoin2 = rParam.aJoin1;
				else
				{
					rParam.aJoin2.X() -= nJoinDx;
					rParam.aJoin2.Y() -= nJoinDy;
				}
				rParam.aJoin3.X() = rEnd.X() + nNextDxW / 2;
				rParam.aJoin3.Y() = rEnd.Y() + nNextDyW / 2;
			}
		}
		else
		{
			rParam.aJoin1.X() += nJoinDx;
			rParam.aJoin1.Y() += nJoinDy;
			rParam.aJoin2.X() -= nJoinDx;
			rParam.aJoin2.Y() -= nJoinDy;
			rParam.aJoin3 = rParam.aJoin1;
		}
		rParam.bHasJoin = TRUE;

		rParam.fLength = fNextLen;
		rParam.nLineDx = nNextDx;
		rParam.nLineDy = nNextDy;
		rParam.nDxW = nNextDxW;
		rParam.nDyW = nNextDyW;
	}
}

/*************************************************************************
|*
|*    XOutputDevice::DrawStartEndPoly()
|*
|*    Linienanfang- bzw. -endpolygon zeichnen
|*    Ersterstellung    17.01.95 ESO
|*    Letzte Aenderung  18.01.95 ESO
|*
\************************************************************************/

void XOutputDevice::DrawStartEndPoly(const Point& rPos,
									 const XPolygon& rXPoly,
									 const XLineParam& rParam)
{
	XPolygon    aXPoly(rXPoly);
	Polygon     aPoly;


	if ( rParam.fLength )
		aXPoly.Rotate(Point(0,0), (double) rParam.nLineDx / rParam.fLength,
								  (double) rParam.nLineDy / rParam.fLength);
	aXPoly.Translate(rPos);
	aPoly = XOutCreatePolygon(aXPoly, pOut);
	pOut->DrawPolygon(aPoly);
}

/*************************************************************************
|*
|*    XOutputDevice::DrawLineStartEnd()
|*
|*    Linienanfang bzw. -ende eines Polygons zeichnen
|*    Ersterstellung    18.01.95 ESO
|*    Letzte Aenderung  20.01.95 ESO
|*
\************************************************************************/

void XOutputDevice::DrawLineStartEnd(const Polygon& rPoly)
{
	Point   aDiff;
	USHORT  nPntMax = rPoly.GetSize() - 1;
	USHORT  i = 0;

	if ( bLineStart || bLineEnd )
	{   // Linien mit Laenge 0 nicht beruecksichtigen
		while ( i < nPntMax )
		{
			aDiff = rPoly[i+1] - rPoly[i];
			if ( aDiff.X() || aDiff.Y() )
				break;
			i++;
		}
		while ( nPntMax > i )
		{
			aDiff = rPoly[nPntMax] - rPoly[0];
			if ( aDiff.X() || aDiff.Y() )
				break;
			nPntMax--;
		}
		if ( i < nPntMax )
		{
			XLineParam  aLineParam;
			const Color	aLineColor( pOut->GetLineColor() );
			const Color	aFillColor( pOut->GetFillColor() );

			pOut->SetLineColor();
			pOut->SetFillColor( aLineColor );

			if( bLineStart )
			{
				aLineParam.Init(rPoly[i], rPoly[i+1], 1);
				DrawStartEndPoly(rPoly[i], aLineStartPoly, aLineParam);
			}
			if ( bLineEnd )
			{
				aLineParam.Init(rPoly[nPntMax], rPoly[nPntMax-1], 1);
				DrawStartEndPoly(rPoly[nPntMax], aLineEndPoly, aLineParam);
			}
			
			pOut->SetFillColor( aFillColor );
			pOut->SetLineColor( aLineColor );
		}
	}
}

/*************************************************************************
|*
|*    XOutputDevice::DrawFatLine()
|*
|*    Beschreibung      Dicke Linie mit oder ohne Linienstile zeichnen
|*    Ersterstellung    28.11.94
|*    Letzte Aenderung  09.10.95
|*
\************************************************************************/

void XOutputDevice::DrawFatLine(const Point& rStart, const Point& rEnd,
							const Point* pNext, XLineParam& rParam)
{
	Polygon aPoly(5);
	BOOL    bLineComplete = FALSE;

	long nLineDx = rParam.nLineDx;
	long nLineDy = rParam.nLineDy;
	double fLength = rParam.fLength;

	long nDxW = rParam.nDxW;
	long nDyW = rParam.nDyW;

	double  fDx = 0, fDy = 0;
	long    nDx, nDy;
	long    nSeg = rParam.nPatSeg;
	long    nPattern;

	// Linienmuster vorhanden?
	if ( !pLinePattern )
	{
		nPattern = -1;
		nSeg = 0;
	}
	else
	{
		// Angefangenes Segment zu Ende zeichnen?
		if ( rParam.nPatRemain )
			nPattern = rParam.nPatRemain;
		else    // sonst naechstes Segment
		{
			nSeg++;
			if ( pLinePattern[nSeg] == 0 )  nSeg = 0;
			nPattern = pLinePattern[nSeg];
		}
	}
	aPoly[0].X() = rStart.X() + nDxW / 2;
	aPoly[0].Y() = rStart.Y() + nDyW / 2;
	aPoly[1].X() = aPoly[0].X() - nDxW;
	aPoly[1].Y() = aPoly[0].Y() - nDyW;
	aPoly[2] = aPoly[1];
	aPoly[3] = aPoly[0];
	aPoly[4] = aPoly[0];
	// An vorheriges Segment anschliessen?
	if ( rParam.bHasJoin )
	{
		aPoly[0] = rParam.aJoin1;
		aPoly[1] = rParam.aJoin2;
		aPoly[4] = rParam.aJoin3;
	}
	// Punkte zum Testen auf erreichen des Linienendes
	Point SegStart = rStart;
	Point SegEnd   = rStart;

	// Anschlusspunkte zunaechst im rechten Winkel an das Linienende
	rParam.aJoin1.X() = rEnd.X() + nDxW / 2;
	rParam.aJoin1.Y() = rEnd.Y() + nDyW / 2;
	rParam.aJoin2 = rParam.aJoin1;
	rParam.aJoin2.X() -= nDxW;
	rParam.aJoin2.Y() -= nDyW;
	rParam.bHasJoin = FALSE;
	rParam.bUseJoin3 = FALSE;

	// Anschlusslinie vorhanden?
	if ( pNext )
		CalcFatLineJoin(rEnd, *pNext, rParam);

	while ( !bLineComplete )
	{
		double fSegLength;
		// nPattern < 0: durchgehende Linie
		if ( nPattern < 0 ) fSegLength = 1.0;
		else                fSegLength = (double) nPattern / fLength;
		fDx += fSegLength * nLineDx;
		fDy += fSegLength * nLineDy;
		nDx = (long) fDx;
		nDy = (long) fDy;
		fDx -= nDx;     // Rundungsfehler ausgleichen
		fDy -= nDy;
		aPoly[2].X() += nDx;
		aPoly[2].Y() += nDy;
		aPoly[3].X() += nDx;
		aPoly[3].Y() += nDy;
		SegEnd.X() += nDx;
		SegEnd.Y() += nDy;

		// wenn das rEnde ueberschritten wurde, hat das Vorzeichen
		// der Abstaende vom rEndpunkt gewechselt; durch Xor-Verknuepfung
		// wird dieser Wechsel festgestellt
		long nEndDiffX = (SegEnd.X() - rEnd.X());
		long nEndDiffY = (SegEnd.Y() - rEnd.Y());

		if ( (nEndDiffX ^ (SegStart.X() - rEnd.X())) < 0 ||
			 (nEndDiffY ^ (SegStart.Y() - rEnd.Y())) < 0 ||
			 (!nEndDiffX && !nEndDiffY) )
		{
			if ( nDx || nDy )
			{
				if ( Abs(nDx) >= Abs(nDy) )
				{
					long nDiffX = SegEnd.X() - rEnd.X();
					rParam.nPatRemain = nPattern * nDiffX / nDx;
				}
				else
				{
					long nDiffY = SegEnd.Y() - rEnd.Y();
					rParam.nPatRemain = nPattern * nDiffY / nDy;
				}
			}
			else
				rParam.nPatRemain = 0;

			rParam.nPatSeg = nSeg;
			if ( rParam.bUseJoin3 )
			{
				aPoly[2] = rParam.aJoin1;
				aPoly[3] = rParam.aJoin3;
			}
			else
			{
				aPoly[2] = rParam.aJoin2;
				aPoly[3] = rParam.aJoin1;
			}
			bLineComplete = TRUE;
		}
		if ( !(nSeg & 0x1) )
			pOut->DrawPolygon(aPoly);

		aPoly[0] = aPoly[3];
		aPoly[1] = aPoly[2];
		aPoly[4] = aPoly[0];
		SegStart = SegEnd;

		if ( pLinePattern )
		{
			nSeg++;
			if ( pLinePattern[nSeg] == 0 )
				nSeg = 0;
			nPattern = pLinePattern[nSeg];
		}
	}
}

/*************************************************************************
|*
|*    XOutputDevice::DrawPatternLine()
|*
|*    Beschreibung      Haarlinie mit Linienstil zeichnen
|*    Ersterstellung    13.08.95 ESO
|*    Letzte Aenderung  14.08.95 ESO
|*
\************************************************************************/

void XOutputDevice::DrawPatternLine(const Point& rStart, const Point& rEnd,
									XLineParam& rParam)
{
	Point   aP1, aP2;
	BOOL    bLineComplete = FALSE;

	long    nLineDx = rParam.nLineDx;
	long    nLineDy = rParam.nLineDy;
	double  fLength = rParam.fLength;

	long    nDxW = rParam.nDxW;
	long    nDyW = rParam.nDyW;

	double  fDx = 0, fDy = 0;
	long    nDx, nDy;
	long    nSeg = rParam.nPatSeg;
	long    nPattern;

	// Linienmuster vorhanden?
	if ( !pLinePattern )
	{
		nPattern = -1;
		nSeg = 0;
	}
	else
	{
		// Angefangenes Segment zu Ende zeichnen?
		if ( rParam.nPatRemain )
			nPattern = rParam.nPatRemain;
		else    // sonst naechstes Segment
		{
			nSeg++;
			if ( pLinePattern[nSeg] == 0 )  nSeg = 0;
			nPattern = pLinePattern[nSeg];
		}
	}
	// Punkte fuer DrawLine initialisieren
	aP1 = rStart;
	aP2 = aP1;

	// Punkte zum Testen auf erreichen des Linienendes
	Point SegStart = rStart;
	Point SegEnd   = rStart;

	while ( !bLineComplete )
	{
		double fSegLength;
		// nPattern < 0: durchgehende Linie
		if ( nPattern < 0 ) fSegLength = 1.0;
		else                fSegLength = (double) nPattern / fLength;
		fDx += fSegLength * nLineDx;
		fDy += fSegLength * nLineDy;
		nDx = (long) fDx;
		nDy = (long) fDy;
		fDx -= nDx;     // Rundungsfehler ausgleichen
		fDy -= nDy;
		aP2.X() += nDx;
		aP2.Y() += nDy;
		SegEnd.X() += nDx;
		SegEnd.Y() += nDy;

		// wenn das Ende ueberschritten wurde, hat das Vorzeichen
		// der Abstaende vom Endpunkt gewechselt; durch Xor-Verknuepfung
		// wird dieser Wechsel festgestellt
		long nEndDiffX = (SegEnd.X() - rEnd.X());
		long nEndDiffY = (SegEnd.Y() - rEnd.Y());

		if ( (nEndDiffX ^ (SegStart.X() - rEnd.X())) < 0 ||
			 (nEndDiffY ^ (SegStart.Y() - rEnd.Y())) < 0 ||
			 (!nEndDiffX && !nEndDiffY) )
		{
			if ( nDx || nDy )
			{
				if ( Abs(nDx) >= Abs(nDy) )
				{
					long nDiffX = SegEnd.X() - rEnd.X();
					rParam.nPatRemain = nPattern * nDiffX / nDx;
				}
				else
				{
					long nDiffY = SegEnd.Y() - rEnd.Y();
					rParam.nPatRemain = nPattern * nDiffY / nDy;
				}
			}
			else
				rParam.nPatRemain = 0;

			rParam.nPatSeg = nSeg;
			aP2 = rEnd;
			bLineComplete = TRUE;
		}
		if ( !(nSeg & 0x1) )
		{
			pOut->DrawLine(aP1, aP2);
		}

		aP1 = aP2;
		SegStart = SegEnd;

		if ( pLinePattern )
		{
			nSeg++;
			if ( pLinePattern[nSeg] == 0 )
				nSeg = 0;
			nPattern = pLinePattern[nSeg];
		}
	}
}

/*************************************************************************
|*
|*    XOutputDevice::DrawLinePolygon()
|*
|*    Beschreibung      Polygon-Linie (ggf. mit Linienmuster) zeichnen
|*    Ersterstellung    28.11.94
|*    Letzte Aenderung  09.10.95 ESO
|*
\************************************************************************/

void XOutputDevice::DrawLinePolygon( const Polygon& rPoly, BOOL bClosePoly )
{
	if( nLineTransparence )
	{
		GDIMetaFile		aMtf;
		VirtualDevice	aVDev;
		OutputDevice*	pOldOut = pOut;
		MapMode			aMap( pOldOut->GetMapMode() );
		const BYTE		cTrans = nLineTransparence * 255 / 100;
		const Color		aTrans( cTrans, cTrans, cTrans );
		Gradient		aTransGradient( GRADIENT_LINEAR, aTrans, aTrans );

		pOut = &aVDev;
		aVDev.EnableOutput( FALSE );
		aVDev.SetMapMode( pOldOut->GetMapMode() );
		aMtf.Record( &aVDev );
		aVDev.SetLineColor( pOldOut->GetLineColor() );
		aVDev.SetFillColor( pOldOut->GetFillColor() );
		aVDev.SetFont( pOldOut->GetFont() );
		aVDev.SetDrawMode( pOldOut->GetDrawMode() );
		aVDev.SetRefPoint( pOldOut->GetRefPoint() );
		ImpDrawLinePolygon( rPoly, bClosePoly );
		aMtf.Stop();
		pOut = pOldOut;

		Rectangle aBound;

		for( MetaAction* pAct = aMtf.FirstAction(); pAct; pAct = aMtf.NextAction() )
		{
			if( pAct->GetType() == META_POLYGON_ACTION )
				aBound.Union( ( (MetaPolygonAction*) pAct )->GetPolygon().GetBoundRect() );
			else if( pAct->GetType() == META_POLYLINE_ACTION )
				aBound.Union( ( (MetaPolyLineAction*) pAct )->GetPolygon().GetBoundRect() );
			else if( pAct->GetType() == META_LINE_ACTION )
			{
				const Point aStart( ( (MetaLineAction*) pAct )->GetStartPoint() );
				const Point aEnd( ( (MetaLineAction*) pAct )->GetEndPoint() );
				aBound.Union( Rectangle( aStart, aEnd ) );
			}
		}

		if( aMtf.GetActionCount() )
		{
			Size		aSizeLog( aBound.GetSize() );
			const Size	aMinSizeLog( pOut->PixelToLogic( Size( 1, 1 ) ) );
			const Size	aSizePix( pOut->LogicToPixel( aSizeLog ) );

			// watch for minimum width
			if( !aSizePix.Width() )
				aSizeLog.Width() = aMinSizeLog.Width();

			// watch for minimum width
			if( !aSizePix.Height() )
				aSizeLog.Height() = aMinSizeLog.Height();

			aMap.SetOrigin( aBound.TopLeft() );
			aMtf.SetPrefMapMode( aMap );
			aMtf.SetPrefSize( aBound.GetSize() );
			aTransGradient.SetSteps(3);
			pOut->DrawTransparent( aMtf, aBound.TopLeft(), aSizeLog, aTransGradient );
		}
	}
	else
		ImpDrawLinePolygon( rPoly, bClosePoly );
}

// -----------------------------------------------------------------------------

void XOutputDevice::ImpDrawLinePolygon(const Polygon& rPoly, BOOL bClosePoly)
{
	Polygon         aPoly( rPoly );
	const Point*    pNextPoint;
	Point           aLineStartPos, aLineEndPos;
	XLineParam      aLParam, aStartParam, aEndParam;
	USHORT          nPntMax = aPoly.GetSize() - 1;

	if( nPntMax >= 1 )
	{
		if( bHair || ( ( XLINE_SOLID == eLineStyle ) && ( nLineWidth ==  0 ) ) )
		{
			// #107240#
			// Since this method is also used for XOR drawing it's not allowed
			// to optimize to line drawing here. DrawLine() does not draw the
			// last point, thus a cycle with XOR drawing with DrawPolyLine() and then
			// deleting the last part with DrawLine() results in one point not being deleted.
			// if( 1 == nPntMax )
			//	pOut->DrawLine( rPoly[ 0 ], rPoly[ 1 ] );
			// else
			pOut->DrawPolyLine( rPoly );

			DrawLineStartEnd( rPoly );
		}
		else if( XLINE_NONE != eLineStyle )
		{
			Color		aOldLineColor;
			Color		aOldFillColor;
			Point		aDiff;
			const ULONG	nOldDrawMode = pOut->GetDrawMode();
			USHORT		i = 0;

			if( !nLineWidth )
			{
				aOldLineColor = pOut->GetLineColor();
				pOut->SetLineColor( aLineColor );
			}

			aOldFillColor = pOut->GetFillColor();

			if( nOldDrawMode & DRAWMODE_WHITEFILL )
			{
				ULONG nNewDrawMode = nOldDrawMode;

				nNewDrawMode &= ~DRAWMODE_WHITEFILL;
				nNewDrawMode |= DRAWMODE_BLACKFILL;
				pOut->SetDrawMode( nNewDrawMode );
			}

            if( nOldDrawMode & DRAWMODE_BLACKLINE )
            {
                const Color aBlack( COL_BLACK );

                pOut->SetDrawMode( pOut->GetDrawMode() & (~DRAWMODE_SETTINGSFILL) );
                pOut->SetFillColor( aBlack );
            }
            else if( nOldDrawMode & DRAWMODE_SETTINGSLINE )
            {
                pOut->SetDrawMode( pOut->GetDrawMode() & (~DRAWMODE_SETTINGSFILL) );
				svtools::ColorConfig aColorConfig;
				Color aColor( aColorConfig.GetColorValue( svtools::FONTCOLOR ).nColor );
                pOut->SetFillColor( aColor );
            }
            else
                pOut->SetFillColor( aLineColor );

			// bei einfachen Linien darf das Polygon nicht geschlossen sein (#24000)
			if ( aPoly[ nPntMax ] == aPoly[ 0 ] )
			{
				if ( nPntMax > 2 )
				{
					nPntMax--;
					bClosePoly = TRUE;
				}
				else if ( 2 == nPntMax )
					bClosePoly = FALSE;
			}

			// Linien mit Laenge 0 nicht beruecksichtigen
			while ( i < nPntMax )
			{
				aDiff = aPoly[i+1] - aPoly[0];

				if ( bLineStart && !bClosePoly )
				{
					long nSqLen = aDiff.X() * aDiff.X() + aDiff.Y() * aDiff.Y();
					if ( nSqLen > nLineStartSqLen || i == nPntMax-1 )
					{
						aLineStartPos = aPoly[0];
						aStartParam.Init(aPoly[0], aPoly[i+1], 1);
						double fLen = sqrt((double)nLineStartSqLen);
						if ( aStartParam.fLength )
							fLen /= aStartParam.fLength;
						aPoly[i].X() = aPoly[0].X() + (long) (fLen * aStartParam.nLineDx);
						aPoly[i].Y() = aPoly[0].Y() + (long) (fLen * aStartParam.nLineDy);
						break;
					}
				}
				else if ( aDiff.X() || aDiff.Y() )
					break;
				i++;
			}
			USHORT  nLastPnt = nPntMax;

			while ( nPntMax > i )
			{
				aDiff = aPoly[nPntMax-1] - aPoly[nLastPnt];
				if ( bLineEnd && !bClosePoly )
				{
					long nSqLen = aDiff.X() * aDiff.X() + aDiff.Y() * aDiff.Y();
					if ( nSqLen > nLineEndSqLen || nPntMax == i+1 )
					{
						aLineEndPos = aPoly[nLastPnt];
						aEndParam.Init(aPoly[nLastPnt], aPoly[nPntMax-1], 1);
						double fLen = sqrt((double)nLineEndSqLen);
						if ( aEndParam.fLength )
							fLen /= aEndParam.fLength;
						aPoly[nPntMax].X() = aPoly[nLastPnt].X() + (long) (fLen * aEndParam.nLineDx);
						aPoly[nPntMax].Y() = aPoly[nLastPnt].Y() + (long) (fLen * aEndParam.nLineDy);
						break;
					}
				}
				else if ( aDiff.X() || aDiff.Y() )
					break;
				nPntMax--;
			}

			if ( bClosePoly )
			{
				aDiff = aPoly[nPntMax] - aPoly[i];
				if ( !aDiff.X() && !aDiff.Y() )
					nPntMax--;
				aLParam.Init(aPoly[nPntMax], aPoly[i], nLineWidth);
				if ( nLineWidth > 0 )
					CalcFatLineJoin(aPoly[i], aPoly[i+1], aLParam);
			}
			else
				aLParam.Init(aPoly[i], aPoly[i+1], nLineWidth);

			while ( i < nPntMax )
			{
				USHORT nPos = i + 1;
				while ( nPos < nPntMax )
				{
					aDiff = aPoly[nPos+1] - aPoly[nPos];
					if ( aDiff.X() || aDiff.Y() )
						break;
					nPos++;
				}
				if ( nPos+1 <= nPntMax )    pNextPoint = &aPoly[nPos+1];
				else if ( bClosePoly )      pNextPoint = &aPoly[0];
				else                        pNextPoint = NULL;

				if ( nLineWidth > 0 )
					DrawFatLine(aPoly[i], aPoly[i+1], pNextPoint, aLParam);
				else
				{
					aLParam.nLineDx = aPoly[i+1].X() - aPoly[i].X();
					aLParam.nLineDy = aPoly[i+1].Y() - aPoly[i].Y();
					aLParam.fLength = sqrt((double) aLParam.nLineDx * aLParam.nLineDx +
										   (double) aLParam.nLineDy * aLParam.nLineDy);
					DrawPatternLine(aPoly[i], aPoly[i+1], aLParam);
				}
				i = nPos;
			}
			if ( bClosePoly )
			{
				if ( nLineWidth > 0 )
					DrawFatLine(aPoly[i], aPoly[0], &aPoly[1], aLParam);
				else
				{
					aLParam.nLineDx = aPoly[0].X() - aPoly[i].X();
					aLParam.nLineDy = aPoly[0].Y() - aPoly[i].Y();
					aLParam.fLength = sqrt((double) aLParam.nLineDx * aLParam.nLineDx +
										   (double) aLParam.nLineDy * aLParam.nLineDy);
					DrawPatternLine(aPoly[i], aPoly[0], aLParam);
				}
			}
			else
			{
				if ( bLineStart )
					DrawStartEndPoly(aLineStartPos, aLineStartPoly, aStartParam);
				if ( bLineEnd )
					DrawStartEndPoly(aLineEndPos, aLineEndPoly, aEndParam);
			}

			if( nLineWidth == 0 )
				pOut->SetLineColor( aOldLineColor );

			pOut->SetFillColor( aOldFillColor );
			pOut->SetDrawMode( nOldDrawMode );
		}
	}
}


