/*************************************************************************
 *
 *  $RCSfile: ObjectBarManager.hxx,v $
 *
 *  $Revision: 1.9 $
 *
 *  last change: $Author: rt $ $Date: 2004/11/30 08:27:52 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef SD_OBJECT_BAR_MANAGER_HXX
#define SD_OBJECT_BAR_MANAGER_HXX

#include "ShellFactory.hxx"

#ifndef _SOLAR_H
#include <tools/solar.h>
#endif
#include <osl/mutex.hxx>
#include <rtl/ustring.hxx>

#include <memory>
#include <vector>

class SfxShell;
class SdrView;

namespace sd {

class ViewShellBase;
class ViewShellManager;

/** Manage the object bars and tool bars of one view shell.  This includes
    lifetime control and changes of sub shell stack.
*/
class ObjectBarManager
{
public:
    typedef ShellFactory<SfxShell> ObjectBarFactory;

    ObjectBarManager (ViewShell& rViewShell);
    ~ObjectBarManager (void);

    /** Register the default factory that is called to create a new instance
        of a shell for a given id when there is no factory that has been
        registered specially for that id.
        @param rFactory
            The factory object that is called to create a new shell instance.
        @param pUserData
            This pointer is passed to the factory function.
    */
    void RegisterDefaultFactory (
        ::std::auto_ptr<ObjectBarFactory> rFactory);

    /** Register a factory that is called to create a shell for the
        specified id.  This takes precedence over the default factory.
    */
    void RegisterFactory (
        ShellId nId, 
        ::std::auto_ptr<ObjectBarFactory> rFactory);

    /** Enable the switching of object bar.  You are encouraged to use the
        DisableObjectBarSwitching() method to disable the object bar
        switching.
        @param bEnable
            This flag decides whether switching object bars is allowed
            (<TRUE/>) or disabled (<FALSE/>).
    */
    void EnableObjectBarSwitching (bool bEnable = true);
    void DisableObjectBarSwitching (void);

    /** Remove all object bars.  This call is ignored when the switching of
        object shells is currently disabled.
    */
    void Clear (void);

    /** The id of the default object bar is stored in ObjectBarManager
        objects so that classes like functions to know about it.  Some
        functions have their own object bars which they turn on or off.
        When turning them off they have to replace it with the default
        object bar and with this method we do not have to hard code the
        default object bar id into the function's code.  
 
        Setting the default object bar id does not change the shell stack.
        You have to call SwitchObjectBar() to activate the default object
        bar.
    */
    void SetDefaultObjectBarId (ShellId nId);

    /** Return the id of the default object bar. See description of
        SetDefaultObjectBarId() for more explanation.
    */
    ShellId GetDefaultObjectBarId (void);

    SfxShell* ActivateObjectBar (ShellId nId);
    
    void DeactivateObjectBar (SfxShell* pObjectBar);
    
    /** Replace the currently active object bar with the specified object
        bar.  This call is ignored when the switching of object shells is
        currently disabled or the requested object bar is already active.
    */
    void SwitchObjectBar (ShellId nId);

    /** Get top-most object bar that has been activated under the given id.
        @return
            Returns the requested object bar or NULL when no such
            object bar is currently active.
    */
    SfxShell* GetObjectBar (ShellId nId) const;

    /** Return the id of the given active object bar.
        @return
            When the given object bar is NULL or is not one of the active
            object bars managed by this class then snInvalidShellId==-1 is
            returned.
    */
    ShellId GetObjectBarId (const SfxShell* pObjectBar) const;

    /** Return the id of the top most object bar on the stack of active
        object bars.
        @return
            Returns snInvalidShellId==-1 when the stack of active object
            bars is empty.
    */
    ShellId GetTopObjectBarId (void) const;

    /** Return the top most object bar on the stack of active object bars.
        @return
            Returns NULL when the stack is empty.
    */
    SfxShell* GetTopObjectBar (void) const;

    /** Send an Invalidate() call to all active object bars.
    */
    void InvalidateAllObjectBars (void) const;

    /** Get a shell list by appending the active object bars to the end of
        the given list of shells.  This is restricted to the object bars
        that are placed BELOW the sub-shell.  This method is typically
        called to gather all shells that are to be taken from or pushed on
        the stack of sub-shells.
    */
    void GetLowerShellList (::std::vector<SfxShell*>& rShellList) const;

    /** Get a shell list by appending the active object bars to the end of
        the given list of shells.  This is restricted to the object bars
        that are placed ABOVE the sub-shell.  This method is typically
        called to gather all shells that are to be taken from or pushed on
        the stack of sub-shells.
    */
    void GetUpperShellList (::std::vector<SfxShell*>& rShellList) const;

    /** Move the specified object bar to the top of the stack, i.e. give it
        priority to handle slots over all other active shells.
        @param nId
            Id of the object bar to move to the top of the stack.  If there
            is no active object bar with this id the call is silently ignored.
    */
    void MoveToTop (USHORT nId);

    /** Move the specified object bar to the bottom of the stack, i.e. it
        will handle a slot only if no other active object bar handles that
        slot.
        @param nId
            Id of the object bar to move to the bottom of the stack.  If there
            is no active object bar with this id the call is silently ignored.
    */
    void MoveToBottom (USHORT nId);

    /** Move the specified object bar below all other object bars and below
        the shell as well, i.e. it will handle a slot only if no other
        active object bar nor the shell handles that slot.
        @param nId
            Id of the object bar to move below the shell.  If there is no
            active object bar with this id the call is silently ignored.
    */
    void MoveBelowShell (USHORT nId);

    /** This is a more high-level method that looks at the context, i.e. the
        type of the currently selected shape, and activates the
        corresponding tool bar.  The name reflects the fact that the context
        may change when the selection changes.
        @param pView
            The context (obtained by calling GetContext()) of this view is
            used to determine which obect bar to show.
    */
    void SelectionHasChanged (SdrView* pView);

    /** Show the tool bar with the given name.
        @param rName
            The toolbar name is the name of the associated XML file.
            Provide the name without the private:resource/toolbar prefix.
    */
    void ShowToolBar (const ::rtl::OUString& rName);
    
    /** Hide the tool bar with the given name.
        @param rName
            The toolbar name is the name of the associated XML file.
            Provide the name without the private:resource/toolbar prefix.
    */
    void HideToolBar (const ::rtl::OUString& rName);

private:
    class Implementation;
    ::std::auto_ptr<Implementation> mpImpl;
};

} // end of namespace sd

#endif
