/*************************************************************************
 *
 *  $RCSfile: PaneManager.hxx,v $
 *
 *  $Revision: 1.7 $
 *
 *  last change: $Author: rt $ $Date: 2005/02/04 14:17:57 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef SD_PANE_MANAGER_HXX
#define SD_PANE_MANAGER_HXX

#include "ViewShell.hxx"

namespace sd {


/** Manage the windows and view shells of the panes of the Multi Pane GUI.
    The pane manager controlls (after requests from the outside) which view
    shells are displayed in which windows.  It furthermore controlls the
    visibility of the docking windows.

    Callers from the outside should generally call only
    RequestMainViewShell() and late the pane manager decide what to do with
    the side panes.
*/
class PaneManager
{
public:
    /** The PaneType is used to specifiy where to show a view shell.  The
        left, center, and right pane are specified by PT_LEFT, PT_CENTER,
        and PT_RIGHT respectively.  PT_NONE is used e.g. to denote that a
        view shell is not shown in any pane.  PT_MAIN is used internally and
        is different to PT_CENTER in that an operation operating on PT_MAIN
        may change the other panes as well.
    */
    enum PaneType { PT_CENTER, PT_LEFT, PT_RIGHT, PT_NONE, PT_MAIN };

    /** The two possible call modes used by RequestViewShellChange().  Using 
        the SFX values SFX_CALLMODE_ASYNCHRON and SFX_CALLMODE_SYNCHRON
        directly would make the call less safe because other values could be
        passes too easily.
    */
    enum CallMode { CM_SYNCHRONOUS, CM_ASYNCHRONOUS };

    /** Create a new pane manager that manages the panes for the given
        ViewShellBase object.  The type of the default main view shell is
        stored.  The next call to LateInit() will then create the associated
        view shell and place it in the center pane.
    */
    PaneManager (
        ViewShellBase& rBase,
        ViewShell::ShellType eDefaultMainViewShellType);

    /** The destructor calls Shutdown() to make sure that the panes and
        their view shells are closed.  Usually Shutdown() is called
        earlier.
    */
    ~PaneManager (void);

    /** The late initialization has to be called directly after a new pane
        manager has been created.
    */
    void LateInit (void);

    void InitPanes (void);

    /** Shutdown the pane manager.  Its methods can still be called but they
        will not do anything.  This method is typically be called from the
        destructor of ViewShellBase.  Its use is somewhat similar to the
        late initialization.
        @param bForgetOnly
            When <TRUE/> is given then the view shells and windows managed
            by the PaneManager for the various panes are simply forgotten.
            In this case the shells have to be shutdown at another place,
            ViewShellManager::Shutdown() for example.
            When the default <FALSE/> is given then the panes are properly
            shut down.
    */
    void Shutdown (bool bForgetOnly = false);

    /** Return the type of the current view shell that is displayed in the
        specified pane.
    */
    ViewShell::ShellType GetViewShellType (PaneType ePane = PT_CENTER) const;

    ViewShell* GetViewShell (PaneType ePane = PT_CENTER) const;

    /** Return the parent window for the specified pane.
        @return
            When there has not been set a window for the specified pane then
            the returned value is <NULL/>.  This usually is the case while the
            child windows associated with the left (and other non-center)
            pane has not yet been created.
            When an unknown pane has been specified then <NULL/> is returned
            as well.
    */
    ::Window* GetWindow (PaneType ePane = PT_CENTER) const;

    /** Return the title of the window displayed in the specfied pane.
        This is important primarily for the side panes as the center pane
        has to visible title bar.  A new docking window will ask the pane
        manager for a title.  It is thus possible to given the left pane
        different titles for Draw and Impress.
    */
    String GetWindowTitle (PaneType ePane) const;

    /** Return the pane in which the given view shell is displayed.
        @return
            When the given view shell is not displayed in any pane then
            PT_NONE is returned.
    */
    PaneType GetPaneOfViewShell (ViewShell* pViewShell) const;

    /** Register a new parent window with the specified pane.
        @param ePane
            The pane in which the window will show a view shell.
        @param pWindow
            Pass <NULL/> to indicate that the specified pane must not be
            used.
        @param eCallMode
            This flag specifies whether the slot call is made synchronously
            or asynchronously.  Use CM_SYNCHRONOUS only when you know what
            you are doing.  This is usually only save when called from
            an Execute() method (when it is called from SFX.)
    */
    void SetWindow (
        PaneType ePane, 
        ::Window* pWindow,
        CallMode eCallMode = CM_ASYNCHRONOUS);

    /** Handle slot calls for the slots SID_DRAWINGMODE, SID_NOTESMODE,
        SID_HANDOUTMODE, SID_DIAMODE, and SID_OUTLINEMODE.  As a result
        RequestViewShellChange is called with the proper shell type.
    */
    void HandleModeChangeSlot (ULONG nSlotId, SfxRequest& rRequest);

    /** Request a change of the main sub-shell.  This is the central method
        for switching view shells.  Switching the shell in the center pane
        is executed synchronously or asynchronously depending on the
        bAsynchronous flag.  Sub-shells in other panes are switched
        independently of that flag.  It is determined internally whether to
        do it synchronously or asynchronously.
        @param eType
            If the given type is the same as that of the current main
            sub-shell then this call is silently ignored.
        @param eCallMode
            This flag specifies whether the slot call is made synchronously
            or asynchronously.
    */
    bool RequestMainViewShellChange (
        ViewShell::ShellType eType,
        CallMode eCallMode = CM_ASYNCHRONOUS);

    /** Callback function for slots related to changing the view or
        edit mode.
    */
    void ExecuteModeChange (SfxRequest& rRequest);

    void ExecuteSlot (SfxRequest& rRequest);
    void GetSlotState (SfxItemSet& rSet);

    /** Add a listener that is informed of events concerning the
        ViewShellBase object that is called.  Multiple calls with the same
        listener result in a single registration.  For a description of
        possible events have a look at the EventId enum.
        @param rEventListener
            The event listener to call for future events.  tCall
            RemoveEventListener() before the listener is destroyed.
    */
    void AddEventListener (const Link& rEventListener);

    /** Remove the given listener from the list of listeners to call for
        future events.
        @param rEventListener
            After this method returns the given listener is not called back
            from this ViewShellBase object.  Passing a listener that has not
            been registered before is safe and is silently ignored.
    */
    void RemoveEventListener (const Link& rEventListener);

private:
	osl::Mutex maMutex;

    /** Collection of member variables and functions that do not have to be
        publicly visible and that may change in the future.
    */
    class Implementation;
    ::std::auto_ptr<Implementation> mpImpl;

    /** Calls are forwarded to the implementation object only when this flag
        is set to <TRUE/>.  It is set to <FALSE/> when Shutdown() is called.
    */
    bool mbValid;
};




/** Objects of this class are sent to listeners to notify them about when
    view shells are changed in a pane.
*/
class PaneManagerEvent
{
public:
    /** The events notified by this class are:
        - EID_VIEW_SHELL_REMOVED   A view shell will be removed in the near
        future.
        - EID_VIEW_SHELL_ADDED     A new view shell has been created and
        placed in a pane.
        - EID_PANE_MANAGER_DYING   The pane manager is about to be
        destroyed.  It should not be accessed after this event.
         
    */
    enum EventId { EID_VIEW_SHELL_REMOVED, 
                   EID_VIEW_SHELL_ADDED, 
                   EID_PANE_MANAGER_DYING };

    PaneManagerEvent (
        const ViewShellBase& rBase,
        EventId eEventId,
        PaneManager::PaneType ePane,
        ViewShell* pShell);
    const ViewShellBase& mrBase;
    EventId meEventId;
    PaneManager::PaneType mePane;
    ViewShell* mpShell;
};

} // end of namespace sd

#endif
