/*************************************************************************
 *
 *  $RCSfile: slide.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: kz $ $Date: 2005/01/21 16:56:32 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <canvas/debug.hxx>
#include <slide.hxx>

#ifndef _COM_SUN_STAR_AWT_SYSTEMPOINTER_HPP_ 
#include <com/sun/star/awt/SystemPointer.hpp>
#endif
#ifndef _COM_SUN_STAR_CONTAINER_XINDEXACCESS_HPP_
#include <com/sun/star/container/XIndexAccess.hpp>
#endif
#ifndef _COM_SUN_STAR_DRAWING_XMASTERPAGETARGET_HPP_
#include <com/sun/star/drawing/XMasterPageTarget.hpp>
#endif
#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSET_HPP_
#include <com/sun/star/beans/XPropertySet.hpp>
#endif
#ifndef _COM_SUN_STAR_CONTAINER_XENUMERATIONACCESS_HPP_
#include <com/sun/star/container/XEnumerationAccess.hpp>
#endif
#ifndef _COM_SUN_STAR_AWT_RECTANGLE_HPP_
#include <com/sun/star/awt/Rectangle.hpp>
#endif

#ifndef _COM_SUN_STAR_PRESENTATION_PARAGRAPHTARGET_HPP_
#include <com/sun/star/presentation/ParagraphTarget.hpp>
#endif
#ifndef _COM_SUN_STAR_PRESENTATION_EFFECTNODETYPE_HPP_
#include <com/sun/star/presentation/EffectNodeType.hpp>
#endif
#ifndef _COM_SUN_STAR_ANIMATIONS_XANIMATIONNODESUPPLIER_HPP_
#include <com/sun/star/animations/XAnimationNodeSupplier.hpp>
#endif
#ifndef _COM_SUN_STAR_ANIMATIONS_XTARGETPROPERTIESCREATOR_HPP_
#include <com/sun/star/animations/XTargetPropertiesCreator.hpp>
#endif

#ifndef _CPPCANVAS_BASEGFXFACTORY_HXX
#include <cppcanvas/basegfxfactory.hxx>
#endif

#ifndef _CANVAS_CANVASTOOLS_HXX
#include <canvas/canvastools.hxx>
#endif

#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYGON_HXX
#include <basegfx/polygon/b2dpolygon.hxx>
#endif
#ifndef _BGFX_NUMERIC_FTOOLS_HXX
#include <basegfx/numeric/ftools.hxx>
#endif

#ifndef INCLUDED_ANIMATIONS_ANIMATIONNODEHELPER_HXX
#include <animations/animationnodehelper.hxx>
#endif

#ifndef _SV_GDIMTF_HXX 
#include <vcl/gdimtf.hxx>
#endif
#ifndef _SV_METAACT_HXX 
#include <vcl/metaact.hxx>
#endif

#ifndef BOOST_BIND_HPP_INCLUDED
#include <boost/bind.hpp>
#endif

#include <iterator>
#include <algorithm>
#include <functional>

#include <intrinsicanimation.hxx>
#include <doctreenode.hxx>
#include <backgroundshape.hxx>
#include <shapeimporter.hxx>
#include <slideshowexceptions.hxx>
#include <eventqueue.hxx>
#include <activitiesqueue.hxx>
#include <usereventqueue.hxx>
#include <event.hxx>
#include <tools.hxx>


using namespace ::drafts::com::sun::star;
using namespace ::com::sun::star;

// -----------------------------------------------------------------------------

namespace presentation
{
    namespace internal
    {

        Slide::Slide( const uno::Reference< drawing::XDrawPage >&			xDrawPage,
					  const uno::Reference< animations::XAnimationNode >& 	xRootNode,
                      sal_Int32												nSlideIndex,
                      EventQueue&											rEventQueue,
                      ActivitiesQueue&										rActivitiesQueue,
                      EventMultiplexer&										rEventMultiplexer,
                      UserEventQueue&										rUserEventQueue,
                      const uno::Reference< uno::XComponentContext >& 		xComponentContext ) :
            mxDrawPage( xDrawPage ),
			mxRootNode( xRootNode ),
            maContext( LayerManagerSharedPtr(
                           new LayerManager( getPageRect() ) ),
                       rEventQueue, 
                       rEventMultiplexer,
                       rActivitiesQueue, 
                       rUserEventQueue,
                       xComponentContext ),
            mpEventBroadcaster( new ShapeEventBroadcaster( rEventMultiplexer ) ),
            maAnimations( maContext ),
            maViews(),
            maUserPaintColor(),
            mpPaintOverlay(),
            maSlideBitmaps(),
            mnNextSlideTimeout( 0.0 ),
            meAnimationState( CONSTRUCTING_STATE ),
            mbImageAnimationsAllowed( true ),
            mbShapesLoaded( false ),
            mbShowLoaded( false ),
            mbHaveAnimations( false ),
            mbMainSequenceFound( false ),
            mbHasAutomaticNextSlide( false )
        {
            ENSURE_AND_THROW( maContext.mpLayerManager.get(),
                              "Slide::Slide(): Could not create layer manager" );
            ENSURE_AND_THROW( mpEventBroadcaster.get(),
                              "Slide::Slide(): Could not create event broadcaster" );

            mpEventBroadcaster->setSlideIndex( nSlideIndex );

            // retrieve slide change parameters from XDrawPage
            // ===============================================

            uno::Reference< beans::XPropertySet > xPropSet( xDrawPage,
                                                            uno::UNO_QUERY_THROW );
            
            sal_Int32 nChange;
            if( !(xPropSet->getPropertyValue( 
                      ::rtl::OUString( 
                          RTL_CONSTASCII_USTRINGPARAM("Change") ) )
                  >>= nChange) )
            {
                ENSURE_AND_THROW(
                    false,
                    "Slide::Slide(): "
                    "Could not extract slide change mode from XDrawPage" );
            }
            
            mbHasAutomaticNextSlide = nChange == 1;
            
            if( !(xPropSet->getPropertyValue( 
                      ::rtl::OUString( 
                          RTL_CONSTASCII_USTRINGPARAM("Duration") ) )
                  >>= mnNextSlideTimeout) )
            {
                ENSURE_AND_THROW(
                    false,
                    "Slide::Slide(): "
                    "Could not extract slide transition timeout from "
                    "XDrawPage" );
            }
        }

        Slide::~Slide()
        {
        }
        
        void Slide::addView( const UnoViewSharedPtr& rView )
        {
            ENSURE_AND_THROW( maContext.mpLayerManager.get(),
                              "Slide::addView(): Invalid layer manager" );

            // forward to layer manager
            maContext.mpLayerManager->addView( rView );

            // forward to user paint overlay
            if( mpPaintOverlay.get() )
                mpPaintOverlay->addView( rView );

            // add locally
            maViews.push_back( rView );
            maSlideBitmaps.push_back( 
                VectorOfSlideBitmaps::value_type(SlideAnimationState_NUM_ENTRIES) );
        }

        bool Slide::removeView( const UnoViewSharedPtr& rView )
        {
            ENSURE_AND_RETURN( maContext.mpLayerManager.get(),
                               "Slide::removeView(): Invalid layer manager" );

            // remove from layer manager
            maContext.mpLayerManager->removeView( rView );

            // forward to user paint overlay
            if( mpPaintOverlay.get() )
                mpPaintOverlay->removeView( rView );

            // remove locally
            const UnoViewVector::iterator aBegin( maViews.begin() );
            const UnoViewVector::iterator aEnd  ( maViews.end() );
            UnoViewVector::iterator aIter;
            if( (aIter=::std::find( aBegin,
                                    aEnd, 
                                    rView)) == aEnd )
            {
                // view seemingly was not added, failed
                return false;
            }

            // actually erase from container
            maViews.erase( aIter );

            const UnoViewVector::difference_type offset(
                ::std::distance( aBegin, aIter ) );
            VectorOfSlideBitmaps::iterator aSlideBitmapIter( maSlideBitmaps.begin() );
            ::std::advance( aSlideBitmapIter, offset );
            maSlideBitmaps.erase( aSlideBitmapIter );

            OSL_ENSURE( ::std::find( maViews.begin(),
                                     maViews.end(),
                                     rView ) == maViews.end(),
                        "Slide::removeView(): View was seemingly added twice, "
                        "this breaks maSlideBitmap synchronicity" );

            return true;
        }

        void Slide::addShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >& xListener, 
                                           const uno::Reference< ::com::sun::star::drawing::XShape >& 					xShape )
        {
            if( !prefetchShapes() )
                ENSURE_AND_THROW( false,
                                  "Slide::addShapeEventListener(): Could not prefetch shapes" );

            ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

            if( pShape.get() )
                mpEventBroadcaster->addShapeEventListener( xListener, xShape, pShape );
        }

        void Slide::removeShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >& 	xListener, 
                                              const uno::Reference< ::com::sun::star::drawing::XShape >& 					xShape )
        {
            if( !prefetchShapes() )
                ENSURE_AND_THROW( false,
                                  "Slide::removeShapeEventListener(): Could not prefetch shapes" );

            ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

            if( pShape.get() )
                mpEventBroadcaster->removeShapeEventListener( xListener, xShape, pShape );
        }

        void Slide::setImageAnimationsAllowed( bool bImageAnimationsAllowed )
        {
            if( mbImageAnimationsAllowed != bImageAnimationsAllowed )
            {
                mbImageAnimationsAllowed = bImageAnimationsAllowed;

                // if animations switched off - immediately disable
                // them. If switched on - immediately enable them
                if( bImageAnimationsAllowed == false )
                    endIntrinsicAnimations();
                else
                    startIntrinsicAnimations();
            }
        }

        void Slide::setShapeCursor( const uno::Reference< ::com::sun::star::drawing::XShape >&	xShape, 
                                    sal_Int16 													nPointerShape )
        {
            if( !prefetchShapes() )
                ENSURE_AND_THROW( false,
                                  "Slide::Slide::setShapeCursor(): Could not prefetch shapes" );

            ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

            if( pShape.get() )
                mpEventBroadcaster->setShapeCursor( pShape, nPointerShape );
        }

        bool Slide::paint()
        {
            ENSURE_AND_RETURN( maContext.mpLayerManager.get(), 
                               "Slide::paint(): Invalid layer manager" );

            if( !prefetchShapes() )
                return false;

            if( !maContext.mpLayerManager->render() )
                return false;

            // update screen, to make changes visible
            ::std::for_each( maViews.begin(),
                             maViews.end(),
                             ::boost::mem_fn( &View::updateScreen ) );

            return true;
        }

        bool Slide::prefetchShow()
        {
            return applyInitialShapeAttributes( mxRootNode );
        }

        namespace
        {
            class SlideRenderer
            {
            public:
                SlideRenderer( Slide& rSlide ) :
                    mrSlide( rSlide )
                {
                }

                void operator()( const UnoViewSharedPtr& rView )
                {
                    SlideBitmapSharedPtr 		 pBitmap( mrSlide.getCurrentSlideBitmap( rView ) );
                    ::cppcanvas::CanvasSharedPtr pCanvas( rView->getCanvas() );

                    const ::basegfx::B2DHomMatrix 	aViewTransform( pCanvas->getTransformation() );
                    const ::basegfx::B2DPoint 		aOutPosPixel( aViewTransform * ::basegfx::B2DPoint() );

                    // setup a canvas with device coordinate space,
                    // the slide bitmap already has the correct
                    // dimension.
                    ::cppcanvas::CanvasSharedPtr pDevicePixelCanvas( pCanvas->clone() );
                    pDevicePixelCanvas->setTransformation( ::basegfx::B2DHomMatrix() );

                    // clear background before (to generate a clean
                    // slate, e.g. after resizes)
                    rView->clear();

                    // render at given output position
                    pBitmap->move( aOutPosPixel );
                    pBitmap->clip( ::basegfx::B2DPolyPolygon() ); // clear
                                                                  // clip
                                                                  // (might
                                                                  // have
                                                                  // been
                                                                  // changed,
                                                                  // e.g. from
                                                                  // comb
                                                                  // transition)
                    pBitmap->draw( pDevicePixelCanvas );
                }

            private:
                Slide& mrSlide;
            };
        }

        bool Slide::show()
        {
            ENSURE_AND_RETURN( maContext.mpLayerManager.get(), 
                               "Slide::show(): Invalid layer manager" );

            if( maViews.empty() )
                return false; // no views

            // set initial shape attributes (e.g. hide 'appear' effect
            // shapes)
            if( !applyInitialShapeAttributes( mxRootNode ) )
                return false;

            // tell the ActivitiesQueue to send us repaints. This call
            // overwrites any previously set layer managers, especially
            // those set by previous slides.
            maContext.mrActivitiesQueue.setLayerManager( 
                maContext.mpLayerManager );

            const bool bIsAnimated( isAnimated() );

            if( bIsAnimated )
            {
                // feed events into queues.
                maAnimations.start();
            }

            // NOTE: this looks slightly weird, but is indeed correct:
            // as isAnimated() might return false, _although_ there is
            // a main sequence (because the animation nodes don't
            // contain any executable effects), we gotta check both
            // conditions here.
            if( !bIsAnimated || !mbMainSequenceFound )
            {
                // manually trigger a slide animation end event (we
                // don't have animations at all, or we don't have a
                // main animation sequence, but if we had, it'd end
                // now). Note that having animations alone does not
                // matter here, as only main sequence animations
                // prevents showing the next slide on nextEvent().
                maContext.mrEventMultiplexer.notifySlideAnimationsEnd();
            }

            // render slide to screen (might not have been done, if
            // e.g. no slide transition took place; or slide
            // transition left ugly remnants). Don't render from first
            // principles, but try to use buffered bitmap
            ::std::for_each( maViews.begin(),
                             maViews.end(),
                             SlideRenderer( *this ) );
            maContext.mpLayerManager->clearPendingUpdates();
            maContext.mpLayerManager->updateScreen();

            // enable shape-intrinsic animations (drawing layer
            // animations or GIF animations)
            if( mbImageAnimationsAllowed )
                startIntrinsicAnimations();

            // TODO(F3): Enable/disable mpEventBroadcaster,
            // too. Otherwise, previous and next slides might produce
            // ghost events and ghost shape cursor changes.

            // enable paint overlay, if maUserPaintColor is valid
            enablePaintOverlay();

            // from now on, animations might be showing
            meAnimationState = SHOWING_STATE;

            return true;
        }

        void Slide::end()
        {
            // from now on, all animations are stopped
            meAnimationState = FINAL_STATE;

            // disable user paint overlay under all circumstances,
            // this slide now ceases to be active.
            disablePaintOverlay();

            // remove layer manager from the ActivitiesQueue, this
            // slide has ended its active life. For the duration of
            // the slide transition, the Presentation class takes over
            // control of the View canvas (note that in theory, the
            // layer manager can remain active, since no shape on the
            // slide is supposed to change. Unfortunately,
            // maAnimations.end() below might actually remove various
            // held effects).
            maContext.mrActivitiesQueue.setLayerManager( LayerManagerSharedPtr() );

			// simply forward to animations
            maAnimations.end();

            // switch off all shape-intrinsic animations. Do that
            // unconditionally (i.e. without checking
            // mbImageAnimationsAllowed), since somebody might have
            // disabled image animations _after_ the slide was
            // started.
            endIntrinsicAnimations();
        }

        bool Slide::isShowing() const
        {
            return meAnimationState == SHOWING_STATE;
        }

        bool Slide::isAnimated()
        {
            // prefetch, but don't apply initial shape attributes
            if( !implPrefetchShow() )
                return false;

            return mbHaveAnimations && maAnimations.isAnimated();
        }

        namespace
        {
            /** Little wrapper around a BitmapCanvas, to render into a slide bitmap
             */
            class BitmapView : public View
            {
            public:
                BitmapView( const ::cppcanvas::BitmapCanvasSharedPtr& 	rCanvas,
                            const ::basegfx::B2ISize&					rSize ) :
                    mpCanvas( rCanvas ),
                    maSize( rSize )
                {
                }

                virtual ViewLayerSharedPtr createViewLayer() const
                {
                    // TODO(Q2): Put that into a separate class
                    return ViewLayerSharedPtr( new BitmapView( mpCanvas,
                                                               maSize ) );
                }

                virtual bool updateScreen() const
                {
                    return true; // NOOP on BitmapCanvas
                }

                virtual void clear() const
                {
                    ::cppcanvas::CanvasSharedPtr pCanvas( mpCanvas->clone() );

                    // set transformation to identitiy (->device pixel)
                    pCanvas->setTransformation( ::basegfx::B2DHomMatrix() );

                    // fill the bounds rectangle in white
                    ::basegfx::B2DPolygon aPoly;                
                    aPoly.append( ::basegfx::B2DPoint() );
                    aPoly.append( ::basegfx::B2DPoint(maSize.getX(), 0) );
                    aPoly.append( ::basegfx::B2DPoint(maSize.getX(), maSize.getY()) );
                    aPoly.append( ::basegfx::B2DPoint(0, maSize.getY()) );
                    aPoly.setClosed(true);

                    ::cppcanvas::PolyPolygonSharedPtr pPolyPoly( 
                        ::cppcanvas::BaseGfxFactory::getInstance().createPolyPolygon( pCanvas, aPoly ) );

                    if( pPolyPoly.get() )
                    {
                        pPolyPoly->setRGBAFillColor( 0xFFFFFFFFU );
                        pPolyPoly->draw();
                    }
                }

                virtual bool isContentDestroyed() const
                {
                    return false;
                }

                virtual ::basegfx::B2DHomMatrix getTransformation() const
                {
                    OSL_ENSURE( false, "BitmapView::createSprite(): This method is not supposed to be called!" );
                    return ::basegfx::B2DHomMatrix();
                }

                virtual void setViewSize( const ::basegfx::B2DSize& rSize )
                {
                    // noop
                    OSL_ENSURE( false, "BitmapView::setViewSize(): This method is not supposed to be called!" );
                }

                virtual void setClip( const ::basegfx::B2DPolyPolygon& rClip )
                {
                    // TODO(F1): Implement clipping
                }

                virtual void setMouseCursor( sal_Int16 nPointerShape )
                {
                    // noop, ignore here
                }

                virtual ::cppcanvas::CustomSpriteSharedPtr createSprite( const ::basegfx::B2DSize& rSpriteSizePixel ) const          
                {
                    OSL_ENSURE( false, "BitmapView::createSprite(): This method is not supposed to be called!" );
                    return ::cppcanvas::CustomSpriteSharedPtr();
                }

                virtual double getSpritePriority( double nSpritePrio ) const
                {
                    return nSpritePrio;
                }

                virtual ::cppcanvas::CanvasSharedPtr getCanvas() const
                {
                    return mpCanvas;
                }

                virtual void setPriority( double nPrio )
                {
                    // noop, ignore here
                    OSL_TRACE( "BitmapView::setPriority(): Called, but ignored" );
                }

            private:
                ::cppcanvas::BitmapCanvasSharedPtr	mpCanvas;
                const ::basegfx::B2ISize			maSize;
            };
        }

        SlideBitmapSharedPtr Slide::getCurrentSlideBitmap( const UnoViewSharedPtr& rView )
        {
            // search view in our view vector - have to rule out
            // unregistered views, and determine the index of rView in
            // our view vector (can use same index for maSlideBitmaps
            // vector)
            UnoViewVector::iterator 	  aIter;
            const UnoViewVector::iterator aBegin( maViews.begin() );
            const UnoViewVector::iterator aEnd  ( maViews.end() );
            if( (aIter = ::std::find_if( aBegin, 
                                         aEnd, 
                                         ::boost::bind(
                                             ::std::equal_to< ::cppcanvas::CanvasSharedPtr >(),
                                             ::boost::cref( rView->getCanvas() ),
                                             ::boost::bind(
                                                 &ViewLayer::getCanvas,
                                                 _1 ) ) ) ) == aEnd )
            {
                // corresponding view not found - maybe view was not
                // added to Slide?
                ENSURE_AND_THROW( false,
                                  "Slide::getInitialSlideBitmap(): view does not "
                                  "match any of the added ones" );
            }

            // ensure that the show is loaded
            if( !mbShowLoaded )
            {
                // only prefetch and init shapes when not done already
                // (otherwise, at least applyInitialShapeAttributes()
                // will be called twice for initial slide
                // rendering). Furthermore,
                // applyInitialShapeAttributes() _always_ performs
                // initializations, which would be highly unwanted
                // during a running show. OTOH, a slide whose
                // mbShowLoaded is false is guaranteed not be running
                // a show.

                // set initial shape attributes (e.g. hide 'appear' effect
                // shapes)
                if( !applyInitialShapeAttributes( mxRootNode ) )
                    ENSURE_AND_THROW(false,"Slide::getCurrentSlideBitmap(): Cannot apply initial attributes");
            }

            // check whether we've already got a bitmap buffered for
            // the current state
            const ::std::size_t nViewIndex( 
                ::std::distance( aBegin, aIter ) );

            SlideBitmapSharedPtr& rBitmap( 
                maSlideBitmaps.at( nViewIndex ).at( meAnimationState ) );

			const ::basegfx::B2ISize& rSlideSize( getSlideSize( rView ) );

			// is the bitmap valid (actually existent, and of correct
			// size)?
            if( !rBitmap.get() ||
				rBitmap->getSize() != rSlideSize )
            {
                // no bitmap there yet, or wrong size - create one
                rBitmap = createCurrentSlideBitmap(rView);
            }

            return rBitmap;
        }

        bool Slide::hasAutomaticNextSlide() const
        {
            return mbHasAutomaticNextSlide;
        }

        double Slide::getAutomaticNextSlideTimeout() const
        {
            return mnNextSlideTimeout;
        }

        void Slide::setUserPaintColor( const ::comphelper::OptionalValue< RGBColor >& rColor )
        {
            maUserPaintColor = rColor;

            if( isShowing() &&
                maUserPaintColor.isValid() )
            {
                enablePaintOverlay();
            }
            else
            {
                disablePaintOverlay();
            }
        }

        // private methods
        //--------------------------------------------------------------------------------------------------------------


        ::basegfx::B2ISize Slide::getSlideSize( const UnoViewSharedPtr& rView )
        {
            ::basegfx::B2ISize aPageSize;
            if( !getPageSize( aPageSize ) )
                ENSURE_AND_THROW(false,"Slide::getSlideSize(): Cannot fetch page size");

            ::cppcanvas::CanvasSharedPtr pCanvas( rView->getCanvas() );

            // determine transformed page bounds
            const ::basegfx::B2DRectangle aRect( 0,0,
                                                 aPageSize.getX(),
                                                 aPageSize.getY() );
            ::basegfx::B2DRectangle aTmpRect;
            ::canvas::tools::calcTransformedRectBounds( aTmpRect, aRect, pCanvas->getTransformation() );

            return ::basegfx::B2ISize( ::basegfx::fround( aTmpRect.getRange().getX() ),
									   ::basegfx::fround( aTmpRect.getRange().getY() ) );
		}

        SlideBitmapSharedPtr Slide::createCurrentSlideBitmap( const UnoViewSharedPtr& rView )
        {
            ENSURE_AND_THROW( rView.get() && rView->getCanvas().get(), 
                              "Slide::createCurrentSlideBitmap(): Invalid view" );
            ENSURE_AND_THROW( maContext.mpLayerManager.get(), 
                              "Slide::createCurrentSlideBitmap(): Invalid layer manager" );
            ENSURE_AND_THROW( mbShowLoaded, 
                              "Slide::createCurrentSlideBitmap(): No show loaded" );

            const ::basegfx::B2ISize& rBmpSize( getSlideSize( rView ) );

            ::cppcanvas::CanvasSharedPtr pCanvas( rView->getCanvas() );

            // create a bitmap of appropriate size
            ::cppcanvas::BitmapSharedPtr pBitmap( 
                ::cppcanvas::BaseGfxFactory::getInstance().createBitmap( 
                    pCanvas, 
                    rBmpSize ) );

            ENSURE_AND_THROW( pBitmap.get(),
                              "Slide::createCurrentSlideBitmap(): Cannot create page bitmap" );

            ::cppcanvas::BitmapCanvasSharedPtr pBitmapCanvas( pBitmap->getBitmapCanvas() );

            ENSURE_AND_THROW( pBitmapCanvas.get(),
                              "Slide::createCurrentSlideBitmap(): Cannot create page bitmap canvas" );

            // apply linear part of destination canvas transformation (linear means in this context: 
            // transformation without any translational components)
            ::basegfx::B2DHomMatrix aLinearTransform( pCanvas->getTransformation() );
            aLinearTransform.set( 0, 2, 0.0 );
            aLinearTransform.set( 1, 2, 0.0 );
            pBitmapCanvas->setTransformation( aLinearTransform );

            ViewSharedPtr pView( new BitmapView( pBitmapCanvas,
                                                 rBmpSize ) );
            maContext.mpLayerManager->addView( pView ); // addView implicitely draws everything
            maContext.mpLayerManager->removeView( pView );

            return SlideBitmapSharedPtr( new SlideBitmap( pBitmap ) );
        }

        namespace
        {
            class MainSequenceSearcher
            {
            public:
                MainSequenceSearcher()
                {
                    maSearchKey.Name = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "node-type" ) );
                    maSearchKey.Value <<= ::com::sun::star::presentation::EffectNodeType::MAIN_SEQUENCE;
                }

                void operator()( const uno::Reference< animations::XAnimationNode >& xChildNode )
                {
                    uno::Sequence< beans::NamedValue > aUserData( xChildNode->getUserData() );
                    
                    if( findNamedValue( aUserData, maSearchKey ) )
                    {
                        maMainSequence = xChildNode;
                    }
                }

                uno::Reference< animations::XAnimationNode > getMainSequence() const
                {
                    return maMainSequence;
                }

            private:
                beans::NamedValue 								maSearchKey;
                uno::Reference< animations::XAnimationNode >	maMainSequence;
            };
        }

        bool Slide::implPrefetchShow()
        {
            if( mbShowLoaded )
                return true;

            ENSURE_AND_RETURN( mxDrawPage.is(), 
                               "Slide::implPrefetchShow(): Invalid draw page" );
            ENSURE_AND_RETURN( maContext.mpLayerManager.get(), 
                               "Slide::implPrefetchShow(): Invalid layer manager" );

            // fetch desired page content
            // ==========================

            if( !prefetchShapes() )
                return false;

            // New animations framework: import the shape effect info
            // ======================================================

            try
            {
                if( mxRootNode.is() )
                {
                    if( !maAnimations.importAnimations( mxRootNode ) )
                    {
                        OSL_ENSURE( false,
                                    "Slide::implPrefetchShow(): have animation nodes, "
                                    "but import animations failed." );

                        // could not import animation framework,
                        // _although_ some animation nodes are there -
                        // this is an error (not finding animations at
                        // all is okay - might be a static slide)
                        return false; 
                    }

                    // now check whether we've got a main sequence (if
                    // not, we must manually call
                    // EventMultiplexer::notifySlideAnimationsEnd()
                    // above, as e.g. interactive sequences alone
                    // don't block nextEvent() from issuing the next
                    // slide)
                    MainSequenceSearcher aSearcher;
                    if( ::anim::for_each_childNode( mxRootNode, aSearcher ) )
                        mbMainSequenceFound = aSearcher.getMainSequence().is();

                    // import successfully done
                    mbHaveAnimations = true;
                }
            }
            catch( uno::Exception& )
            {
                // TODO(E2): Error handling. For now, bail out
                OSL_ENSURE( false,
                            "Slide::implPrefetchShow(): caught exception while importing "
                            "animations - slide will be static" );
            }

            mbShowLoaded = true;

            return true;
        }

        void Slide::enablePaintOverlay()
        {
            if( maUserPaintColor.isValid() )
            {
                mpPaintOverlay.reset( new UserPaintOverlay( maUserPaintColor.getValue(),
                                                            2.0,
                                                            maContext.mrEventMultiplexer ) );
                // add all views to the overlay
                ::std::for_each( maViews.begin(),
                                 maViews.end(),
                                 ::boost::bind( &UserPaintOverlay::addView,
                                                ::boost::ref( mpPaintOverlay ),
                                                _1 ) );
            }
        }

        void Slide::disablePaintOverlay()
        {
            mpPaintOverlay.reset();
        }

        bool Slide::getPageSize( ::basegfx::B2ISize& o_rPageSize ) const
        {
            uno::Reference< beans::XPropertySet > xPropSet( mxDrawPage, 
                                                            uno::UNO_QUERY );
            if( !xPropSet.is() )
                return false;

            sal_Int32 nDocWidth;
            sal_Int32 nDocHeight;
            xPropSet->getPropertyValue( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM("Width") ) ) >>= nDocWidth;
            xPropSet->getPropertyValue( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM("Height") ) ) >>= nDocHeight;

            o_rPageSize.setX( nDocWidth );
            o_rPageSize.setY( nDocHeight );

            return true;
        }

        ::basegfx::B2DRectangle Slide::getPageRect() const
        {
            ::basegfx::B2ISize aPageSize;
            if( !getPageSize( aPageSize ) )
                ENSURE_AND_THROW(false,"Slide::getPageRect(): Cannot fetch page size");
            
            return ::basegfx::B2DRectangle(0.0,0.0,
                                           aPageSize.getX(),
                                           aPageSize.getY());
        }

        namespace
        {
            /** Adapt ShapeSharedPtr to the startIntrinsicAnimation()
                method.
             */
            class StartIntrinsicAnimationAdaptor
            {
            public:
                StartIntrinsicAnimationAdaptor( const SlideShowContext& rContext ) :
                    maContext( rContext )
                {
                }

                void operator()( ShapeSharedPtr& rShape ) const
                {
                    if( rShape.get() &&
                        rShape->hasIntrinsicAnimation() )
                    {
                        IntrinsicAnimationSharedPtr pIntrinsicAnimation(
                            ::boost::dynamic_pointer_cast< IntrinsicAnimation >( rShape ) );

                        if( pIntrinsicAnimation.get() )
                            pIntrinsicAnimation->startIntrinsicAnimation( maContext,
                                                                          rShape );
                    }
                }

            private:
                SlideShowContext maContext;
            };

            class EndIntrinsicAnimationAdaptor
            {
            public:
                EndIntrinsicAnimationAdaptor()
                {
                }

                void operator()( ShapeSharedPtr& rShape ) const
                {
                    if( rShape.get() &&
                        rShape->hasIntrinsicAnimation() )
                    {
                        IntrinsicAnimationSharedPtr pIntrinsicAnimation(
                            ::boost::dynamic_pointer_cast< IntrinsicAnimation >( rShape ) );

                        if( pIntrinsicAnimation.get() )
                            pIntrinsicAnimation->endIntrinsicAnimation();
                    }
                }
            };
        }

        void Slide::endIntrinsicAnimations()
        {
            if( maContext.mpLayerManager.get() )
            {
                maContext.mpLayerManager->forEachShape( 
                    EndIntrinsicAnimationAdaptor() );
            }
        }

        void Slide::startIntrinsicAnimations()
        {
            if( maContext.mpLayerManager.get() )
            {
                maContext.mpLayerManager->forEachShape( 
                    StartIntrinsicAnimationAdaptor( maContext ) );
            }
        }

        bool Slide::applyInitialShapeAttributes( const uno::Reference< animations::XAnimationNode >& xRootAnimationNode )
        {
            if( !implPrefetchShow() )
                return false;

            if( !xRootAnimationNode.is() )
            {
                meAnimationState = INITIAL_STATE;

                return true; // no animations - no attributes to apply
                             // - succeeded
            }

            uno::Reference< animations::XTargetPropertiesCreator > xPropsCreator;

            try
            {
                ENSURE_AND_RETURN( maContext.mxComponentContext.is(),
                                   "Slide::applyInitialShapeAttributes(): Invalid component context" );

                uno::Reference<lang::XMultiComponentFactory> xFac( 
                    maContext.mxComponentContext->getServiceManager() );

                xPropsCreator.set(
                    xFac->createInstanceWithContext(
                        ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM(
                                             "com.sun.star.animations.TargetPropertiesCreator") ),
                        maContext.mxComponentContext ), 
                    uno::UNO_QUERY_THROW );
            }
            catch( uno::Exception& ) 
            {
                // could not determine initial shape attributes - this
                // is an error, as some effects might then be plainly
                // invisible
                ENSURE_AND_RETURN( false,
                                   "Slide::applyInitialShapeAttributes(): "
                                   "couldn't create TargetPropertiesCreator." );
            }

            uno::Sequence< animations::TargetProperties > aProps(
                xPropsCreator->createInitialTargetProperties( xRootAnimationNode ) );

            // apply extracted values to our shapes
            const ::std::size_t nSize( aProps.getLength() );
            for( ::std::size_t i=0; i<nSize; ++i )
            {
                sal_Int16 							nParaIndex( -1 );
                uno::Reference< drawing::XShape >	xShape( aProps[i].Target,
                                                            uno::UNO_QUERY );

                if( !xShape.is() ) 
                {
                    // not a shape target. Maybe a ParagraphTarget?
                    ::com::sun::star::presentation::ParagraphTarget aParaTarget;
                    
                    if( (aProps[i].Target >>= aParaTarget) )
                    {
                        // yep, ParagraphTarget found - extract shape
                        // and index
                        xShape = aParaTarget.Shape;
                        nParaIndex = aParaTarget.Paragraph;
                    }
                }

                if( xShape.is() )
                {
                    ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

                    if( !pShape.get() )
                    {
                        OSL_ENSURE( false,
                                    "Slide::applyInitialShapeAttributes(): no shape found for given target" );
                        continue;
                    }

                    AttributableShapeSharedPtr pAttrShape( 
                        ::boost::dynamic_pointer_cast< AttributableShape >( pShape ) );

                    if( !pAttrShape.get() )
                    {
                        OSL_ENSURE( false,
                                    "Slide::applyInitialShapeAttributes(): shape found does not "
                                    "implement AttributableShape interface" );
                        continue;
                    }

                    if( nParaIndex != -1 )
                    {
                        // our target is a paragraph subset, thus look
                        // this up first.
                        pAttrShape = pAttrShape->querySubset( 
                            DocTreeNode::createFromShape( pAttrShape,
                                                          nParaIndex,
                                                          DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH ) );

                        if( !pAttrShape.get() )
                        {
                            OSL_ENSURE( false,
                                        "Slide::applyInitialShapeAttributes(): shape found does not "
                                        "provide a subset for requested paragraph index" );
                            continue;
                        }
                    }

                    const uno::Sequence< beans::NamedValue >& rShapeProps( aProps[i].Properties );
                    const ::std::size_t nShapePropSize( rShapeProps.getLength() );
                    for( ::std::size_t j=0; j<nShapePropSize; ++j )
                    {
                        bool bVisible;
                        if( rShapeProps[j].Name.equalsIgnoreAsciiCaseAscii("visibility") &&
                            extractValue( bVisible,
                                          rShapeProps[j].Value,
                                          pShape,
                                          maContext.mpLayerManager ) )
                        {
                            pAttrShape->setVisibility( bVisible );
                        }
                        else
                        {
                            OSL_ENSURE( false,
                                        "Slide::applyInitialShapeAttributes(): Unexpected "
                                        "(and unimplemented) property encountered" );
                        }
                    }
                }
            }

            meAnimationState = INITIAL_STATE;

            return true;
        }

        bool Slide::prefetchShapes()
        {
            if( mbShapesLoaded )
                return true;

            ENSURE_AND_RETURN( mxDrawPage.is(), 
                               "Slide::prefetchShapes(): Invalid draw page" );
            ENSURE_AND_RETURN( maContext.mpLayerManager.get(), 
                               "Slide::prefetchShapes(): Invalid layer manager" );

            // fetch desired page content
            // ==========================

            // also take master page content
            uno::Reference< drawing::XMasterPageTarget > xMasterPageTarget( mxDrawPage, 
                                                                            uno::UNO_QUERY );
            if( !xMasterPageTarget.is() )
                return false;

            uno::Reference< drawing::XDrawPage > xMasterPage( xMasterPageTarget->getMasterPage() );
            if( !xMasterPage.is())
                return false;

            uno::Reference< drawing::XShapes > xMasterPageShapes( xMasterPage, 
                                                                  uno::UNO_QUERY );
            if( !xMasterPageShapes.is() )
                return false;

            // TODO(P2): maybe cache master pages here (or treat the
            // masterpage as a single metafile. At least currently,
            // masterpages do not contain animation effects)
            try
            {
                // add the background shape
                // -------------------------------------------------------------------------
                maContext.mpLayerManager->addShape( 
                    ShapeSharedPtr(
                        new BackgroundShape( mxDrawPage, 
                                             xMasterPage ) ) );


                // load the masterpage shapes
                // -------------------------------------------------------------------------

                ShapeImporter aMPShapesFunctor( xMasterPage, 
                                                mxDrawPage,
                                                0,
                                                true );

                while( !aMPShapesFunctor.isImportDone() )
                    maContext.mpLayerManager->addShape( aMPShapesFunctor.importShape() );


                // load the normal page shapes
                // -------------------------------------------------------------------------

                ShapeImporter aShapesFunctor( mxDrawPage, 
                                              mxDrawPage,
                                              xMasterPageShapes->getCount(), 
                                              false );

                while( !aShapesFunctor.isImportDone() )
                    maContext.mpLayerManager->addShape( aShapesFunctor.importShape() );
            }
            catch( ImportFailedException& )
            {
                // TODO(E2): Error handling. For now, bail out
                OSL_ENSURE( false,
                            "Slide::prefetchShapes(): caught ImportFailedException" );
                return false;
            }

            mbShapesLoaded = true;

            return true;
        }

    }
}
