# Copyright (C) 2000-2001 The OpenRPG Project
#
#   openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: voxchat.py
# Author: Ted Berg
# Maintainer:
# Version:
#   $Id: voxchat.py,v 1.14 2003/02/11 19:48:55 posterboy Exp $
#
# Description: nodehandler for alias.
#

__version__ = "$Id: voxchat.py,v 1.14 2003/02/11 19:48:55 posterboy Exp $"

import core
import orpg.tools.scriptkit
# import base
import orpg.tools.predTextCtrl
from wxPython.misc import wxNewId
import orpg.tools.rgbhex
import orpg.dirpath
import re
import orpg.networking.mplay_client
import os
import string
from orpg.orpg_windows import *
import orpg.tools.inputValidator
import orpg.minidom
import orpg.tools.orpg_sound


ANTI_SPOOF = ''
NEWLINE = '\n'

# openrpg constants
COMPONENT_SESSION = 'session'
COMPONENT_SETTINGS = 'settings'
COMPONENT_CHAT = 'chat'
COMPONENT_MAP = 'map'
LAYER_MINIATURES = 'miniatures'
SETTINGS_PLAYER = 'player'
SETTINGS_SEND_SOUND = 'SendSound'
SETTINGS_UNIX_SOUND_PLAYER = 'UnixSoundPlayer'

# node names
VOXCHAT_FILTER_RULE = 'rule'
VOXCHAT_ALIAS = 'voxchat.alias'
VOXCHAT_FILTER = 'voxchat.filter'

# node attributes
ATTRIBUTE_NAME = 'name'
ATTRIBUTE_USE_FILTER  = 'use.filter'
ATTRIBUTE_MATCH = 'match'
ATTRIBUTE_SUB = 'sub'
ATTRIBUTE_FILTER_NAME = 'filter.name'

# GUI constants
# Menu Constants
MENU_DIALOG_CLOSE=wxNewId()
MENU_TRANSMIT_SEND = wxNewId()
MENU_TRANSMIT_EMOTE = wxNewId()
MENU_TRANSMIT_WHISPER = wxNewId()
MENU_TRANSMIT_RAW = wxNewId()
MENU_TOOLS_LOOKUP_NAME = wxNewId()
MENU_ALIAS_ADD_TEMP=wxNewId()
MENU_ALIAS_ADD=wxNewId()
MENU_ALIAS_DEL=wxNewId()
MENU_ALIAS_EDIT=wxNewId()
MENU_FILTER_EXPORT=wxNewId()
MENU_FILTER_ADD=wxNewId()
MENU_FILTER_DEL=wxNewId()
MENU_FILTER_EDIT=wxNewId()
# Other
ALIAS_EDIT_LIST = wxNewId()
ALIAS_USE_LIST = wxNewId()
FILTER_EDIT_LIST = wxNewId()
FILTER_USE_LIST = wxNewId()
CHAT_ACTION = wxNewId()
EMOTE_ACTION = wxNewId()
RAW_SEND_ACTION = wxNewId()
DONE_ACTION = wxNewId()
WHISPER_ACTION = wxNewId()
FILTER_USE = wxNewId()
AUTO_CLEAR_USE = wxNewId()
ADD_ALIAS = wxNewId()
DEL_ALIAS = wxNewId()
EDIT_ALIAS = wxNewId()
ADD_FILTER = wxNewId()
DEL_FILTER = wxNewId()
EDIT_FILTER = wxNewId()
ADD_RULE = wxNewId()
DEL_RULE = wxNewId()
MOVE_RULE_UP = wxNewId()
MOVE_RULE_DOWN = wxNewId()
BOLD_TEXT = wxNewId()
ITAL_TEXT = wxNewId()
UNDR_TEXT = wxNewId()
COLOR_TEXT = wxNewId()
UPDATE_LIST_ACTION = wxNewId()
EXPORT_FILTER_ACTION = wxNewId()

# GUI Labels
LABEL_ADD_RULE = 'Add Rule'
LABEL_DELETE_RULE = 'Delete Rule'
LABEL_MOVE_UP = 'Move Up'
LABEL_MOVE_DOWN = 'Move Down'
LABEL_MATCH = 'Match'
LABEL_REPLACE = 'Replace'
LABEL_FILTER_TEXT = 'Filter Text'
LABEL_AUTO_CLEAR_TEXT = 'Auto Clear Text'
LABEL_SAY = 'Say'
LABEL_DO = 'Do'
LABEL_WHISPER = 'Whisper'
LABEL_SEND_RAW = 'Send Raw'
LABEL_DONE = 'Done'

# GUI Tooltips
TOOLTIP_1 = 'Add temporary aliases from map'
TOOLTIP_2 = 'Add a new alias'
TOOLTIP_3 = 'Delete the selected alias'
TOOLTIP_4 = 'Edit the selected alias'
TOOLTIP_5 = 'Add a new filter'
TOOLTIP_6 = 'Delete the selected filter'
TOOLTIP_7 = 'Edit the selected filter'
TOOLTIP_8 = 'Export the selected filter to the gametree'

#
# Good things to add: 
# 1.u'use filter' per alias  [ this should be done ]
# 2. make aliases remember which filter they're using  [ lisbox in gtk appaears to ignore SetSelection( <= 0 )
#

class default_node_handler( core.node_handler ):
    """
    <nodehandler class="voxchat_handler" icon="book" module="voxchat" name="?" use.filter="?">
      <voxchat.alias name="?" />
      <voxchat.filter name="?" >
        <rule match="?" sub="?"/>
      </voxchat.filter>
    </nodehandler>
      
    """

    def __init__( self, xml_dom, tree_node, openrpg, editpanel_class, caption_attrib ):
        core.node_handler.__init__( self, xml_dom, tree_node, openrpg )
        self.myeditor = None
        self.editpanel_class = editpanel_class
        self.caption_attrib = caption_attrib


##    def on_design( self, evt ):
##        if self.myeditor == None or self.myeditor.destroyed == 1:
##            caption = self.master_dom.getAttribute( self.caption_attrib )
##            self.myeditor = wxPFrame( self.frame, caption )
##            wnd = self.editpanel_class( self.myeditor, self )
##            self.mywnd = wnd
##            self.myeditor.panel = wnd
##            self.myeditor.Show( 1 )
##        else:
##            self.myeditor.Raise()



class filter_nodehandler( core.node_handler ):
    def __init__( self, xml_dom, tree_node, openrpg ):
        core.node_handler.__init__( self, xml_dom, tree_node, openrpg )

    def  toHtml( self ):
        return self.master_dom.getElementsByTagName( "description" ).firstChild.data



class voxchat_handler( default_node_handler ):
    def __init__( self, xml_dom, tree_node, openrpg ):
        default_node_handler.__init__( self, xml_dom, tree_node, openrpg, voxchat_newuse_panel, ATTRIBUTE_NAME )
        self.refresh_names()
        self.mywindow = None

        
    def refresh_names( self ):
        self.names = []
        list = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        for entry in list:
            name = entry.getAttribute( ATTRIBUTE_NAME )
            self.names.append( name )
        self.names.sort()
        return self.names

    
    def refresh_filters( self ):
        self.filters = []
        list = self.master_dom.getElementsByTagName( VOXCHAT_FILTER )
        for entry in list:
            name = entry.getAttribute( ATTRIBUTE_NAME )
            self.filters.append( name )
        self.filters.sort()
        return self.filters

        
##    def on_use( self, evt ):
##        if self.mywindow == None or self.mywindow.destroyed == 1:
##            caption = self.master_dom.getAttribute( ATTRIBUTE_NAME )
##            self.mywindow = wxPFrame( self.frame, caption )
##            wnd = voxchat_newuse_panel( self.mywindow, self )
##            self.mywindow.panel = wnd
##            self.wnd = wnd
##            self.mywindow.Show( 1 )
##        else:
##            self.mywindow.Raise()
    

    def get_design_panel( self, parent ):
        return voxchat_newuse_panel( parent, self )


    def get_use_panel( self, parent ):
        return voxchat_newuse_panel( parent, self )


    def add_alias( self, alias ):
        node = core.minidom.Element( VOXCHAT_ALIAS )
        node.setAttribute( ATTRIBUTE_NAME, alias )
        self.master_dom.appendChild( node )
        self.update_alias_list()

        
    def del_alias( self, alias ):
        list = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        for node in list:
            if node.getAttribute(ATTRIBUTE_NAME) == alias:
                self.master_dom.removeChild( node )
                self.update_alias_list()
                break

        
    def change_alias( self, old, new, destructive_refresh=1 ):
        list = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        for node in list:
            if node.getAttribute(ATTRIBUTE_NAME) == old:
                node.setAttribute(ATTRIBUTE_NAME, new )
                if destructive_refresh:
                    self.update_alias_list()
                break


    def set_alias_does_filter( self, alias_name, value="1" ):
        nl = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        for node in nl:
            if node.getAttribute( ATTRIBUTE_NAME ) == alias_name:
                node.setAttribute( ATTRIBUTE_USE_FILTER, value )
                return


    def get_alias_does_filter( self, alias_name ):
        nl = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        for node in nl:
            if node.getAttribute( ATTRIBUTE_NAME ) == alias_name:
                result = node.getAttribute( ATTRIBUTE_USE_FILTER )
                return result
        return ''


    def set_alias_filter( self, alias_name, filter_name ):
        nl = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        node = None
        for node in nl:
            if node.getAttribute( ATTRIBUTE_NAME ) == alias_name:
                node.setAttribute( ATTRIBUTE_FILTER_NAME, filter_name )
                break


    def get_alias_filter_name( self, alias_name ):
        nl = self.master_dom.getElementsByTagName( VOXCHAT_ALIAS )
        for node in nl:
            if node.getAttribute( ATTRIBUTE_NAME ) == alias_name:
                return node.getAttribute( ATTRIBUTE_FILTER_NAME )
        return ''                
    

    def update_alias_list( self ):
        if self.myeditor != None and self.myeditor.destroyed == 0:
            self.myeditor.panel.update_alias_listbox()
        if self.mywindow != None and self.mywindow.destroyed == 0:
            self.mywindow.panel.update_alias_listbox()


    def update_filter_list( self ):
        if self.myeditor != None and self.myeditor.destroyed == 0:
            self.myeditor.panel.update_filter_listbox()
        if self.mywindow != None and self.mywindow.destroyed == 0:
            self.mywindow.panel.update_filter_listbox()


    def add_filter( self, name ):
        node = core.minidom.Element( VOXCHAT_FILTER )
        node.setAttribute( ATTRIBUTE_NAME, name )
        self.master_dom.appendChild( node )
        self.update_filter_list()


    def del_filter( self, name ):
        list = self.master_dom.getElementsByTagName( VOXCHAT_FILTER )
        for node in list:
            if node.getAttribute( ATTRIBUTE_NAME ) == name:
                self.master_dom.removeChild( node )
                self.update_filter_list()
                break;
    

    def rename_filter( self, oldname, newname ):
        list = self.master_dom.getElementsByTagName( VOXCHAT_FILTER )
        for node in list:
            if node.getAttribute( ATTRIBUTE_NAME ) == oldname:
                node.setAttribute( ATTRIBUTE_NAME, newname )
                self.update_filter_list()
                break;
    

    def get_use_filter( self ):
        try:
            return int( self.master_dom.getAttribute( ATTRIBUTE_USE_FILTER ) )
        except:
            return 0


    def set_use_filter( self, value ):
        self.master_dom.setAttribute( ATTRIBUTE_USE_FILTER, `value` )


    def export_filter( self, filter_name ):
        e = orpg.minidom.Element( 'nodehandler' )
        e.setAttribute( 'class', 'filter_nodehandler' )
        e.setAttribute( 'icon', 'note' )
        e.setAttribute( 'module', 'voxchat' )
        nl = self.master_dom.getElementsByTagName( VOXCHAT_FILTER )
        for f in nl:
            if f.getAttribute( ATTRIBUTE_NAME ) == filter_name:
                filter = f
                break
        node = filter.cloneNode( 1 )
        e.setAttribute( 'name', node.getAttribute( 'name' ) )
        e.appendChild( node )
        gametree = self.myopenrpg.get_component( 'tree' )
        gametree.insert_xml( e.toxml() )
        e.unlink()


    def on_drop( self, evt ):
        drag_obj = self.tree.drag_obj
        if  drag_obj == self or self.tree.is_parent_node( self.mytree_node, drag_obj.mytree_node ):
            return
        if isinstance( drag_obj, filter_nodehandler ):
            xml_dom = drag_obj.delete()
            obj = xml_dom.removeChild( xml_dom.firstChild )
            xml_dom.unlink()
            self.master_dom.appendChild( obj )
            print self.__dict__.keys()



class voxchat_filter_edit_panel( wxPanel ):
    def __init__( self, parent, nodehandler, node ):
        wxPanel.__init__( self, parent, -1 )
        self.parent = parent
        self.handler = nodehandler
        self.node = node
        self.create_gui()
        self.histidx = 0
        self.history = []
        self.lasthistevt = None


    def create_gui( self ):
        sizer = wxBoxSizer( wxVERTICAL )
        self.grid = filterGrid( self, self.handler, self.node )
        # self.grid.SetSelectionMode( orpg_windows.wxGrid.wxGRID_SELECT_ROWS )
        
        buttonsizer = wxBoxSizer( wxHORIZONTAL )
        buttonsizer.Add( wxButton( self, ADD_RULE, LABEL_ADD_RULE ) )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, DEL_RULE, LABEL_DELETE_RULE ) )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, MOVE_RULE_UP, LABEL_MOVE_UP ) )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, MOVE_RULE_DOWN, LABEL_MOVE_DOWN ) )

        sizer.Add( self.grid, 1, wxEXPAND )
        sizer.Add( buttonsizer, 0, wxEXPAND )
        self.sizer = sizer
        EVT_SIZE( self, self.on_size )
        EVT_BUTTON( self, ADD_RULE, self.on_add_rule )
        EVT_BUTTON( self, DEL_RULE, self.on_del_rule )
        EVT_BUTTON( self, MOVE_RULE_UP, self.on_move_rule_up )
        EVT_BUTTON( self, MOVE_RULE_DOWN, self.on_move_rule_down )


    def on_add_rule( self, evt ):
        self.grid.do_add_rule()


    def on_del_rule( self, evt ):
        self.grid.do_del_rule( self.grid.GetGridCursorRow() ) # GetSelection() )


    def on_move_rule_up( self, evt ):
        index = self.grid.GetGridCursorRow() # GetSelection()
        if index > 0:
            self.grid.do_move_rule_up( index )


    def on_move_rule_down( self, evt ):
        index = self.grid.GetGridCursorRow() # GetSelection()
        if index < ( self.grid.GetNumberRows() -1 ):
            self.grid.do_move_rule_down( index )


    def on_size( self, evt ):
        s = self.GetClientSizeTuple()
        self.sizer.SetDimension( 10, 10, s[0] - 20, s[1] - 20 )



class filterGrid( wxGrid ):
    def __init__( self, parent, handler, node ):
        wxGrid.__init__( self, parent, -1, style = wxSUNKEN_BORDER | wxWANTS_CHARS )
        self.mynode = node
        self.parent = parent
        self.handler = handler
        self.CreateGrid( 0, 2 )
        self.SetColLabelValue( 0, LABEL_MATCH )
        self.SetColLabelValue( 1, LABEL_REPLACE )
        self.fix_selection( 0 )
        self.update_grid_from_dom()
        self.AutoSizeColumns()
        EVT_GRID_CELL_CHANGE( self, self.on_cell_change )
        EVT_GRID_SELECT_CELL( self, self.select_cell )


    def on_cell_change( self, evt ):
        list = self.mynode.getElementsByTagName( VOXCHAT_FILTER_RULE )
        row = evt.GetRow()
        col = evt.GetCol()
        node = list[ row ]
        
        if col == 0:
            node.setAttribute( ATTRIBUTE_MATCH, self.GetCellValue( row, col ) )
        elif col == 1:
            node.setAttribute( ATTRIBUTE_SUB, self.GetCellValue( row, col ) )


    def select_cell( self, evt ):
        self.fix_selection( evt.GetRow() )
        evt.Skip()


    def fix_selection( self, row ):
        # self.BeginBatch()
        self.selectedRow = row
        # self.SelectRow( self.selectedRow )
        # self.EndBatch()

  
    def do_add_rule( self ):
        self.AppendRows()
        n = core.minidom.Element( VOXCHAT_FILTER_RULE )
        self.mynode.appendChild( n )


    def do_del_rule( self, index ):
        list = self.mynode.getElementsByTagName( VOXCHAT_FILTER_RULE )
        n = list[ index ]
        self.mynode.removeChild( n )
        self.update_grid_from_dom()


    def do_move_rule_up( self, index ):
        list = self.mynode.getElementsByTagName( VOXCHAT_FILTER_RULE )
        n = list[ index ]
        p = list[ index - 1 ]
        self.mynode.removeChild( n )
        self.mynode.insertBefore( n, p )
        self.update_grid_from_dom()
        self.SelectRow( index - 1 )
        self.SetGridCursor( index - 1, 0 )


    def do_move_rule_down( self, index ):
        self.do_move_rule_up( index + 1 )
        self.SelectRow( index + 1 )
        self.SetGridCursor( index +1, 0 )
  

    def update_grid_from_dom( self ):
        list = self.mynode.getElementsByTagName( VOXCHAT_FILTER_RULE )
        rows = self.GetNumberRows()
        ents = len( list )
        if rows < ents:
            diff = ents - rows
            self.AppendRows( diff )
        if rows > ents:
            diff = rows - ents
            self.DeleteRows( diff )
        index = 0
        for rule in list:
            pre = rule.getAttribute( ATTRIBUTE_MATCH )
            post = rule.getAttribute( ATTRIBUTE_SUB )
            self.SetCellValue( index, 0, pre )
            self.SetCellValue( index, 1, post )
            index += 1


    def update_dom_from_grid( self ):
        list = self.mynode.getElementByTagName( VOXCHAT_FILTER_RULE )
        rows = self.GetNumberRows()
        ents = len( list )
        while rows < ents:
            n = core.minidom.Element( VOXCHAT_FILTER_RULE )
            self.mynode.appendChild( n )
        while ents > rows:
            self.mynode.removeChild( self.mynode._get_firstChild() )
    
        index = 0
        for rule in list:
            rule.setAttribute( ATTRIBUTE_MATCH, self.GetValue( index, 0 ) )
            rule.setAttribute( ATTRIBUTE_SUB, self.GetValue( index, 1 ) )
            index += 1



class mySplitter( wxSplitterWindow ):
    target = None
    def OnChar( self, evt ):
        if self.target != None:
            self.target.OnChar( evt )
        else:
            evt.Skip()



class voxchat_newuse_panel( wxPanel ):
    def __init__( self, parent, nodehandler ):
        wxPanel.__init__( self, parent, -1 )
        self.parent = parent
        self.handler = nodehandler
        self.kit = orpg.tools.scriptkit.scriptkit( self.handler.myopenrpg )
        self.myeditor = None
        self.session = self.handler.myopenrpg.get_component(COMPONENT_SESSION)
        self.settings = self.handler.myopenrpg.get_component(COMPONENT_SETTINGS)
        self.chat = self.handler.myopenrpg.get_component(COMPONENT_CHAT)
        self.r_h = orpg.tools.rgbhex.RGBHex()
        self.histidx = 0
        self.history = []
        self.lasthistevt = None
        self.create_gui()
        self.update_alias_listbox()
        self.on_alias_change( None )

        # Heroman - Create Menubar.
        menuBar=wxMenuBar()
        dialogMenu=wxMenu()
        dialogMenu.Append(MENU_DIALOG_CLOSE,"Close","Close Alias Library")
        EVT_MENU(self.parent, MENU_DIALOG_CLOSE,self.parent.OnCloseWindow)
        menuBar.Append(dialogMenu,"Dialog")
        aliasMenu=wxMenu()
        aliasMenu.Append(MENU_ALIAS_ADD_TEMP,"Add temporary",TOOLTIP_1)
        EVT_MENU(self.parent, MENU_ALIAS_ADD_TEMP,self.on_update_list_action)
        aliasMenu.Append(MENU_ALIAS_ADD,"Add",TOOLTIP_2)
        EVT_MENU(self.parent, MENU_ALIAS_ADD,self.add_alias)
        aliasMenu.Append(MENU_ALIAS_DEL,"Delete",TOOLTIP_3)
        EVT_MENU(self.parent, MENU_ALIAS_DEL,self.del_alias)
        aliasMenu.Append(MENU_ALIAS_EDIT,"Edit",TOOLTIP_4)
        EVT_MENU(self.parent, MENU_ALIAS_EDIT,self.edit_alias)
        menuBar.Append(aliasMenu,"Alias")

        filterMenu=wxMenu()
        filterMenu.Append(MENU_FILTER_EXPORT,"Export",TOOLTIP_8)
        EVT_MENU(self.parent, MENU_FILTER_EXPORT,self.on_export_filter_action)
        filterMenu.Append(MENU_FILTER_ADD,"Add",TOOLTIP_5)
        EVT_MENU(self.parent, MENU_FILTER_ADD,self.add_filter)
        filterMenu.Append(MENU_FILTER_DEL,"Delete",TOOLTIP_6)
        EVT_MENU(self.parent, MENU_FILTER_DEL,self.del_filter)
        filterMenu.Append(MENU_FILTER_EDIT,"Edit",TOOLTIP_7)
        EVT_MENU(self.parent, MENU_FILTER_EDIT,self.edit_filter)
        menuBar.Append(filterMenu,"Filter")

        transmitMenu=wxMenu()
        transmitMenu.Append(MENU_TRANSMIT_SEND, "&Send\tCtrl+S", "Send message")
        EVT_MENU(self.parent, MENU_TRANSMIT_SEND, self.on_chat_action)
        transmitMenu.Append(MENU_TRANSMIT_EMOTE, "&Emote\tCtrl+E", "Emote message")
        EVT_MENU(self.parent, MENU_TRANSMIT_EMOTE, self.on_emote_action)
        transmitMenu.Append(MENU_TRANSMIT_WHISPER, "&Whispher\tCtrl+W", "Whisper message")
        EVT_MENU(self.parent, MENU_TRANSMIT_WHISPER, self.on_whisper_action)
        transmitMenu.Append(MENU_TRANSMIT_RAW, "Send &Raw\tCtrl+R", "Send message")
        EVT_MENU(self.parent, MENU_TRANSMIT_RAW, self.on_raw_send_action)

        menuBar.Append(transmitMenu,"Transmit")
        self.parent.SetMenuBar(menuBar)


    def update_alias_listbox( self ):
        self.aliaslist.Clear()
        names = self.handler.refresh_names()
        names.sort()
        for name in names:
            self.aliaslist.Append( name )
        self.aliaslist.Refresh()
            #self.aliaslist.Append( name )

    def import_map_minis( self ):
        self.aliaslist.Clear()
        list = self.handler.refresh_names()
        names = []
        for name in list:
            names.append( name )
        map = self.handler.myopenrpg.get_component( COMPONENT_MAP )
        minis = map.canvas.layers[LAYER_MINIATURES].miniatures
        for min in minis:
            name = min._protect_label[0]
            if name not in names:
                names.append( name )
        names.sort()
        for name in names:
            self.aliaslist.Append( name )
        self.aliaslist.Refresh()            
    
    def update_filter_listbox( self ):
        self.filterlist.Clear()
        list = self.handler.refresh_filters()
        list.sort()
        for name in list:
            self.filterlist.Append( name )
        self.filterlist.Refresh()            

    # def create_button( self, image, tooltip, id, mask_color=wxWHITE, image_type=wxBITMAP_TYPE_GIF ):
    #     gif = wxImage( image, image_type ).ConvertToBitmap()
    #     mask = wxMaskColour( gif, mask_color )
    #     gif.SetMask( mask )
    #     btn = wxBitmapButton(self, id, gif )
    #     btn.SetToolTip( wxToolTip( tooltip ) )
    #     return btn

    def create_gui( self ):
        splitter = mySplitter( self, -1 ) #1wxSplitterWindow( self, -1 )
        splitter.target = self
        splitter1 = wxSplitterWindow( splitter, -1 )
        self.aliaslist = wxListBox( splitter1, ALIAS_USE_LIST, ( 10, 10 ), ( 50, 50 ), self.handler.refresh_names(), wxLB_SINGLE )
        self.filterlist = wxListBox( splitter1, FILTER_USE_LIST, ( 10, 10 ), ( 50, 50 ), self.handler.refresh_filters(), wxLB_SINGLE )
        splitter1.SplitHorizontally( self.aliaslist, self.filterlist )
        
        self.inputbuf = orpg.tools.predTextCtrl.predTextCtrl( splitter, -1, "", style=orpg.tools.predTextCtrl.wxTE_PROCESS_TAB | orpg.tools.predTextCtrl.wxTE_MULTILINE , keyHook = self.chat.myKeyHook )
        self.splitter = splitter
        self.splitter.SplitVertically( splitter1, self.inputbuf )#self.aliaslist, self.inputbuf )
        self.splitter.SetSashPosition( 100 )
        splitter1.SetSashPosition( 150 )

        self.useFilter = wxCheckBox( self, FILTER_USE, LABEL_FILTER_TEXT )
        if self.handler.get_use_filter():
            self.useFilter.SetValue( 1 )

        self.autoClearText = wxCheckBox( self, AUTO_CLEAR_USE, LABEL_AUTO_CLEAR_TEXT )

        buttonbarsizer = wxBoxSizer( wxHORIZONTAL )
        buttonbarsizer.Add( createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'install.gif', TOOLTIP_1, UPDATE_LIST_ACTION, '#c0c0c0' ) )
        buttonbarsizer.Add( createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'folder.gif', TOOLTIP_8, EXPORT_FILTER_ACTION ) )
        buttonbarsizer.Add( 2, 2 )
        buttonbarsizer.Add( createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'player.gif', TOOLTIP_2, ADD_ALIAS ) )
        buttonbarsizer.Add( createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'noplayer.gif', TOOLTIP_3, DEL_ALIAS ) )
        buttonbarsizer.Add( createMaskedButton(self, orpg.dirpath.dir_struct["icon"]+'questionhead.gif', TOOLTIP_4, EDIT_ALIAS ) )
        buttonbarsizer.Add( 2, 2 )
        buttonbarsizer.Add( createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'add_filter.gif', TOOLTIP_5, ADD_FILTER, wxBLUE ) )
        buttonbarsizer.Add( createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'delete_filter.gif', TOOLTIP_6, DEL_FILTER, wxBLUE) )
        buttonbarsizer.Add( createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'edit_filter.gif', TOOLTIP_7, EDIT_FILTER, wxRED) )
        buttonbarsizer.Add( 2, 2 )
        buttonbarsizer.Add( createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'bold.gif', '', BOLD_TEXT, '#bdbdbd') )
        buttonbarsizer.Add( createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'italic.gif', '', ITAL_TEXT, '#bdbdbd' ) )
        buttonbarsizer.Add( createMaskedButton( self, orpg.dirpath.dir_struct["icon"]+'underlined.gif', '', UNDR_TEXT, '#bdbdbd' ) )

        colorButton = wxButton( self, COLOR_TEXT, "C" )
        colorButton.SetBackgroundColour( wxBLACK )
        colorButton.SetForegroundColour( wxBLUE )
        buttonbarsizer.Add( colorButton, 0, wxEXPAND )

        buttonsizer = wxBoxSizer( wxHORIZONTAL )
        btn = wxButton( self, CHAT_ACTION, LABEL_SAY )
        # btn.SetDefault()
        buttonsizer.Add( btn, 0, wxEXPAND )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, EMOTE_ACTION, LABEL_DO ), 0, wxEXPAND )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, WHISPER_ACTION, LABEL_WHISPER ), 0, wxEXPAND )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, RAW_SEND_ACTION, LABEL_SEND_RAW ), 0, wxEXPAND )
        buttonsizer.Add( 2, 2 )
        buttonsizer.Add( wxButton( self, DONE_ACTION, LABEL_DONE ), 0, wxEXPAND )
        
        self.sizer = wxBoxSizer( wxVERTICAL )
        self.sizer.Add( buttonbarsizer, 0, wxEXPAND )
        self.sizer.Add( 2, 2 )
        self.sizer.Add( self.splitter, 1, wxEXPAND )
        self.sizer.Add( 2, 2 )
        self.sizer.Add( self.useFilter )
        self.sizer.Add( 100, 2 )
        self.sizer.Add( self.autoClearText )
        self.sizer.Add( 2, 2 )
        self.sizer.Add( buttonsizer, 0, wxEXPAND )
        EVT_SIZE( self, self.on_size )
        EVT_BUTTON( self, UPDATE_LIST_ACTION, self.on_update_list_action )
        EVT_BUTTON( self, EXPORT_FILTER_ACTION, self.on_export_filter_action )
        EVT_BUTTON( self, ADD_ALIAS, self.add_alias )
        EVT_BUTTON( self, DEL_ALIAS, self.del_alias )
        EVT_BUTTON( self, EDIT_ALIAS, self.edit_alias )
        EVT_BUTTON( self, ADD_FILTER, self.add_filter )
        EVT_BUTTON( self, DEL_FILTER, self.del_filter )
        EVT_BUTTON( self, EDIT_FILTER, self.edit_filter )
        EVT_BUTTON( self, BOLD_TEXT, self.on_format_text ) 
        EVT_BUTTON( self, ITAL_TEXT, self.on_format_text ) 
        EVT_BUTTON( self, UNDR_TEXT, self.on_format_text ) 
        EVT_BUTTON( self, COLOR_TEXT, self.on_text_color )
        EVT_BUTTON( self, CHAT_ACTION, self.on_chat_action )
        EVT_BUTTON( self, EMOTE_ACTION, self.on_emote_action )
        EVT_BUTTON( self, WHISPER_ACTION, self.on_whisper_action )
        EVT_BUTTON( self, RAW_SEND_ACTION, self.on_raw_send_action )
        EVT_BUTTON( self, DONE_ACTION, self.parent.OnCloseWindow )
        EVT_CHECKBOX( self, FILTER_USE, self.on_filter_use_action )
        EVT_CHAR( self.inputbuf, self.inputbuf.OnChar )
        EVT_LISTBOX( self, ALIAS_USE_LIST, self.on_alias_change )
        EVT_LISTBOX( self, FILTER_USE_LIST, self.on_filter_change )


    def on_update_list_action( self, evt ):
        # self.update_alias_listbox()
        self.import_map_minis()

    def on_export_filter_action( self, evt ):
        f = self.get_selected_filter()
        self.handler.export_filter( f )

    def on_size( self, evt ):
        s = self.GetClientSizeTuple()
        self.sizer.SetDimension( 10, 10, s[0] - 20, s[1] - 20 )


    def get_selected_alias( self ):
        index = self.aliaslist.GetSelection()

        if index > -1:
            alias = self.aliaslist.GetString( index )
        else:
            alias = self.settings.get_setting( SETTINGS_PLAYER )

        return alias


    def get_selected_filter( self ):
        index = self.filterlist.GetSelection()
        filter = self.filterlist.GetString( index )
        return filter

    def clear_chat_text( self ):
        if self.autoClearText.GetValue():
            self.inputbuf.SetValue( '' )

    def filter_text( self, text ):
        # if self.handler.get_use_filter():
        if self.useFilter.GetValue():
            list = self.handler.master_dom.getElementsByTagName( VOXCHAT_FILTER )
            index = self.filterlist.GetSelection()
            fname = self.filterlist.GetString( index )
            for node in list:
                if node.getAttribute( ATTRIBUTE_NAME ) == fname:
                    filter = []
                    list = node.getElementsByTagName( VOXCHAT_FILTER_RULE )
                    for rule in list:
                        match = rule.getAttribute( ATTRIBUTE_MATCH )
                        sub = rule.getAttribute( ATTRIBUTE_SUB )
                        text = re.sub( match, sub, text )
                    return text
        return text

    def on_chat_action( self, evt ):
        self.do_chat( self.get_selected_alias(), self.get_chat_text() )

    def on_emote_action( self, evt ):
        self.do_emote( self.get_selected_alias(), self.get_chat_text() )

    def on_whisper_action( self, evt ):
        self.do_whisper( self.get_selected_alias(), self.get_chat_text() )

    def on_raw_send_action( self, evt ):
        self.do_raw_send( self.get_chat_text() )
        
    def on_filter_use_action( self, evt ):
        alias = self.get_selected_alias()
        self.handler.set_alias_does_filter( alias, `self.useFilter.GetValue()` )
        # self.handler.set_use_filter( self.useFilter.GetValue() )


    def on_alias_change( self, evt ):
        alias = self.get_selected_alias()
        filter = self.handler.get_alias_filter_name( alias )
        index = self.filterlist.FindString( filter )
        # if self.filterlist.FindString( filter ):
        if index > -1:
            self.filterlist.SetSelection( index )
            value = self.handler.get_alias_does_filter( alias )
            if value == '1':
                self.useFilter.SetValue( 1 )
            else:
                self.useFilter.SetValue( 0 )
            return
        self.useFilter.SetValue( 0 )
        #self.filterlist.Deselect( self.filterlist.GetSelection() )
  
    def on_filter_change( self, evt ):
        alias = self.get_selected_alias()
        filter = self.get_selected_filter()
        self.handler.set_alias_filter( alias, filter )

    def am_myself( self, alias ):
        player = self.settings.get_setting(SETTINGS_PLAYER)
        return player == alias

    def break_text( self, text ):
        return string.replace( string.strip( text ), NEWLINE, '<br>' )

    def norm_alias( self, alias ):
        if not self.am_myself( alias ):
            return ANTI_SPOOF + alias
        return alias            


    def do_chat( self, alias, text ):
        self.add_to_history( text )
        text = self.filter_text( text )
        text = self.break_text( text )
        self.kit.sendToChatAs( self.norm_alias( alias ), text )
        self.clear_chat_text()


    def do_emote( self, alias, text ):
        self.add_to_history( text )
        text = self.filter_text( text )
        text = self.break_text( text )
        self.kit.emoteToChatAs( self.norm_alias( alias ), text )
        self.clear_chat_text()

    # Heroman - added to strip html from whisper select list
    def strip_html(self,player):
        ret_string = ""
        x = 0
        in_tag = 0

        for x in range(len(player[0])) :
            if player[0][x] == "<" or player[0][x] == ">" or in_tag == 1 :
                if player[0][x] == "<" :
                    in_tag = 1
                elif player[0][x] == ">" :
                    in_tag = 0
                else :
                    pass
            else :
                ret_string = ret_string + player[0][x]
        return ret_string


    def do_whisper( self, alias, text ):
        self.add_to_history( text )
        text = self.filter_text( text )
        session = self.handler.myopenrpg.get_component(COMPONENT_SESSION)
        players = session.get_players()
        opts = []
        target = []
        myid = session.get_id()
        me = None
        for p in players:
            if p[2] != myid:
                opts.append("("+p[2]+") " +self.strip_html(p))
                target.append( p[2] )
            else:
                me = p

        if len(opts):
            players.remove(me)

            dlg = wxMultiCheckBoxDlg( self.GetParent(),opts,"Select Players:","Send To", [] )
            if dlg.ShowModal() == wxID_OK:
                who = ''
                for s in dlg.get_selections():
                    if len( who ) > 0:
                        who = who + ", " + target[ s ]
                    else:
                        who = target[ s ]
                self.kit.whisperToChatAs( who, self.norm_alias( alias ), text )
                self.clear_chat_text()



    def do_raw_send( self, text ):
        self.add_to_history( text )
        name = self.settings.get_setting(SETTINGS_PLAYER)
        lines = string.split( string.strip( text ), NEWLINE )
        for line in lines:
            self.kit.sendToChat( line )

        self.kit.become( name )
        self.clear_chat_text()


    def add_alias( self, evt ):
        result = self.get_input( 'Enter the new alias', 'Add alias', '' )
        if result != None:
            self.do_add_alias( result )


    def del_alias( self, evt ):
        index = self.aliaslist.GetSelection()
        alias = self.aliaslist.GetString( index )
        dlg = wxMessageDialog( self, 'Really delete alias '+alias+'?', 'Confirm alias deletion', wxYES_NO | wxICON_QUESTION | wxNO_DEFAULT )
        if dlg.ShowModal() == wxID_YES:
            self.do_del_alias( alias )
        dlg.Destroy()


    def edit_alias( self, evt ):
        index = self.aliaslist.GetSelection()
        alias = self.aliaslist.GetString( index )
        self.do_edit_alias( alias )


    def do_add_alias( self, alias ):
        if alias not in self.handler.refresh_names():
            self.handler.add_alias( alias )
        else:
            dlg = wxMessageDialog( self, alias+' already appears in the alias list\nYou cannot add duplicates', 'Duplicate entry found', wxOK )
            dlg.ShowModal()
            dlg.Destroy()


    def do_del_alias( self, alias ):
        if alias in self.handler.refresh_names():
            self.handler.del_alias( alias )
        else: # alias is a temporary one
            print 'deleting temporary alias'
            index = self.aliaslist.FindString( alias )
            self.aliaslist.Delete( index )
            self.aliaslist.Refresh()
            

    def do_edit_alias( self, alias ):
        result = self.get_input( 'Enter the new value for '+alias, 'Edit alias', alias )
        if result != None:
            if result in self.handler.refresh_names():
                dlg = wxMessageDialog( self, result+' already appears in the alias list\nYou cannot add duplicates', 'Duplicate entry found', wxOK )
                dlg.ShowModal()
                dlg.Destroy()
            elif alias in self.handler.refresh_names():
                self.handler.change_alias( alias, result, 0 )
                index = self.aliaslist.GetSelection()
                self.aliaslist.SetString( index, result )
                self.aliaslist.Refresh()
            else:
                index = self.aliaslist.GetSelection()
                self.aliaslist.SetString( index, result )
                self.aliaslist.Refresh()

    def get_input( self, message, title, default ):
        dlg = wxTextEntryDialog( self, message, title, default )
        result = None
        if dlg.ShowModal() == wxID_OK:
            result = dlg.GetValue()
        dlg.Destroy()
        return result

    def add_filter( self, evt ):
        result = self.get_input( 'Enter a name for your new filter', 'New Filter', '' )
        if result != None:
            self.do_add_filter( result )

    def del_filter( self, evt ):
        index = self.filterlist.GetSelection()
        filter = self.filterlist.GetString( index )
        dlg = wxMessageDialog( self, 'Really delete filter '+filter+'?', 'Confirm filter deletion', wxYES_NO | wxICON_QUESTION | wxNO_DEFAULT )
        if dlg.ShowModal() == wxID_YES:
            self.do_del_filter( filter )
        dlg.Destroy()

    def edit_filter( self, evt ):
        index = self.filterlist.GetSelection()
        name = self.filterlist.GetString( index )
        self.do_edit_filter( name )
    
    def do_add_filter( self, name ):
        if name not in self.handler.filters:
            self.handler.add_filter( name )
        else:
            dlg = wxMessageDialog( self, name+' already appears in the filter list\nYou cannot add duplicates', 'Duplicate entry found', wxOK )
            dlg.ShowModal()
            dlg.Destroy()
        
    def do_del_filter( self, name ):
        self.handler.del_filter( name )

    def do_edit_filter( self, name ):
        filters = self.handler.master_dom.getElementsByTagName( VOXCHAT_FILTER )
        for filter in filters:
            if filter.getAttribute( ATTRIBUTE_NAME ) == name :
                if self.myeditor == None or self.myeditor.destroyed == 1:
                    self.myeditor = wxPFrame( self, name )
                    wnd = voxchat_filter_edit_panel( self.myeditor, self.handler, filter )
                    self.myeditor.panel = wnd
                    self.wnd = wnd
                    self.myeditor.Show( 1 )
                else:
                    self.myeditor.Raise()
                break


    # This subroutine sets the color of selected text, or base text color if
    # nothing is selected
    def on_text_color( self, event ):
        hexcolor = self.r_h.do_hex_color_dlg(self)
        if hexcolor != None:
            (beg,end) = self.inputbuf.GetSelection()
            txt = self.inputbuf.GetValue()
            if beg == end:
                beg = 0
                end = len( txt )
                
            txt = txt[:beg] + '<font color=' + hexcolor + '>' + txt[beg:end] + '</font>' + txt[end:] 
            self.inputbuf.SetValue(txt)
            self.inputbuf.SetInsertionPointEnd()
            self.inputbuf.SetFocus()
    # def on_text_color - end


    def on_format_text(self,event):
        id = event.GetId()
        txt = self.inputbuf.GetValue()
        (beg,end) = self.inputbuf.GetSelection()
        if beg != end:
            sel_txt = txt[beg:end]
        else:
            sel_txt = txt

        if id == BOLD_TEXT:
            sel_txt = "<b>" + sel_txt + "</b>"
        elif id == ITAL_TEXT:
            sel_txt = "<i>" + sel_txt + "</i>"
        elif id == UNDR_TEXT:
            sel_txt = "<u>" + sel_txt + "</u>"

        if beg != end:
            txt = txt[:beg] + sel_txt + txt[end:]
        else:
            txt = sel_txt

        self.inputbuf.SetValue(txt)
        self.inputbuf.SetInsertionPointEnd()
        self.inputbuf.SetFocus()

    # def OnChar( self, evt ):
    #     print evt.KeyCode()
    #     print evt.AltDown()
    #     print evt.ControlDown()
    #     evt.Skip()
    #     pass


    def get_chat_text( self ):
        s = self.inputbuf.GetValue()
        self.histlen = len( self.history ) - 1
        return s
   
    def add_to_history( self, text ):
        self.histidx = 0
        self.history.append( text )
        self.lasthistevt = None

    def OnChar(self,event):
        # s = self.inputbuf.GetValue()
        # self.histlen = len(self.history) - 1
        s = self.get_chat_text()

        ## RETURN KEY (no matter if there is text in inputbuf)
        #  This section is run even if there is nothing in the inputbuf (as opposed to the next WXK_RETURN handler
        if event.KeyCode() == WXK_RETURN:
            if self.session.get_status() == orpg.networking.mplay_client.MPLAY_CONNECTED:          #  only do if we're connected
                self.chat.sendTyping(0)                                    #  Send a "not_typing" event on enter key press

        ## RETURN KEY (and not text in control)
        if event.KeyCode() == WXK_RETURN and s != "" and event.ControlDown():
            # self.histidx=0
            # self.history.append(s)
            # self.lasthistevt=None
            self.add_to_history( s )

            if s[0] != "/": ## it's not a slash command
                self.on_chat_action( None )
                # s = self.ParsePost(self.colorize(self.mytextcolor, s),true,true)
                # self.inputbuf.SetValue("")
                # #self.session.send(s)
            else:
                self.on_emote_action( None )
                # s = self.ParseDice(s)
                # self.emote.docmd(s) # emote is in chatutils.py
                # self.inputbuf.SetValue("")


            # playe sound
            sound_file = self.settings.get_setting( SETTINGS_SEND_SOUND )
            sound_player = orpg.tools.orpg_player.orpg_player(self.settings.get_setting("UnixSoundPlayer"));
            sound_player.play(sound_file)
            

        ## UP KEY
        elif event.KeyCode() == WXK_UP and event.ControlDown() and self.histlen >= 0:
            if self.lasthistevt == 'dn' and self.histlen != self.histidx:
                self.histidx = self.histidx + 1
            try:
                histdata = self.history[self.histlen - self.histidx]
                self.inputbuf.SetValue(histdata)
                self.inputbuf.SetInsertionPointEnd()
                self.lasthistevt='up'
            except:
                print "bad value in the up key: histidx = %s" % self.histidx
            if self.histidx < self.histlen:
                self.histidx = self.histidx + 1
            else:
                self.lasthistevt=None

        ## DOWN KEY
        elif event.KeyCode() == WXK_DOWN and event.ControlDown() and self.histlen >= 0:
            if self.histidx > 0:
                if self.lasthistevt == 'up' and self.histidx != 0:
                    self.histidx = self.histidx - 1
                self.histidx = self.histidx - 1
                try:
                    histdata = self.history[self.histlen - self.histidx]
                    self.inputbuf.SetValue(histdata)
                    self.inputbuf.SetInsertionPointEnd()
                    self.lasthistevt='dn'
                except:
                    print "bad value in the down key: histidx = %s" % self.histidx
            elif self.histidx == 0:
                self.inputbuf.SetValue("")
                self.lasthistevt=None

        ## TAB KEY
        elif  event.KeyCode() == WXK_TAB:
            # Heroman - updated to select name better
            if s !="":
                # If beg=end, grab the first fullword behind the cursor
                (beg,end) = self.inputbuf.GetSelection()
                if beg == end:
                    while beg > 0 and s[beg-1:beg] != " ":
                        beg = beg - 1
                if beg == -1:
                    beg=0
                matchName=s[beg:end]
                
                found = 0
                nicks = []
                testnick = ""
                inlength = len(matchName)
                for getnames in self.session.players.keys():
                    testnick = self.strip_html(self.session.players[getnames]);
                    if string.lower(matchName) == string.lower(testnick[:inlength]):
                        found = found + 1
                        nicks[len(nicks):]=[testnick]
                if found == 0: ## no nick match
                    self.kit.chatMessage(" ** No match found")
                elif found > 1: ## matched more than 1, tell user what matched
                    nickstring = ""
                    for foundnicks in nicks:
                        nickstring = nickstring + foundnicks + ", "
                    nickstring = nickstring[:-2]
                    self.kit.chatMessage(" ** Multiple matches found: " + nickstring)
                else: ## put the matched name in the inputbuf box
                    settext = s[:beg] + nicks[0] + s[end:]
                    
                    self.inputbuf.SetValue(settext)
                    self.inputbuf.SetInsertionPointEnd()
            else: ## not online, and no text in inputbuf box
                self.kit.chatMessage(" ** That's the Tab key, Dave")
        # ## PAGE UP
        # elif event.KeyCode() == WXK_PRIOR and event.ControlDown():
        #     if self.bufferpointer < len(self.chatbuffer)-self.buffersize and len(self.chatbuffer) > self.buffersize:
        #         self.bufferpointer = self.bufferpointer + self.buffersize
        #         self.do_chat_action( None )
        #         # self.Post()
        #     else:
        #         event.Skip()
        # ## PAGE DOWN
        # elif event.KeyCode() == WXK_NEXT and event.ControlDown():
        #     if self.bufferpointer > 0:
        #         self.bufferpointer = self.bufferpointer - self.buffersize
        #         # self.Post()
        #         self.do_chat_action( None )
        #     else:
        #         event.Skip()
        #
        # ## END
        # elif event.KeyCode() == WXK_END and event.ControlDown():
        #     self.bufferpointer = 0
        #     self.do_chat_action( None )
        #     # self.Post()
        #     event.Skip()

        ## NOTHING
        else:
            event.Skip()
    # def OnChar - end
