//                                               -*- C++ -*-
/**
 * @file    CalculationC3.cxx
 * @brief
 *
 * @author  Romuald Conty
 * @date    2006-09-07 11:43:41
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2008-09-22 11:34:11 +0200 (lun 22 sep 2008) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "CalculationC3.hxx"

// OT::UI::GUI
#include "Parameter.hxx"
#include "BlockC3.hxx"
#include "Result.hxx"

// Qt
#include <qvariant.h>
#include <qdatetime.h>

// std
#include <iostream>

// OT
#include "OT.hxx"

#include "NumericalMathFunction.hxx"
#include "Collection.hxx"
#include "Pointer.hxx"
#include "NumericalSample.hxx"
#include "Matrix.hxx"
#include "SymmetricTensor.hxx"
#include "IdentityMatrix.hxx"
#include "Distribution.hxx"
#include "Normal.hxx"
#include "NumericalMathFunction.hxx"
#include "LinearNumericalMathEvaluationImplementation.hxx"
#include "ConstantNumericalMathGradientImplementation.hxx"
#include "ConstantNumericalMathHessianImplementation.hxx"

#include "ComparisonOperatorImplementation.hxx"

#include "RandomVector.hxx"
#include "Event.hxx"

// Simulation
#include "MonteCarlo.hxx"
#include "LHS.hxx"
#include "ImportanceSampling.hxx"
#include "DirectionalSampling.hxx"

// Optimization
#include "NearestPointAlgorithm.hxx"

// Analytical
#include "FORM.hxx"
#include "FORMResult.hxx"
#include "SORM.hxx"
#include "SORMResult.hxx"

#include "StrongMaximumTest.hxx"

#define HAVE_PTHREAD_H
#define HAVE_SEMAPHORE_H
#include "Log.hxx"

typedef OT::Bool                        Bool;
typedef OT::UnsignedLong                UnsignedLong;
typedef OT::NumericalScalar             NumericalScalar;

typedef OT::Base::Type::SymmetricTensor SymmetricTensor;
typedef OT::Base::Type::NumericalPoint  NumericalPoint;

// We define a Pair structure, which should be available in OT::Base::Common as a generic
// class
struct Pair
{
	QString name_;
	UnsignedLong index_;
	Pair() : name_(), index_()
	{}
	Pair ( QString name, UnsignedLong index ) : name_ ( name ), index_ ( index )
{}}
;
typedef OT::Base::Type::Collection<Pair>                            PairCollection;
typedef OT::Base::Type::IdentityMatrix                              IdentityMatrix;
typedef OT::Base::Stat::NumericalSample                             NumericalSample;
typedef OT::Uncertainty::Model::Distribution                        Distribution;
typedef OT::Uncertainty::Distribution::Normal                       Normal;
typedef OT::Base::Type::Collection<Distribution>                    DistributionCollection;
typedef OT::Base::Func::NumericalMathFunction                       NumericalMathFunction;
typedef OT::Base::Func::LinearNumericalMathEvaluationImplementation   LinearNumericalMathEvaluationImplementation;
typedef OT::Base::Func::ConstantNumericalMathGradientImplementation ConstantNumericalMathGradientImplementation;
typedef OT::Base::Func::ConstantNumericalMathHessianImplementation  ConstantNumericalMathHessianImplementation;

typedef OT::Base::Common::Log                                       Log;

typedef OT::Uncertainty::Model::RandomVector                        RandomVector;
typedef OT::Uncertainty::Model::Event                               Event;
typedef OT::Uncertainty::Model::StandardEvent                       StandardEvent;

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{

			typedef OT::Uncertainty::Algorithm::MonteCarlo          MonteCarlo;
			typedef OT::Uncertainty::Algorithm::LHS                 LHS;
			typedef OT::Uncertainty::Algorithm::ImportanceSampling  ImportanceSampling;
			typedef OT::Uncertainty::Algorithm::DirectionalSampling DirectionalSampling;

			typedef OT::Uncertainty::Algorithm::FORM                FORM;
			typedef OT::Uncertainty::Algorithm::FORMResult          FORMResult;
			typedef OT::Uncertainty::Algorithm::SORM                SORM;
			typedef OT::Uncertainty::Algorithm::SORMResult          SORMResult;

			typedef OT::Uncertainty::Algorithm::Simulation::Result  SimulationResult;
			typedef OT::Uncertainty::Algorithm::StrongMaximumTest   StrongMaximumTest;

			typedef OT::Base::Optimisation::NearestPointAlgorithm   NearestPointAlgorithm;

			CalculationC3::CalculationC3 ( QOTObject* parent, const char* name ) : Calculation ( parent, name )
			{
				setTitle ( "Threshold exceedance probability" );

				connect ( this, SIGNAL ( postprocessing() ), this, SLOT ( fillResult() ) );
				connect ( this, SIGNAL ( preprocessing() ), this, SLOT ( flushResult() ) );

				thresholdExceedanceMethodProperty_ = new ThresholdExceedanceMethodProperty ( this, "method" );
				thresholdExceedanceMethodProperty_->setDefaultChoice();
			}

			CalculationC3::~CalculationC3()
			{}

			int CalculationC3::processing()
			{
				int errorCode = 0;

				BlockC3* blockC3 = ( BlockC3* ) parent();
				Q_CHECK_PTR ( blockC3 );
				if ( blockC3 == NULL )
				{
					qFatal ( "Unable to find C3 block..." );
				}

				FinalVariable* finalVariable = blockC3->getFinalVariable_ptr();
				Q_CHECK_PTR ( finalVariable );
				if ( blockC3 == NULL )
				{
					qFatal ( "Unable to find final variable..." );
				}

				BlockB* blockB = blockC3->getBlockB_ptr();
				Q_CHECK_PTR ( blockC3 );
				if ( blockC3 == NULL )
				{
					qFatal ( "Unable to find B block..." );
				}

				Log::Show ( Log::INFO | Log::WARN | Log::ERROR );

				// General comment: the connection with the OpenTURNS library seems to be a hard way!
				// It is mainly due to the fact that in the GUI, we create a king of ghost of the entry
				// variables in the first step of part A, but we have to wait until the end of part B to
				// know their nature and their distribution. If the GUI has followed the library logic,
				// all the manipulations done here could have been made on the fly of the study case
				// construction... Nevertheless, it is possible to do it, but I will only give
				// indications to do it at the end of this method: I don't have the time to look over
				// the whole GUI to see where the several parts have to take place! Don't forget to have
				// a look at the comments following the code.

				// Step A.2.2:
				// Get the ComparisonOperator and the threshold. To be done.
				const ThresholdExceedance* thresholdExceedance = blockC3->getThresholdExceedance_ptr();

				NumericalScalar threshold ( thresholdExceedance->getThreshold() );
				std::cout << "threshold=" << threshold << std::endl;

				ComparisonOperator comparisonOperator ( thresholdExceedance->getOperator() );
				std::cout << "comparison operator=" << comparisonOperator.str() << std::endl;

				RandomVector outputVector = createOutputRandomVector ( finalVariable, blockB );

				// We build the event from the output vector, the comparison operator and the threshold
				Event failureEvent ( outputVector, comparisonOperator, threshold );
				std::cout << "failureEvent=" << failureEvent << std::endl;

				StandardEvent stdEvent ( failureEvent );
				QDateTime startDateTime ( QDateTime::currentDateTime() );

				const QString method = thresholdExceedanceMethodProperty_->getMethod();

				if ( method == "monte carlo" )
				{
					// We build the MonteCarlo algorithm
					MonteCarlo monteCarlo ( failureEvent );
					std::cout << "monteCarlo=" << monteCarlo.str() << std::endl;
					// We change the value of the algorithm parameters
					// Maximum number of shoots
					monteCarlo.setMaximumOuterSampling ( thresholdExceedanceMethodProperty_->getShootCount() );
					// Set maximum calculation time
					//monteCarlo.setCalculationTime( methodOptions_[ 1 ].toMap() [ "shoot_count" ].toUInt() );
					// Maximum coefficient of variation
					monteCarlo.setMaximumCoefficientOfVariation ( thresholdExceedanceMethodProperty_->getPrecision() );
					std::cout << "monteCarlo=" << monteCarlo.str() << std::endl;
					// Perform the actual computation
					monteCarlo.run();
					// Get the result and print it to std::cout
					SimulationResult mcResult ( monteCarlo.getResult() );

					std::cout << "Result=" << mcResult.str() << std::endl;
					new Result ( this, "MonteCarlo", startDateTime, QDateTime::currentDateTime(), mcResult );

				}
				else if ( method == "LHS" )
				{
					/* We create a LHS algorithm */
					LHS lhs ( failureEvent );
					lhs.setMaximumOuterSampling ( thresholdExceedanceMethodProperty_->getShootCount() );
					lhs.setBlockSize ( 4 );
					std::cout << "LHS=" << lhs.str() << std::endl;
					/* Perform the simulation */
					lhs.run();
					/* Stream out the result */
					// std::cout << "LHS result=" << myAlgo.getResult().str() << std::endl;
					SimulationResult lhsResult = lhs.getResult();

					std::cout << "Result=" << lhsResult.str() << std::endl;
					new Result ( this, "LHS", startDateTime, QDateTime::currentDateTime(), lhsResult );

				}
				else if ( method == "FORM" )
				{
					NumericalPoint defaultStartingPoint ( failureEvent.getImplementation() ->getAntecedent() ->getDistribution().computeQuantile ( .5 ) );

					NearestPointAlgorithm nearestPointAlgorithm ( thresholdExceedanceMethodProperty_->getNearestPointAlgorithm() );

					/* We create a FORM algorithm */
					/* The first parameter is a NearestPointAlgorithm */
					/* The second parameter is an event */
					/* The third parameter is a starting point for the design point research */
					FORM form ( nearestPointAlgorithm, failureEvent, defaultStartingPoint );
					std::cout << "form=" << form.str() << std::endl;
					/* Perform the simulation */
					form.run();
					/* Stream out the result */
					FORMResult formResult ( form.getResult() );
					new Result ( this, "FORM", startDateTime, QDateTime::currentDateTime(), formResult );

					StrongMaximumTestProperty* strongMaximumTestProperty_ = thresholdExceedanceMethodProperty_->getStrongMaximumTextProperty();
					if ( strongMaximumTestProperty_->isEnabled() )
					{
						startDateTime = QDateTime::currentDateTime();
						StrongMaximumTest strongMaximumTest ( stdEvent,
						                                      formResult.getStandardSpaceDesignPoint(),
						                                      strongMaximumTestProperty_->getImportanceLevel(),
						                                      strongMaximumTestProperty_->getAccuracyLevel(),
						                                      strongMaximumTestProperty_->getConfidenceLevel() );
						//            strongMaximumTest.run();
						new Result ( this, "StrongMaximumTest", startDateTime, QDateTime::currentDateTime(), strongMaximumTest );
					}
				}
				else if ( method == "SORM" )
				{
					NumericalPoint defaultStartingPoint ( failureEvent.getImplementation() ->getAntecedent() ->getDistribution().computeQuantile ( .5 ) );

					/* We create a NearestPoint algorithm */
					NearestPointAlgorithm nearestPointAlgorithm ( thresholdExceedanceMethodProperty_->getNearestPointAlgorithm() );

					/* We create a FORM algorithm */
					/* The first parameter is a NearestPointAlgorithm */
					/* The second parameter is an event */
					/* The third parameter is a starting point for the design point research */
					SORM sorm ( nearestPointAlgorithm, failureEvent, defaultStartingPoint );

					std::cout << "sorm=" << sorm.str() << std::endl;
					/* Perform the simulation */
					sorm.run();
					/* Stream out the result */
					SORMResult sormResult ( sorm.getResult() );
					std::cout << "sormResult=" << sormResult.str() << std::endl;
					new Result ( this, "SORM", startDateTime, QDateTime::currentDateTime(), sormResult );

					StrongMaximumTestProperty* strongMaximumTestProperty_ = thresholdExceedanceMethodProperty_->getStrongMaximumTextProperty();
					if ( strongMaximumTestProperty_->isEnabled() )
					{
						startDateTime = QDateTime::currentDateTime();
						StrongMaximumTest strongMaximumTest ( stdEvent,
						                                      sormResult.getStandardSpaceDesignPoint(),
						                                      strongMaximumTestProperty_->getImportanceLevel(),
						                                      strongMaximumTestProperty_->getAccuracyLevel(),
						                                      strongMaximumTestProperty_->getConfidenceLevel() );
						strongMaximumTest.run();
						new Result ( this, "StrongMaximumTest", startDateTime, QDateTime::currentDateTime(), strongMaximumTest );
					}
				}
				else if ( method == "importance sampling" )
				{
					NumericalPoint defaultStartingPoint ( failureEvent.getImplementation() ->getAntecedent() ->getDistribution().computeQuantile ( .5 ) );
					std::cout << "defaultStartingPoint=" << defaultStartingPoint << std::endl;

					/* We create a NearestPoint algorithm */
					NearestPointAlgorithm nearestPointAlgorithm ( thresholdExceedanceMethodProperty_->getNearestPointAlgorithm() ) ;

					/* We create a FORM algorithm */
					/* The first parameter is a NearestPointAlgorithm */
					/* The second parameter is an event */
					/* The third parameter is a starting point for the design point research */
					FORM form ( nearestPointAlgorithm, failureEvent, defaultStartingPoint );
					std::cout << "form=" << form.str() << std::endl;

					/* Perform the simulation */
					form.run();
					/* Stream out the result */
					FORMResult formResult ( form.getResult() );

					UnsignedLong dimension = defaultStartingPoint.getDimension();
					Normal importanceDistribution ( formResult.getStandardSpaceDesignPoint(), NumericalPoint ( dimension, 1. ), IdentityMatrix ( dimension ) );

					StrongMaximumTestProperty* strongMaximumTestProperty_ = thresholdExceedanceMethodProperty_->getStrongMaximumTextProperty();
					if ( strongMaximumTestProperty_->isEnabled() )
					{
						startDateTime = QDateTime::currentDateTime();
						StrongMaximumTest strongMaximumTest ( stdEvent,
						                                      formResult.getStandardSpaceDesignPoint(),
						                                      strongMaximumTestProperty_->getImportanceLevel(),
						                                      strongMaximumTestProperty_->getAccuracyLevel(),
						                                      strongMaximumTestProperty_->getConfidenceLevel() );
						strongMaximumTest.run();
						new Result ( this, "StrongMaximumTest", startDateTime, QDateTime::currentDateTime(), strongMaximumTest );
					}

					ImportanceSampling importanceSampling ( stdEvent, importanceDistribution );
					// Maximum number of shoots
					importanceSampling.setMaximumOuterSampling ( thresholdExceedanceMethodProperty_->getShootCount() );
					// Maximum coefficient of variation
					importanceSampling.setMaximumCoefficientOfVariation ( thresholdExceedanceMethodProperty_->getPrecision() );
					std::cout << "importanceSampling=" << importanceSampling.str() << std::endl;
					importanceSampling.run();

					SimulationResult importanceSamplingResult = importanceSampling.getResult();
					new Result ( this, "ImportanceSampling", startDateTime, QDateTime::currentDateTime(), importanceSamplingResult );
				}
				else if ( method == "directional sampling" )
				{
					DirectionalSampling directionalSampling ( failureEvent );
					std::cout << "directionalSampling=" << directionalSampling.str() << std::endl;
					// We change the value of the algorithm parameters
					// Maximum number of shoots
					directionalSampling.setMaximumOuterSampling ( thresholdExceedanceMethodProperty_->getShootCount() );
					// Set maximum calculation time
					//monteCarlo.setCalculationTime( methodOptions_[ 1 ].toMap() [ "shoot_count" ].toUInt() );
					// Maximum coefficient of variation
					directionalSampling.setMaximumCoefficientOfVariation ( thresholdExceedanceMethodProperty_->getPrecision() );
					std::cout << "directionalSampling=" << directionalSampling.str() << std::endl;
					// Perform the actual computation
					directionalSampling.run();
					// Get the result and print it to std::cout
					SimulationResult dsResult ( directionalSampling.getResult() );

					std::cout << "Result=" << dsResult.str() << std::endl;
					new Result ( this, "DirectionalSampling", startDateTime, QDateTime::currentDateTime(), dsResult );

				}
				qWarning ( QString ( "errorCode=" + QString::number ( errorCode ) ) );
				return errorCode;

			} // calculate()

			void CalculationC3::fillResult()
			{
				emit updated();
			}

			void CalculationC3::flushResult()
			{}
		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

