//                                               -*- C++ -*-
/**
 * @file		QOTObject.hxx
 * @brief		QOTObject is the base class for all non-visuals QOpenTURNS classes.
 *
 * @author	Romuald Conty
 * @date		2006-07-17 09:46:51
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
#ifndef OPENTURNS_GUIQOTOBJECT_HXX
#define OPENTURNS_GUIQOTOBJECT_HXX

// Qt
#include <qobject.h>
#include <qstring.h>
#include <quuid.h>
#include <qobjectlist.h>
#include <qdom.h>
#include <qptrlist.h>

// OT::UI::GUI
#include "Parameter.hxx"
#include "QOTNameProperty.hxx"
#include "QOTAction.hxx"

// OT
#include "Description.hxx"

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{

			typedef OT::Base::Type::Description Description;

			/**
			 * @class		QTOObject
			 * @brief		Top-level class for QOpenTURNS objects.
			 *
			 * All classes derivated from this one can use the Q_PROPERTY mechanism
			 * to input and display values. (i.e. EntryVariable)
			 */
			class QOTObject : public QObject
			{
					Q_OBJECT

					Q_PROPERTY ( QString description READ getDescription WRITE setDescription )
#ifdef QT_DEBUG
					//				Q_PROPERTY( QString uuid READ getUuid )
					Q_PROPERTY ( QString classname READ getClassName )
#endif
				public:
					/** Default constructor */
					QOTObject ( QOTObject *parent, const char *name = 0 );

					/** Copy constructor */
					QOTObject ( const QOTObject & other, QOTObject* parent, const char* name = 0 );

					/** Destructor */
					virtual ~QOTObject();

					/** Assignment */
					QOTObject & operator = ( const QOTObject & other );

					/** QOTObject name accessor */
					virtual bool setName ( const QString & name );

					/** QOTObject name accessor */
					void setName ( const char * name );

					/** QOTObject name accessor */
					QString getName() const;

					/** */
					virtual QString getFullName() const;

					/** QOTObject renameEnabled accessor */
					void setRenameEnabled ( bool renameEnabled );

					/** QOTObject renameEnabled accessor */
					bool getRenameEnabled();

					/** QOTObject description accessor */
					void setDescription ( const QString & description );

					/** QOTObject description accessor */
					QString getDescription() const;

					/** QOTObject uuid accessor (read only)*/
					QString getUuid() const;

					/** QOTObject className accessor */
					QString getClassName() const;

					/** QOTObject short className accessor */
					QString getShortClassName() const;

					/**
					 * @fn static void getChildren( const QObject* startPoint, QObjectList* resultList, const char * inheritsClass, bool recursiveSearch, bool inherits );
					 * @brief search all objects in startPoint's children list that match with the className
					 * @param startPoint search start at this QObject.
					 * @param resultList pointer on already created QObjectList. (see returns)
					 * @param className search criteria.
					 * @param recursiveSearch If recursiveSearch is TRUE, children() performs a depth-first search of the startPoint object's children.
					 * @param inherits If inherits is TRUE all children that inherits className are added to resultList else if FALSE only children that are className are selected.
					 * @return all results that match criteria are appended on resultList.
					 */
					static void getChildren ( const QObject* startPoint, QObjectList* objectList, const char * inheritsClass, bool recursiveSearch, bool inherits );

					/**
					 * @fn void getChildren( QObjectList* resultList, const char * className, bool recursiveSearch = TRUE , bool inherits = TRUE );
					 * @brief search all objects in children list that match with the className
					 * @param resultList pointer on already created QObjectList. (see returns)
					 * @param className search criteria.
					 * @param recursiveSearch If recursiveSearch is TRUE (the default), children() performs a depth-first search of the object's children.
					 * @param inherits If inherits is TRUE (the default) all children that inherits className are added to resultList else if FALSE only children that are className are selected.
					 * @return all results that match criteria are append on resultList.
					 */
					void getChildren ( QObjectList* childrenList, const char * className, bool recursiveSearch = TRUE , bool inherits = TRUE );

					const QObjectList * getChildren () const;

					QOTActions getActions();

					QString giveUniqueChildName ( QString wantedName ) const;

					static QDomElement toXmlNode ( const QOTObject* startPoint, QDomDocument &d );

					QDomElement toXmlNode ( QDomDocument &d );

					QObject* findParent ( const QString& className );

					static QValueList<QVariant> convertParameters ( const Parameters parameters );

					QOTProperties getProperties ( bool recursiveSearch = false );

				public slots:
					void propertyUpdated ( QOTProperty* property );

				signals:
					/** This signal is emitted whenever the name is changed */
					void renamed();
					void updated();
					void updated ( QOTObject* otObject );
					void updated ( QOTProperty* property );

				protected:
					/** The function that actually realizes assignment between objects */
					void assignToMe ( const QOTObject & other );

					/** QOTObject description */
					Description description_;

					/** QOTObject rename enabled */
					bool renameEnabled_;

					QOTProperty* nameProperty_;
				private:
					/** QOTObject Universally Unique Identifier */
					QUuid uuid_;

				private slots:
					void updatedExt();
			}
			; /* class QOTObject */

			typedef QPtrList<QOTObject> QOTObjectList;

		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_GUIQOTOBJECT_HXX */
