//                                               -*- C++ -*-
/**
 * @file    QVisualizationTable.cxx
 * @brief
 *
 * @author  Romuald Conty
 * @date    2006-07-10 14:38:52
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "QVisualizationTable.hxx"

// OT::UI::GUI
#include "QOTObject.hxx"
#include "Block.hxx"
#include "Chapter.hxx"
#include "Content.hxx"
#include "BlockB.hxx"

// Qt
#include <qmemarray.h>

// Visual Components
#include "QOTTableItem.hxx"
#include "QOTTitleTableItem.hxx"
#include "QOTAction.hxx"

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			QVisualizationTable::QVisualizationTable ( QString caption, QWidget * parent, const char * name ) : QTable ( 0, 4, parent, name )
			{
				setCaption ( caption );
				setColumnStretchable ( 2, true );
				setSelectionMode ( QTable::SingleRow );

				horizontalHeader() ->setLabel ( 0, tr ( "uuid" ) );

				verticalHeader() ->hide();
				setLeftMargin ( 0 );
				// #ifndef QT_DEBUG
				hideColumn ( 0 );
				// #endif
				connect ( this, SIGNAL ( clicked ( int, int, int, const QPoint & ) ) , this, SLOT ( clicked ( int, int, int, const QPoint & ) ) );
			}

			QVisualizationTable::~QVisualizationTable()
			{}

			void QVisualizationTable::append ( QOTObject* otObject, bool recursive )
			{
				uint row = numRows();
				if ( !row )
				{
					//first object
					setIcon ( QPixmap::fromMimeSource ( otObject->getShortClassName() + ".png" ) );
					if ( otObject->isA ( "OpenTURNS::UI::GUI::BlockB" ) )
					{
						horizontalHeader() ->setLabel ( 1, tr ( "entry variable" ) );
						horizontalHeader() ->setLabel ( 2, tr ( "distribution" ) );
						horizontalHeader() ->setLabel ( 3, " " );
					}
					else
					{
						horizontalHeader() ->setLabel ( 1, tr ( "name" ) );
						horizontalHeader() ->setLabel ( 2, tr ( "description" ) );
						horizontalHeader() ->setLabel ( 3, " " );
					}
				}
				insertAt ( row - 1 , otObject );
				row++;
				if ( recursive )
				{
					QObjectList * contentList = new QObjectList();
					otObject->getChildren ( contentList, "OpenTURNS::UI::GUI::Content", false, true );
					const uint contentCount = contentList->count();
					for ( uint i = 0; i < contentCount; i++ )
					{
						const QObject* object = contentList->at ( i );
						insertAt ( row - 1 , ( QOTObject* ) object );
						row++;
					}
					delete contentList;

					QObjectList* chapterList = new QObjectList();
					otObject->getChildren ( chapterList, "OpenTURNS::UI::GUI::Chapter", false, true );
					const uint chapterCount = chapterList->count();
					for ( uint i = 0; i < chapterCount; i++ )
					{
						const QObject* object = chapterList->at ( i );
						append ( ( QOTObject* ) object, true );
					}
					delete chapterList;
				}
			}

			void QVisualizationTable::insertAt ( uint row, QOTObject* otObject )
			{
				row++;
				insertRows ( row, 1 );

				setOTTableItem ( row, 0, otObject->getUuid(), otObject );

				if ( otObject->inherits ( "OpenTURNS::UI::GUI::Block" ) )
				{
					Block * block = ( Block* ) otObject;
					setOTTableItem ( row, 1, block->getName(), otObject, true ) ->setSpan ( 1, 2 );
					setCellWidget ( row, 3, createToolbar ( otObject ) );
					adjustRow ( row );
					connect ( block, SIGNAL ( updated ( QOTObject* ) ), this, SLOT ( update ( QOTObject* ) ) );
				}
				else if ( otObject->inherits ( "OpenTURNS::UI::GUI::Chapter" ) )
				{
					Chapter * chapter = ( Chapter* ) otObject;
					Block* block = chapter->getParentBlock();
					Q_CHECK_PTR ( block );
					if ( block != NULL )
					{
						const QString title = chapter->getFullName() + " (" + block->getFullName() + ")";

						setOTTableItem ( row, 1, title, chapter, true ) ->setSpan ( 1, 2 );

						connect ( chapter, SIGNAL ( childCreated ( QOTObject* ) ) , this , SLOT ( add
						          ( QOTObject* ) ) );
						connect ( chapter, SIGNAL ( updated ( QOTObject* ) ), this, SLOT ( update ( QOTObject* ) ) );
						setCellWidget ( row, 3, createToolbar ( otObject ) );
						adjustRow ( row );
					}
				}
				else if ( otObject->isA ( "OpenTURNS::UI::GUI::EntryVariableType" ) )
				{
					EntryVariableType * entryVariableType = ( EntryVariableType* ) otObject;
					QString entryVariableName = "";
					Q_CHECK_PTR ( entryVariableType->getEntryVariable_ptr() );
					if ( entryVariableType->getEntryVariable_ptr() != NULL )
						entryVariableName = entryVariableType->getEntryVariable_ptr() ->getName();
					QString distributionType = entryVariableType->getType();

					setOTTableItem ( row, 1, entryVariableName, otObject );
					setOTTableItem ( row, 2, distributionType, otObject );
					connect ( entryVariableType, SIGNAL ( destroyed ( QOTObject* ) ), this, SLOT ( remove
					          ( QOTObject* ) ) );
					connect ( entryVariableType, SIGNAL ( updated ( QOTObject* ) ), this, SLOT ( update ( QOTObject* ) ) );
					setCellWidget ( row, 3, createToolbar ( otObject ) );
					adjustRow ( row );
				}
				else if ( otObject->inherits ( "OpenTURNS::UI::GUI::Content" ) )
				{
					Content * content = ( Content* ) otObject;

					setOTTableItem ( row, 1, content->getName(), otObject );
					setOTTableItem ( row, 2, content->getDescription(), otObject );

					connect ( content, SIGNAL ( destroyed ( QOTObject* ) ), this, SLOT ( remove
					          ( QOTObject* ) ) );
					connect ( content, SIGNAL ( updated ( QOTObject* ) ), this, SLOT ( update ( QOTObject* ) ) );
					setCellWidget ( row, 3, createToolbar ( otObject ) );
				}
				adjustRow ( row );
				adjustColumn ( 3 );
				adjustColumn ( 1 );
			}

			void QVisualizationTable::clicked ( int row, int col, int button, const QPoint & mousePos )
			{
				if ( col > -1 && row > -1 )
				{
					/** @todo find a better way to mask toolbar's doubleClick (on col = 3)... */
					if ( col != 3 )
					{
						QOTTableItem * otItem = ( QOTTableItem * ) item ( row, col );
						Q_CHECK_PTR ( otItem );
						if ( otItem != NULL )
						{
							emit clicked ( otItem->getQOTObject() );
						}
						else
						{
							qWarning ( "Strange state : item( row, col ) == NULL; where row = " + QString::number ( row ) + ", col = " + QString::number ( col ) + "." );
						}
					}
				}
			}

			int QVisualizationTable::find ( QOTObject* object )
			{
				for ( int i = 0; i < numRows(); i++ )
				{
					if ( text ( i, 0 ) == object->getUuid() )
						return i;
				}
				return -1;
			}

			void QVisualizationTable::remove
			( QOTObject* child )
			{
				Q_CHECK_PTR ( child );
				if ( child != NULL && child->inherits ( "OpenTURNS::UI::GUI::QOTObject" ) )
				{
					int row = find ( ( QOTObject* ) child );
					if ( row >= 0 )
					{
						child->disconnect ( this );
						removeRow ( row );
					}
					else
					{
						qWarning ( "QVisualizationTable::remove() : No child have been found !" );
					}
				}
				else
				{
					if ( child != NULL )
					{
						qWarning ( QString ( QString ( "Strange state : child != NULL but child is an " ) + QString ( child->className() ) + QString ( " !!!" ) ).ascii() );
					}
				}
			}

			void QVisualizationTable::add
			( QOTObject* child )
			{
				Chapter * parent = ( Chapter* ) child->parent();
				int row = find ( parent );
				QObjectList* contentList = new QObjectList();
				for ( uint i = 0; i < parent->getChildrenClassesName() ->count(); i++ )
				{
					parent->getChildren ( contentList, ( *parent->getChildrenClassesName() ) [ i ], false, false );
				}
				row += ( contentList->count() - 1 );
				delete contentList;
				insertAt ( row, child );
			}

			void QVisualizationTable::update ( QOTObject* child )
			{
				int row = find ( child );
				if ( row != -1 )
				{
					if ( child->isA ( "OpenTURNS::UI::GUI::EntryVariableType" ) )
					{
						EntryVariableType * entryVariableType = ( EntryVariableType* ) child;
						QString entryVariableName = "";
						Q_CHECK_PTR ( entryVariableType->getEntryVariable_ptr() );
						if ( entryVariableType->getEntryVariable_ptr() != NULL )
							entryVariableName = entryVariableType->getEntryVariable_ptr() ->getName();
						QString distributionType = entryVariableType->getType();
						setText ( row, 1, entryVariableName );
						setText ( row, 2, distributionType );
					}
					else if ( child->inherits ( "OpenTURNS::UI::GUI::Content" ) )
					{
						setText ( row, 1, child->getName() );
						setText ( row, 2, child->getDescription() );
					}
					else if ( child->inherits ( "OpenTURNS::UI::GUI::Block" ) && ( row == 0 ) )
					{
						clear();
						append ( child, true );
					}
					else if ( child->inherits ( "OpenTURNS::UI::GUI::Chapter" ) && ( row == 0 ) )
					{
						clear();
						append ( child, true );
					}
				}
			}

			void QVisualizationTable::clear()
			{
				QMemArray<int> allRows ( numRows() );
				for ( int i = ( numRows() - 1 ); i >= 0 ; i-- )
				{
					QOTTableItem * otItem = ( QOTTableItem* ) item ( i , 0 );
					otItem->getQOTObject() ->disconnect ( this );
					allRows[ i ] = i;
				}
				removeRows ( allRows );
			}

			QOTTableItem* QVisualizationTable::setOTTableItem ( int row, int col, const QString& text, QOTObject * otObject, bool title )
			{
				QOTTableItem * item = title ? new QOTTitleTableItem ( this, otObject ) : new QOTTableItem ( this, QOTTableItem::Never, otObject );
				item->setText ( text );
				setItem ( row, col, item );
				return item;
			}

			QToolBar* QVisualizationTable::createToolbar ( QOTObject * otObject )
			{
				QToolBar * toolBar = new QToolBar();
				QOTActions actions ( otObject->getActions() );
				for ( uint i = 0; i < actions.count(); i++ )
				{
					actions.at ( i ) ->addTo ( toolBar );
				}
				return toolBar;
			}
		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

