//                                               -*- C++ -*-
/**
 *  @file  ResourceMap.hxx
 *  @brief ResourceMap defines a resource catalog
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: ResourceMap.hxx 990 2008-10-29 17:54:45Z dutka $
 */
#ifndef OPENTURNS_RESOURCEMAP_HXX
#define OPENTURNS_RESOURCEMAP_HXX

#include <map>
#include "Pointer.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /**
       * @class ResourceMap
       *
       * ResourceMap defines a resource catalog
       */

      class ResourceMap : public std::map<String,String>
      {

      public:
#ifdef SWIG
	typedef String                                  KeyType;
	typedef String                                  MappedType;
#else
	typedef std::map<String,String>::key_type       KeyType;
	typedef std::map<String,String>::mapped_type    MappedType;
#endif

	typedef std::map<String,String>::iterator       iterator;
	typedef std::map<String,String>::const_iterator const_iterator;

	/** Since ResourceMap is a singleton, GetInstance gives access to the object */
	static  ResourceMap & GetInstance();

	/** The instance of ResourceMap */
	static Pointer<ResourceMap> P_instance_;


      public:

	/** Method for retrieving information from the resource map */
	const String get(KeyType key) const;

	/** Method for setting information into the resource map */
	void set(KeyType key, MappedType value);

	/** String converter */
	String str() const;

#ifndef SWIG
	/** Begin points to the first pair of the map */
	iterator begin();
	const_iterator begin() const;

	/** End points past the last pair of the map */
	iterator end();
	const_iterator end() const;
#endif

      protected:

	/** Update the ResourceMap with information from the configuration file */
	void readConfigurationFile(const FileName & configurationFile);

	/** Find the configuration file in specific path (see Path class for algorithm) */
	FileName findConfigurationFile() const;

	/** Load the configuration file */
	void loadConfigurationFile();

	/** Load the configuration defined at installation time */
	void loadDefaultConfiguration();

      private:
	
	/** Default constructor */
	ResourceMap();

      }; /* class ResourceMap */

      /**
       * @fn std::ostream & operator <<(std::ostream & os, const ResourceMap & obj)
       * @brief Output stream converter
       * @param os A STL output stream resourceMap
       * @param obj The resourceMap read by \em os
       * @return A reference to \em os
       *
       * Operator << converts the ResourceMap object to an output stream
       * so it is easy to show the reason of the resourceMap.
       */
      std::ostream & operator <<(std::ostream & os, const ResourceMap & obj);



    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_RESOURCEMAP_HXX */
