//                                               -*- C++ -*-
/**
 *  @file  StorageManager.cxx
 *  @brief StorageManager provides an interface for different storage classes
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: StorageManager.cxx 990 2008-10-29 17:54:45Z dutka $
 */
#include "StorageManager.hxx"
#include "PersistentObject.hxx"
#include "InterfaceObject.hxx"
#include "Study.hxx"
#include "OSS.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {
      CLASSNAMEINIT(StorageManager);



      StorageManager::Advocate::Advocate(const Implementation & p_implementation)
	: p_implementation_(p_implementation)
      {
	// Nothing to do
      }
	

      StorageManager::Advocate::~Advocate()
      {
	// Nothing to do
      }



      StorageManager * StorageManager::Advocate::getManager() const
      {
	return p_implementation_->getManager();
      }


      /* List methods */
      StorageManager::List StorageManager::Advocate::getList(EntityName entity) const
      {
	return p_implementation_->getList(entity);
      }



      /* Methods to write out class members */
      void StorageManager::Advocate::writeAttribute(AttributeName attribute, const String & st) const
      {
	p_implementation_->writeAttribute(attribute, st);
      }

      void StorageManager::Advocate::writeAttribute(AttributeName attribute, UnsignedLong ul) const
      {
	p_implementation_->writeAttribute(attribute, ul);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, NumericalScalar value) const
      {
	p_implementation_->writeValue(index, value);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, NumericalComplex value) const
      {
	p_implementation_->writeValue(index, value);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, const InterfaceObject & obj) const
      {
	p_implementation_->writeValue(index, obj);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, const PersistentObject & obj) const
      {
	p_implementation_->writeValue(index, obj);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, const String & st) const
      {
	p_implementation_->writeValue(index, st);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, UnsignedLong value) const
      {
	p_implementation_->writeValue(index, value);
      }

      void StorageManager::Advocate::writeValue(UnsignedLong index, Bool value) const
      {
	p_implementation_->writeValue(index, value);
      }

      void StorageManager::Advocate::writeValue(const String & name, NumericalScalar value) const
      {
	p_implementation_->writeValue(name, value);
      }

      void StorageManager::Advocate::writeValue(const String & name, NumericalComplex value) const
      {
	p_implementation_->writeValue(name, value);
      }

      void StorageManager::Advocate::writeValue(const String & name, const InterfaceObject & obj) const
      {
	p_implementation_->writeValue(name, obj);
      }

      void StorageManager::Advocate::writeValue(const String & name, const PersistentObject & obj) const
      {
	p_implementation_->writeValue(name, obj);
      }

      void StorageManager::Advocate::writeValue(const String & name, const String & st) const
      {
	p_implementation_->writeValue(name, st);
      }

      void StorageManager::Advocate::writeValue(const String & name, UnsignedLong value) const
      {
	p_implementation_->writeValue(name, value);
      }

      void StorageManager::Advocate::writeValue(const String & name, Bool value) const
      {
	p_implementation_->writeValue(name, value);
      }

      void StorageManager::Advocate::writeValue(const InterfaceObject & obj) const
      {
	p_implementation_->writeValue(obj);
      }

      void StorageManager::Advocate::writeValue(const InterfaceObject & obj, AttributeName attribute, const String & st) const
      {
	p_implementation_->writeValue(obj, attribute, st);
      }

      void StorageManager::Advocate::writeValue(const PersistentObject & obj) const
      {
	p_implementation_->writeValue(obj);
      }

      void StorageManager::Advocate::writeValue(const PersistentObject & obj, AttributeName attribute, const String & st) const
      {
	p_implementation_->writeValue(obj, attribute, st);
      }


      /* Methods to read class members */
      void StorageManager::Advocate::readAttribute(AttributeName attribute, String & st) const
      {
	p_implementation_->readAttribute(attribute, st);
      }

      void StorageManager::Advocate::readAttribute(AttributeName attribute, UnsignedLong & ul) const
      {
	p_implementation_->readAttribute(attribute, ul);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, NumericalScalar & value) const
      {
	p_implementation_->readValue(index, value);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, NumericalComplex & value) const
      {
	p_implementation_->readValue(index, value);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, InterfaceObject & obj) const
      {
	p_implementation_->readValue(index, obj);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, PersistentObject & obj) const
      {
	p_implementation_->readValue(index, obj);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, String & st) const
      {
	p_implementation_->readValue(index, st);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, UnsignedLong & value) const
      {
	p_implementation_->readValue(index, value);
      }

      void StorageManager::Advocate::readValue(UnsignedLong & index, Bool & value) const
      {
	p_implementation_->readValue(index, value);
      }

      void StorageManager::Advocate::readValue(String & name, NumericalScalar & value) const
      {
	p_implementation_->readValue(name, value);
      }

      void StorageManager::Advocate::readValue(String & name, NumericalComplex & value) const
      {
	p_implementation_->readValue(name, value);
      }

      void StorageManager::Advocate::readValue(String & name, InterfaceObject & obj) const
      {
	p_implementation_->readValue(name, obj);
      }

      void StorageManager::Advocate::readValue(String & name, PersistentObject & obj) const
      {
	p_implementation_->readValue(name, obj);
      }

      void StorageManager::Advocate::readValue(String & name, String & st) const
      {
	p_implementation_->readValue(name, st);
      }

      void StorageManager::Advocate::readValue(String & name, UnsignedLong & value) const
      {
	p_implementation_->readValue(name, value);
      }

      void StorageManager::Advocate::readValue(String & name, Bool & value) const
      {
	p_implementation_->readValue(name, value);
      }

      void StorageManager::Advocate::readValue(InterfaceObject & obj, AttributeName attribute, const String & st) const
      {
	p_implementation_->readValue(obj, attribute, st);
      }

      void StorageManager::Advocate::readValue(PersistentObject & obj, AttributeName attribute, const String & st) const
      {
	p_implementation_->readValue(obj, attribute, st);
      }



      /* Implementations */


      StorageManager::AdvocateImplementation::AdvocateImplementation(StorageManager * p_mgr)
	: p_manager_(p_mgr)
      {
	p_manager_->pushState();
      }
	
      StorageManager::AdvocateImplementation::AdvocateImplementation(const AdvocateImplementation & other)
	: p_manager_(other.p_manager_)
      {
	p_manager_->pushState();
      }
	
      StorageManager::AdvocateImplementation * StorageManager::AdvocateImplementation::clone() const
      {
	return new AdvocateImplementation(*this);
      }

      StorageManager::AdvocateImplementation::~AdvocateImplementation()
      {
	p_manager_->popState();
      }


      /* List methods */
      StorageManager::List StorageManager::AdvocateImplementation::getList(EntityName entity) const
      {
	return p_manager_->getList(entity);
      }



      /* Methods to write out class members */
      void StorageManager::AdvocateImplementation::writeAttribute(AttributeName attribute, const String & st) const
      {
	p_manager_->writeAttribute(attribute, st);
      }

      void StorageManager::AdvocateImplementation::writeAttribute(AttributeName attribute, UnsignedLong ul) const
      {
	p_manager_->writeAttribute(attribute, ul);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, NumericalScalar value) const
      {
	p_manager_->writeValue(index, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, NumericalComplex value) const
      {
	p_manager_->writeValue(index, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, const InterfaceObject & obj) const
      {
	obj.save(*p_manager_);
	const PersistentObject & refPersObj = *(obj.getImplementationAsPersistentObject());
	p_manager_->writeValue(index, refPersObj);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, const PersistentObject & obj) const
      {
	obj.save(*p_manager_);
	p_manager_->writeValue(index, obj);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, const String & st) const
      {
	p_manager_->writeValue(index, st);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, UnsignedLong value) const
      {
	p_manager_->writeValue(index, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(UnsignedLong index, Bool value) const
      {
	p_manager_->writeValue(index, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, NumericalScalar value) const
      {
	p_manager_->writeValue(name, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, NumericalComplex value) const
      {
	p_manager_->writeValue(name, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, const InterfaceObject & obj) const
      {
	obj.save(*p_manager_);
	const PersistentObject & refPersObj = *(obj.getImplementationAsPersistentObject());
	p_manager_->writeValue(name, refPersObj);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, const PersistentObject & obj) const
      {
	obj.save(*p_manager_);
	p_manager_->writeValue(name, obj);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, const String & st) const
      {
	p_manager_->writeValue(name, st);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, UnsignedLong value) const
      {
	p_manager_->writeValue(name, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(const String & name, Bool value) const
      {
	p_manager_->writeValue(name, value);
      }

      void StorageManager::AdvocateImplementation::writeValue(const InterfaceObject & obj) const
      {
	obj.save(*p_manager_);
	const PersistentObject & refPersObj = *(obj.getImplementationAsPersistentObject());
	p_manager_->writeValue(refPersObj);
      }

      void StorageManager::AdvocateImplementation::writeValue(const InterfaceObject & obj, AttributeName attribute, const String & st) const
      {
	obj.save(*p_manager_);
	const PersistentObject & refPersObj = *(obj.getImplementationAsPersistentObject());
	p_manager_->writeValue(refPersObj, attribute, st);
      }

      void StorageManager::AdvocateImplementation::writeValue(const PersistentObject & obj) const
      {
	obj.save(*p_manager_);
	p_manager_->writeValue(obj);
      }

      void StorageManager::AdvocateImplementation::writeValue(const PersistentObject & obj, AttributeName attribute, const String & st) const
      {
	obj.save(*p_manager_);
	p_manager_->writeValue(obj, attribute, st);
      }


      /* Methods to read class members */
      void StorageManager::AdvocateImplementation::readAttribute(AttributeName attribute, String & st) const
      {
	p_manager_->readAttribute(attribute, st);
      }

      void StorageManager::AdvocateImplementation::readAttribute(AttributeName attribute, UnsignedLong & ul) const
      {
	p_manager_->readAttribute(attribute, ul);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, NumericalScalar & value) const
      {
	p_manager_->readValue(index, value);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, NumericalComplex & value) const
      {
	p_manager_->readValue(index, value);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, InterfaceObject & obj) const
      {
	p_manager_->readValue(index, obj);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, PersistentObject & obj) const
      {
	p_manager_->readValue(index, obj);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, String & st) const
      {
	p_manager_->readValue(index, st);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, UnsignedLong & value) const
      {
	p_manager_->readValue(index, value);
      }

      void StorageManager::AdvocateImplementation::readValue(UnsignedLong & index, Bool & value) const
      {
	p_manager_->readValue(index, value);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, NumericalScalar & value) const
      {
	p_manager_->readValue(name, value);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, NumericalComplex & value) const
      {
	p_manager_->readValue(name, value);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, InterfaceObject & obj) const
      {
	p_manager_->readValue(name, obj);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, PersistentObject & obj) const
      {
	p_manager_->readValue(name, obj);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, String & st) const
      {
	p_manager_->readValue(name, st);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, UnsignedLong & value) const
      {
	p_manager_->readValue(name, value);
      }

      void StorageManager::AdvocateImplementation::readValue(String & name, Bool & value) const
      {
	p_manager_->readValue(name, value);
      }

      void StorageManager::AdvocateImplementation::readValue(InterfaceObject & obj, AttributeName attribute, const String & st) const
      {
	p_manager_->readValue(obj, attribute, st);
      }

      void StorageManager::AdvocateImplementation::readValue(PersistentObject & obj, AttributeName attribute, const String & st) const
      {
	p_manager_->readValue(obj, attribute, st);
      }


      StorageManager * StorageManager::AdvocateImplementation::getManager() const
      {
	return p_manager_;
      }



      StorageManager::WriteAdvocateImplementation::WriteAdvocateImplementation(StorageManager * p_mgr, const PersistentObject & obj, bool fromStudy)
	: AdvocateImplementation(p_mgr), obj_(obj)
      {
	getManager()->writeObjectPrologue(obj_, fromStudy);
      }
	
      StorageManager::AdvocateImplementation * StorageManager::WriteAdvocateImplementation::clone() const
      {
	return new WriteAdvocateImplementation(*this);
      }

      StorageManager::WriteAdvocateImplementation::~WriteAdvocateImplementation()
      {
	getManager()->writeObjectEpilogue(obj_);
      }





      StorageManager::ReadAdvocateImplementation::ReadAdvocateImplementation(StorageManager * p_mgr)
	: AdvocateImplementation(p_mgr)
      {
	// Nothing to do
      }
	
      StorageManager::AdvocateImplementation * StorageManager::ReadAdvocateImplementation::clone() const
      {
	return new ReadAdvocateImplementation(*this);
      }

      StorageManager::ReadAdvocateImplementation::~ReadAdvocateImplementation()
      {
	// Nothing to do
      }







      /* Default constructor */
      StorageManager::StorageManager(UnsignedLong defaultVersion)
	: Object(),
	  p_study_(0),
	  entity_(EntityNameEnumSize),
	  attribute_(AttributeNameEnumSize),
	  defaultVersion_(defaultVersion),
	  version_(defaultVersion),
	  savedObjects_()
      {
	// We fill the entity list
	entity_[ObjectEntity]             = "object";
	entity_[NumericalScalarEntity]    = "numerical-scalar";
	entity_[NumericalComplexEntity]   = "numerical-complex";
	entity_[StringEntity]             = "string";
	entity_[UnsignedLongEntity]       = "unsigned-long";
	entity_[BoolEntity]               = "bool";

	// We fill the attribute list
	attribute_[IdAttribute]           = "id";
	attribute_[ClassAttribute]        = "class";
	attribute_[NameAttribute]         = "name";
	attribute_[SizeAttribute]         = "size";
	attribute_[DimensionAttribute]    = "dimension";
	attribute_[IndexAttribute]        = "index";
	attribute_[StudyVisibleAttribute] = "study-visible";
	attribute_[MemberNameAttribute]   = "member-name";
	attribute_[VersionAttribute]      = "version";
	attribute_[KeyAttribute]          = "key";
	attribute_[ValueAttribute]        = "value";
	attribute_[MaxSizeAttribute]      = "maxSize";
      }

      /*
       * Virtual constructor
       */
      StorageManager * StorageManager::clone() const
      {
	return new StorageManager(*this);
      }


      /* String converter */
      String StorageManager::str() const
      {
	return OSS() << "class=" << getClassName();
      }


      /* Do some administrative tasks before saving/reloading */
      void StorageManager::initialize(const MethodCall)
      {
	throw NotDefinedException(HERE);
      }

      /* Do some administrative tasks after saving/reloading */
      void StorageManager::finalize(const MethodCall)
      {
	throw NotDefinedException(HERE);
      }


      /* Read and create the internal representation */
      void StorageManager::read()
      {
	throw NotDefinedException(HERE);
      }

      /* Write the internal representation */
      void StorageManager::write()
      {
	throw NotDefinedException(HERE);
      }


      /* Returns true if an object is already saved */
      Bool StorageManager::isSavedObject(Id id) const
      {
	return savedObjects_[id];
      }

      /* Tells that an object is saved */
      void StorageManager::markObjectAsSaved(Id id) const
      {
	savedObjects_[id] = true;
      }

      /* Refresh the status of saved objects */
      void StorageManager::cleanSavedObjects() const
      {
	savedObjects_.erase(savedObjects_.begin(), savedObjects_.end());
      }


      /*
       * This method saves the PersistentObject onto the medium
       *
       * @todo implement StorageManager::save method
       */
      void StorageManager::save(const PersistentObject & obj, bool fromStudy)
      {
	throw NotDefinedException(HERE);
      }

      /*
       * This method reloads the PersistentObject from the medium
       *
       * @todo implement StorageManager::load method
       */
      void StorageManager::load(Study & study)
      {
	throw NotDefinedException(HERE);
      }


      /* Study accessor */
      Study * StorageManager::getStudy() const
      {
	return p_study_;
      }

      void StorageManager::setStudy(Study * p_study)
      {
	p_study_ = p_study;
      }


      /* Returns the version of the study that the manager can read/write */
      UnsignedLong StorageManager::getDefaultStudyVersion() const
      {
	return defaultVersion_;
      }
      
      /* Query the manager if the version is correct */
      Bool StorageManager::canManageVersion(UnsignedLong version) const
      {
	return (version == defaultVersion_);
      }

      /* Study version accessor */
      UnsignedLong StorageManager::getStudyVersion() const
      {
	return version_;
      }

      void StorageManager::setStudyVersion(UnsignedLong version)
      {
	if (canManageVersion(version)) version_ = version;
	else throw InvalidArgumentException(HERE) << "Study version " << version << " is NOT supported";
      }



      /*
       * This method build a new advocate to talk to the storage manager
       */
      Pointer<StorageManager::WriteAdvocateImplementation> StorageManager::registerObject(const PersistentObject & obj, bool fromStudy) const
      {
	return new WriteAdvocateImplementation(const_cast<StorageManager *>(this), obj, fromStudy);
      }


      /* Returns the string associated with the entity */
      /*
       * This method build a new advocate to talk to the storage manager
       */
      Pointer<StorageManager::ReadAdvocateImplementation> StorageManager::readObject() const
      {
	return new ReadAdvocateImplementation(const_cast<StorageManager *>(this));
      }


      String StorageManager::getEntityString(EntityName entity) const
      {
	return entity_[entity];
      }


      /* Returns the string associated with the attribute */
      String StorageManager::getAttributeString(AttributeName attribute) const
      {
	return attribute_[attribute];
      }




      /* Methods to control manager internal state */
      void StorageManager::pushState()
      {
	// Nothing to do
      }

      void StorageManager::popState()
      {
	// Nothing to do
      }



      /* List methods */
      StorageManager::List StorageManager::getList(EntityName entity) const
      {
	throw NotDefinedException(HERE);
      }



      /* Methods to write out objects (NOT IMPLEMENTED) */
      void StorageManager::writeObjectPrologue(const PersistentObject & obj, bool fromStudy)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeObjectEpilogue(const PersistentObject & obj)
      {
	throw NotDefinedException(HERE);
      }

	
      /* Methods to write out class members (NOT IMPLEMENTED) */
      void StorageManager::writeAttribute(AttributeName attribute, const String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeAttribute(AttributeName attribute, UnsignedLong ul)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(UnsignedLong index, NumericalScalar value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(UnsignedLong index, NumericalComplex value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(UnsignedLong index, const PersistentObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(UnsignedLong index, const String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(UnsignedLong index, UnsignedLong value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(UnsignedLong index, Bool value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const String & name, NumericalScalar value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const String & name, NumericalComplex value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const String & name, const PersistentObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const String & name, const String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const String & name, UnsignedLong value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const String & name, Bool value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const PersistentObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::writeValue(const PersistentObject & obj, AttributeName attribute, const String & st)
      {
	throw NotDefinedException(HERE);
      }


      /* Methods to read class members (NOT IMPLEMENTED) */
      void StorageManager::readAttribute(AttributeName attribute, String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readAttribute(AttributeName attribute, UnsignedLong & ul)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, NumericalScalar & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, NumericalComplex & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, InterfaceObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, PersistentObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, UnsignedLong & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(UnsignedLong & index, Bool & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, NumericalScalar & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, NumericalComplex & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, InterfaceObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, PersistentObject & obj)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, UnsignedLong & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(String & name, Bool & value)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(InterfaceObject & obj, AttributeName attribute, const String & st)
      {
	throw NotDefinedException(HERE);
      }

      void StorageManager::readValue(PersistentObject & obj, AttributeName attribute, const String & st)
      {
	throw NotDefinedException(HERE);
      }


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
