//                                               -*- C++ -*-
/**
 *  @file  Study.cxx
 *  @brief Study keeps all PersistentObjects in a file
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: Study.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <algorithm>
#include "InterfaceObject.hxx"
#include "Study.hxx"
#include "StorageManager.hxx"
#include "Exception.hxx"
#include "Catalog.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      CLASSNAMEINIT(Study);

      /* Assignment */
      // Study & Study::operator =(const Study & other);


      /*
       * Default constructor
       */
      Study::Study()
	: p_storageManager_(new StorageManager)
      {
	p_storageManager_->setStudy(this);
      }



      /* String converter */
      String Study::str() const
      {
	OSS oss;
	oss << "class=" << getClassName();

	MapCollection::const_iterator map_it;
	for(map_it = maps.begin(); map_it != maps.end(); map_it++) {
	  oss << " map #" << (*map_it).first << "={\n";
	  ObjectMap map = (*map_it).second;
	  ObjectMap::const_iterator it;
	  const char * separator = "  ";
	  for(it = map.begin(); it != map.end(); it++, separator="\n  ") {
#if 0
	    oss << separator << (*it).first << " => " << ((*it).second)->getClassName()
		<< " name='" << ((*it).second)->getName()
		<< "' id=" << ((*it).second)->getId();
#endif
#if 0
	    if (((*it).second)->getVisibility())
	      oss << separator << (*it).first << " => id=" << ((*it).second)->getId()
		  << " " << ((*it).second)->str();
#endif
#if 1
	    oss << separator << (*it).first << " => " << (((*it).second)->getVisibility()?"*":" ")<< " id=" << ((*it).second)->getId()
		<< " " << ((*it).second)->str();
#endif
	  }
	  oss << "\n}";
	}

	return oss;
      }


      /* This method saves the study through the storage manager */
      void Study::save()
      {
	p_storageManager_->initialize(StorageManager::SAVE);

	MapCollection::const_iterator map_it;
	for(map_it = maps.begin(); map_it != maps.end(); ++map_it) {
	  const ObjectMap & map = map_it->second;
	  ObjectMap::const_iterator it;
	  for(it = map.begin(); it != map.end(); it++) {
	    p_storageManager_->save(*(it->second), true);
	  }
	}

	p_storageManager_->write();
	p_storageManager_->finalize(StorageManager::SAVE);
      }

      /* This method reloads the study from the storage manager */
      void Study::load()
      {
	p_storageManager_->initialize(StorageManager::LOAD);
	p_storageManager_->read();
	p_storageManager_->load(*this);
	p_storageManager_->finalize(StorageManager::LOAD);

	cleanUnvisibleObject();
	translateId();
      }

      /* This method purges the study from the reloaded objects that are tagged unvisible */
      void Study::cleanUnvisibleObject()
      {
	MapCollection::iterator map_it;
	for(map_it = maps.begin(); map_it != maps.end(); ++map_it) {
	  ObjectMap & map = map_it->second;
	  ObjectMap newMap;
	  ObjectMap::iterator it;
	  for(it = map.begin(); it != map.end(); ++it) {
	    if (it->second->getVisibility())
	      newMap[it->second->getShadowedId()] = it->second;
	  }
	  map = newMap;
	}
      }

      /* This method performs the translation of ids after a study load */
      void Study::translateId()
      {
	MapCollection::iterator map_it;
	for(map_it = maps.begin(); map_it != maps.end(); ++map_it) {
	  ObjectMap & map = map_it->second;
	  ObjectMap newMap;
	  ObjectMap::iterator it;
	  for(it = map.begin(); it != map.end(); ++it) {
	    newMap[it->second->getId()] = it->second;
	  }
	  map = newMap;
	}
      }

 



      /* Query if object is stored in study */
      Bool Study::hasObject(Id id) const
      {
	Bool result = false;

	MapCollection::const_iterator map_it;
	for(map_it = maps.begin(); !result && (map_it != maps.end()); map_it++) {
	  const ObjectMap & map = map_it->second;
	  result |= (map.find(id) != map.end());
	}
	
	return result;
      }

      /* Get object whose id is given */
      Study::MapElement Study::getObject(Id id) const
      {
	if (!hasObject(id)) throw ObjectNotInStudyException(HERE) << "Object of id " << id << " is NOT in study";
	MapElement element;
	MapCollection::const_iterator map_it;
	for(map_it = maps.begin(); map_it != maps.end(); map_it++) {
	  const ObjectMap & map = map_it->second;
	  ObjectMap::const_iterator it;
	  if ((it=map.find(id)) != map.end()) {
	    element = (*it).second;
	    break;
	  }
	}
	return element;
      }



      /* Local class for the following method. Should have been declared inside the method but find_if crashes */
      struct class_and_name_are
      {
	String className_;
	String name_;
	class_and_name_are(const String & className, const String & name) : className_(className), name_(name) {}
	Bool operator()(const Study::ObjectMap::value_type & element) const
	{
	  return (element.second->getClassName() == className_) && (element.second->getName() == name_);
	}
      };

      /* Get object whose class and name are given */
      Study::MapElement Study::getObjectByName(const String & className, const String & name) const
      {
	MapElement element;
	MapCollection::const_iterator map_it;
	for(map_it = maps.begin(); map_it != maps.end(); map_it++) {
	  const ObjectMap & map = map_it->second;
	  ObjectMap::const_iterator it;
	  if ((it=std::find_if(map.begin(),map.end(),class_and_name_are(className,name))) != map.end()) {
	    element = (*it).second;
	    break;
	  }
	}
	return element;
      }


      /* Fill an object with one got from study */
      void Study::fillObject(PersistentObject & po, const String & name) const
      {
	MapElement element = getObjectByName(po.getClassName(), name);
	if (!element.isNull()) Catalog::GetInstance().get(po.getClassName()).assign(po, *element);
	else throw InvalidArgumentException(HERE) << "No object of name " << name << " in study";
      }

      void Study::fillObject(InterfaceObject  & io, const String & name) const
      {
	MapElement element = getObjectByName(io.getImplementationAsPersistentObject()->getClassName(), name);
	if (!element.isNull()) {
	  io.setImplementationAsPersistentObject(element);
	} else throw InvalidArgumentException(HERE) << "No object of name " << name << " in study";	
      }

      void Study::fillObject(PersistentObject & po, const Id id) const
      {
	MapElement element = getObject(id);
	if (!element.isNull()) Catalog::GetInstance().get(po.getClassName()).assign(po, *element);
	else throw InvalidArgumentException(HERE) << "No object of id " << id << " in study";
      }

      void Study::fillObject(InterfaceObject & io, const Id id) const
      {
	MapElement element = getObject(id);
	if (!element.isNull()) {
	  io.setImplementationAsPersistentObject(element);
	} else throw InvalidArgumentException(HERE) << "No object of id " << id << " in study";	
      }


      /* Storage manager accessor */
      void Study::setStorageManager(const StorageManager & smgr)
      {
	p_storageManager_.reset(smgr.clone());
	p_storageManager_->setStudy(this);
      }

      /* Storage manager accessor */
      Study::StorageManagerImplementation Study::getStorageManager() const
      {
	return p_storageManager_;
      }


      /* Add a PersistentObject to the study (main map) */
      void Study::add(const InterfaceObject & io, Map mapTag)
      {
	add(io, maps[mapTag] );
      }
      
      /* Remove a PersistentObject from the study (main map) */
      void Study::remove(const InterfaceObject & io, Map mapTag)
      {
	remove(io, maps[mapTag] );
      }

      /* Add a PersistentObject to the study (any map) */
      void Study::add(const InterfaceObject & io, ObjectMap & map)
      {
	map[io.getId()] = io.getImplementationAsPersistentObject();
      }
      
      /* Remove a PersistentObject from the study (any map) */
      void Study::remove(const InterfaceObject & io, ObjectMap & map)
      {
	map.erase(io.getId());
      }



      /* Add a PersistentObject to the study (main map) */
      void Study::add(const PersistentObject * po, Map mapTag)
      {
	add(po, maps[mapTag] );
      }
      
      /* Add a PersistentObject to the study (main map) */
      void Study::add(const PersistentObject & po, Map mapTag)
      {
	add(po.clone(), maps[mapTag] );
      }
      
      /* Add a PersistentObject to the study (any map) */
      void Study::add(const PersistentObject * po, ObjectMap & map)
      {
	map[po->getShadowedId()] = const_cast<PersistentObject *>(po);
      }
      







    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
