//                                               -*- C++ -*-
/**
 *  @file  TypedCollectionInterfaceObject.hxx
 *  @brief TypedCollectionInterfaceObject implements TypedInterfaceObject for a class that manages a collection
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: TypedCollectionInterfaceObject.hxx 862 2008-06-26 11:50:17Z dutka $
 */
#ifndef OPENTURNS_TYPEDCOLLECTIONINTERFACEOBJECT_HXX
#define OPENTURNS_TYPEDCOLLECTIONINTERFACEOBJECT_HXX

#include "TypedInterfaceObject.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /**
       * @class TypedCollectionInterfaceObject
       *
       * TypedCollectionInterfaceObject implements TypedInterfaceObject for a class
       * that manages a collection
       */
      template <class T>
      class TypedCollectionInterfaceObject
	: public TypedInterfaceObject<T>
      {
      public:

	typedef typename TypedInterfaceObject<T>::ImplementationAsPersistentObject ImplementationAsPersistentObject;
	typedef typename TypedInterfaceObject<T>::ImplementationType               ImplementationType;
	typedef typename ImplementationType::ElementType                           ImplementationElementType;
	typedef typename TypedInterfaceObject<T>::Implementation                   Implementation;
	typedef typename ImplementationType::iterator                              iterator;
	typedef typename ImplementationType::const_iterator                        const_iterator;
	typedef typename ImplementationType::reverse_iterator                      reverse_iterator;
	typedef typename ImplementationType::const_reverse_iterator                const_reverse_iterator;

	/** Constructor */
	TypedCollectionInterfaceObject() {}
	TypedCollectionInterfaceObject(const Implementation & impl) : TypedInterfaceObject<T>(impl) {}


#ifndef SWIG
	/** Operator[]() gives access to the elements of the collection */
	inline ImplementationElementType & operator [] (UnsignedLong i)
	{
	  if ( (getSize() == 0) ||
	       (i >= getSize() ) ) throw OutOfBoundException(HERE) << "Incorrect index or dimension: size=" << getSize()
								   << " index=" << i;
	  
	  TypedInterfaceObject<T>::copyOnWrite();
	  return (*TypedInterfaceObject<T>::getImplementation())[i];
	}

	/** Operator[]() gives access to the elements of the const collection */
	inline const ImplementationElementType & operator [] (UnsignedLong i) const
	{
	  if ( (getSize() == 0) ||
	       (i >= getSize() ) ) throw OutOfBoundException(HERE) << "Incorrect index or dimension: size=" << getSize()
								   << " index=" << i;
	  
	  return (*TypedInterfaceObject<T>::getImplementation())[i];
	}
#endif

	/* Method __getitem__() is for Python */
	inline const ImplementationElementType & __getitem__(UnsignedLong i) const
	{
	  return (*this)[i];
	}

	/* Method __setitem__() is for Python */
	inline void __setitem__(UnsignedLong i, const ImplementationElementType & val)
	{
	  (*this)[i] = val;
	}

	/** Method add() appends an element to the collection */
	inline void add(const ImplementationElementType & elt)
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  TypedInterfaceObject<T>::getImplementation()->add(elt);
	}

#ifndef SWIG
	/** Erase the elements between first and last */
	inline iterator erase(iterator first, iterator last)
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  return TypedInterfaceObject<T>::getImplementation()->erase(first, last);
	}
      
	/** Erase the element pointed by position */
	inline iterator erase(iterator position)
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  return TypedInterfaceObject<T>::getImplementation()->erase(position);
	}

	/** Erase the element pointed by position */
	inline iterator erase(UnsignedLong position)
	{
	  return erase(begin() + position);
	}
#endif

	/** Method getSize() returns the number of elements of the collection (viewed as a size) */
	inline UnsignedLong getSize() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->getSize();
	}

	/** Method getDimension() returns the number of elements of the collection (viewed as a dimension) */
	inline UnsignedLong getDimension() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->getDimension();
	}

	/** Method empty() returns true if there is no element in the collection */
	inline Bool isEmpty() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->isEmpty();
	}

#ifndef SWIG
	/** Method begin() points to the first element of the collection */
	inline iterator begin()
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  return TypedInterfaceObject<T>::getImplementation()->begin();
	}
	inline const_iterator begin() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->begin();
	}

	/** Method end() points beyond the last element of the collection */
	inline iterator end()
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  return TypedInterfaceObject<T>::getImplementation()->end();
	}
	inline const_iterator end() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->end();
	}

	/** Method rbegin() points to the last element of the collection */
	inline reverse_iterator rbegin()
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  return TypedInterfaceObject<T>::getImplementation()->rbegin();
	}
	inline const_reverse_iterator rbegin() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->rbegin();
	}

	/** Method rend() points before the first element of the collection */
	inline reverse_iterator rend()
	{
	  TypedInterfaceObject<T>::copyOnWrite();
	  return TypedInterfaceObject<T>::getImplementation()->rend();
	}
	inline const_reverse_iterator rend() const
	{
	  return TypedInterfaceObject<T>::getImplementation()->rend();
	}
#endif

      }; /* class TypedCollectionInterfaceObject */




    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_TYPEDCOLLECTIONINTERFACEOBJECT_HXX */
