//                                               -*- C++ -*-
/**
 *  @file  XMLStorageManager.cxx
 *  @brief XMLStorageManager provides an interface for different storage classes
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: XMLStorageManager.cxx 990 2008-10-29 17:54:45Z dutka $
 */
#include "XMLStorageManager.hxx"
#include "PersistentObject.hxx"
#include "InterfaceObject.hxx"
#include "Exception.hxx"
#include "Catalog.hxx"
#include "Study.hxx"
#include "OSS.hxx"
#include "Log.hxx"


namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      static const char * RootElementName = "openturns-study";

      XMLStorageManager::XMLList::XMLList(const XMLStorageManager & mgr, State state, EntityName entity)
	: nodeList_(XML::GetFirstChild(state.current_)),
	  cursor_(0),
	  manager_(mgr),
	  length_(0),
	  entity_(entity)
      {
	// we count the number of children
	for (XML::Node cur = nodeList_; cur; cur = XML::GetNextNode(cur))
	  if (XML::IsElement(cur, manager_.getEntityString(entity_))) ++length_;
      }

      StorageManager::ListImplementation * XMLStorageManager::XMLList::clone() const
      {
	return new XMLList(*this);
      }

      UnsignedLong XMLStorageManager::XMLList::getSize() const
      {
	return length_;
      }

      void XMLStorageManager::XMLList::firstValueToRead()
      {
	for( cursor_ = nodeList_; cursor_; cursor_ = XML::GetNextNode(cursor_) )
	  if (XML::IsElement(cursor_, manager_.getEntityString(entity_))) break;
      }

      Bool XMLStorageManager::XMLList::moreValuesToRead()
      {
	return (cursor_ != NULL);
      }

      void XMLStorageManager::XMLList::nextValueToRead()
      {
	while ( (cursor_ = XML::GetNextNode(cursor_)) != NULL )
	  if (XML::IsElement(cursor_, manager_.getEntityString(entity_))) break;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, NumericalScalar & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(NumericalScalarEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    value = 0.;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, NumericalComplex & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(NumericalComplexEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    value = 0.;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, InterfaceObject & obj)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    Id shadowedId;
	    String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	    std::istringstream iss (stid);
	    iss >> shadowedId;
	    if (! manager_.getStudy()->hasObject(shadowedId))
	      throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	    obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, PersistentObject & obj)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    Id shadowedId;
	    String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	    std::istringstream iss (stid);
	    iss >> shadowedId;
	    if (! manager_.getStudy()->hasObject(shadowedId))
	      throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	    //obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	    Catalog::GetInstance().get(obj.getClassName()).assign(obj, *(manager_.getStudy()->getObject(shadowedId)));
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, String & st)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(StringEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    String tmp = XML::GetNodeValue( cursor_ );
	    st = tmp;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, UnsignedLong & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(UnsignedLongEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    value = 0;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(UnsignedLong & index, Bool & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(BoolEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IndexAttribute));
	    std::istringstream iss (stidx);
	    iss >> index;
	  }
	  {
	    value = false;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, NumericalScalar & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(NumericalScalarEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(IndexAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    value = 0.;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, NumericalComplex & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(NumericalComplexEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(NameAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    value = 0.;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, InterfaceObject & obj)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(NameAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    Id shadowedId;
	    String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	    std::istringstream iss (stid);
	    iss >> shadowedId;
	    if (! manager_.getStudy()->hasObject(shadowedId))
	      throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	    obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, PersistentObject & obj)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(NameAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    Id shadowedId;
	    String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	    std::istringstream iss (stid);
	    iss >> shadowedId;
	    if (! manager_.getStudy()->hasObject(shadowedId))
	      throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	    //obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	    Catalog::GetInstance().get(obj.getClassName()).assign(obj, *(manager_.getStudy()->getObject(shadowedId)));
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, String & st)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(StringEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(NameAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    String tmp = XML::GetNodeValue( cursor_ );
	    st = tmp;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, UnsignedLong & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(UnsignedLongEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(NameAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    value = 0;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(String & name, Bool & value)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(BoolEntity))
	  && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(NameAttribute));
	if (result) {
	  {
	    String stidx = XML::GetAttributeByName(cursor_, manager_.getAttributeString(NameAttribute));
	    std::istringstream iss (stidx);
	    iss >> name;
	  }
	  {
	    value = false;
	    String stval = XML::GetNodeValue( cursor_ );
	    std::istringstream iss (stval);
	    iss >> value;
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(InterfaceObject & obj)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity));
	if (result) {
	  Id shadowedId;
	  String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	  std::istringstream iss (stid);
	  iss >> shadowedId;
	  if (! manager_.getStudy()->hasObject(shadowedId))
	    throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	  obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(InterfaceObject & obj, AttributeName attribute, const String & attributeValue)
      {
	Bool result = true;
	for(firstValueToRead(); moreValuesToRead(); nextValueToRead()) {
	  result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity))
	    && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(attribute));
	  if (result) {
	    String attrReadValue = XML::GetAttributeByName(cursor_, manager_.getAttributeString(attribute));

	    if (attrReadValue == attributeValue) {
	      Id shadowedId;
	      String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	      std::istringstream iss (stid);
	      iss >> shadowedId;

	      if (! manager_.getStudy()->hasObject(shadowedId))
		throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	      obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	      break;
	    }
	  }
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(PersistentObject & obj)
      {
	Bool result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity));
	if (result) {
	  Id shadowedId;
	  String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	  std::istringstream iss (stid);
	  iss >> shadowedId;
	  if (! manager_.getStudy()->hasObject(shadowedId))
	    throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	  // obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	  Catalog::GetInstance().get(obj.getClassName()).assign(obj, *(manager_.getStudy()->getObject(shadowedId)));
	}
	return result;
      }

      Bool XMLStorageManager::XMLList::readValue(PersistentObject & obj, AttributeName attribute, const String & attributeValue)
      {
	Bool result = true;
	for(firstValueToRead(); moreValuesToRead(); nextValueToRead()) {
	  result = XML::IsElement(cursor_, manager_.getEntityString(ObjectEntity))
	    && XML::ElementHasAttribute(cursor_, manager_.getAttributeString(attribute));
	  if (result) {
	    String attrReadValue = XML::GetAttributeByName(cursor_, manager_.getAttributeString(attribute));

	    if (attrReadValue == attributeValue) {
	      Id shadowedId;
	      String stid = XML::GetAttributeByName(cursor_, manager_.getAttributeString(IdAttribute));
	      std::istringstream iss (stid);
	      iss >> shadowedId;

	      if (! manager_.getStudy()->hasObject(shadowedId))
		throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId << " in file '" << manager_.getFileName() << "' referenced before used";
	      // obj.setImplementationAsPersistentObject(manager_.getStudy()->getObject(shadowedId));
	      Catalog::GetInstance().get(obj.getClassName()).assign(obj, *(manager_.getStudy()->getObject(shadowedId)));
	      break;
	    }
	  }
	}
	return result;
      }








      CLASSNAMEINIT(XMLStorageManager);
      const XMLStorageManager::VersionList XMLStorageManager::SupportedVersions;

      /* Default constructor */
      XMLStorageManager::XMLStorageManager(const FileName & filename)
	: StorageManager(1),
	  fileName_(filename),
	  state_(),
	  document_()
      {
	// Nothing to do
      }
      


      /*
       * Virtual constructor
       */
      XMLStorageManager * XMLStorageManager::clone() const
      {
	return new XMLStorageManager(*this);
      }

      /* String converter */
      String XMLStorageManager::str() const
      {
	return OSS() << "class=" << getClassName();
      }


      /* Filename accessors */
      String XMLStorageManager::getFileName() const
      {
	return fileName_;
      }

      void XMLStorageManager::setFileName(const String & fileName)
      {
	fileName_ = fileName;
      }



      /* Query the manager if the version is correct */
      Bool XMLStorageManager::canManageVersion(UnsignedLong version) const
      {
	return XMLStorageManager::SupportedVersions.contains(version);
      }


      /* Do some administrative tasks before saving/reloading */
      void XMLStorageManager::initialize(const MethodCall caller)
      {
	State state;

	switch (caller) {
	case StorageManager::SAVE :
	  {
	    cleanSavedObjects();
	    OSS oss;
	    oss << getStudyVersion();
	    document_ = XMLDoc();
	    state.root_ = XML::NewNode( RootElementName );
	    XML::SetAttribute(state.root_, getAttributeString(VersionAttribute), oss);
	    XML::SetRootNode(document_, state.root_);
	  }
	break;

	case StorageManager::LOAD :
	  {
	    document_ = XMLDoc( fileName_ );
	  }
	break;

	default:
	  throw InvalidArgumentException(HERE) << "XMLStorageManager::initialize(const MethodCall caller) called with wrong value: " << caller;
	}

	state_.push(state);
      }

      /* Do some administrative tasks after saving/reloading */
      void XMLStorageManager::finalize(const MethodCall caller)
      {
	switch (caller) {
	case StorageManager::SAVE:
	  break;

	case StorageManager::LOAD:
	  break;

	default:
	  throw InvalidArgumentException(HERE) << "XMLStorageManager::initialize(const MethodCall caller) called with wrong value: " << caller;
	}

	state_.pop();
      }



      /* Read and create the internal representation */
      void XMLStorageManager::read()
      {
	state_.top().root_ = XML::GetRootNode( document_ );
	if (! state_.top().root_) throw StudyFileParsingException(HERE) << "Study file has no root element (" << fileName_ << ")";
	if (! XML::IsElement( state_.top().root_, RootElementName ))
	  throw StudyFileParsingException(HERE) << "Can NOT find root element '" << RootElementName << "' in file '" << fileName_ << "'. Got '" << XML::GetNodeName( state_.top().root_ ) << "'";
	UnsignedLong version = 0;
	String stul = XML::GetAttributeByName(state_.top().root_, getAttributeString(VersionAttribute));
	std::istringstream iss (stul);
	iss >> version;
	setStudyVersion(version);
      }

      /* Write the internal representation */
      void XMLStorageManager::write()
      {
	document_.save( fileName_ );
      }


      /*
       * This method saves the PersistentObject onto the medium
       */
      void XMLStorageManager::save(const PersistentObject & obj, bool fromStudy)
      {
	obj.save(*this, fromStudy);
      }

      /*
       * This method reloads the PersistentObject from the medium
       */
      void XMLStorageManager::load(Study & study)
      {
	XML::Node node = XML::GetFirstChild( state_.top().root_ );
	while( node ) {
	  state_.top().current_ = node;
	  PersistentObject * p_po = readDOMElement();
	  if (p_po) {
	    study.add(p_po);
	  }
	  node = XML::GetNextNode(node);
	}
      }


      /* Methods to control manager internal state */
      void XMLStorageManager::pushState()
      {
	state_.push(state_.top());
      }

      void XMLStorageManager::popState()
      {
	state_.pop();
      }



      /* Methods to read DOM elements */
      PersistentObject * XMLStorageManager::readDOMElement() const
      {
	PersistentObject * p_po = 0;
	if ( state_.top().current_ ) {
	  if (XML::IsElement(state_.top().current_, getEntityString(ObjectEntity) ) ) {
	    String className = XML::GetAttributeByName(state_.top().current_, getAttributeString(ClassAttribute));
	    String stFromStudy = XML::GetAttributeByName(state_.top().current_, getAttributeString(StudyVisibleAttribute));
	    if (stFromStudy.empty()) stFromStudy = "false";
	    p_po = Catalog::GetInstance().get(className).build(*this);
	    p_po->setVisibility(stFromStudy == "true");
	  }
	}

	return p_po;
      }



      /* Methods to write out objects */
      void XMLStorageManager::writeObjectPrologue(const PersistentObject & obj, bool fromStudy)
      {
	// We create the node
	state_.top().current_ = XML::NewNode(getEntityString(ObjectEntity));
	XML::SetAttribute(state_.top().current_,
			  getAttributeString(ClassAttribute),
			  obj.getClassName() );
	XML::SetAttribute(state_.top().current_,
			  getAttributeString(StudyVisibleAttribute),
			  (fromStudy ? "true" : "false") );
      }

      void XMLStorageManager::writeObjectEpilogue(const PersistentObject & obj)
      {
	// We insert the node in the document structure
	XML::AddChild( state_.top().root_, state_.top().current_ );
      }

	


      /* List methods */
      StorageManager::List XMLStorageManager::getList(EntityName entity) const
      {
	return XMLList(*this, state_.top(), entity);
      }



      /* Methods to write out class members */
      void XMLStorageManager::writeAttribute(AttributeName attribute, const String & st)
      {
	XML::SetAttribute(state_.top().current_, getAttributeString(attribute), st);
      }

      void XMLStorageManager::writeAttribute(AttributeName attribute, UnsignedLong ul)
      {
	OSS oss;
	oss << ul;
	XML::SetAttribute(state_.top().current_, getAttributeString(attribute), oss);
      }

      void XMLStorageManager::writeValue(UnsignedLong index, NumericalScalar value)
      {
	OSS oss1;
	oss1 << index;
	OSS oss2;
	oss2.setPrecision(20) << value;

	state_.top().sub_ = XML::NewNode(getEntityString(NumericalScalarEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IndexAttribute), oss1);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(UnsignedLong index, NumericalComplex value)
      {
	OSS oss1;
	oss1 << index;
	OSS oss2;
	oss2.setPrecision(20) << value;

	state_.top().sub_ = XML::NewNode(getEntityString(NumericalComplexEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IndexAttribute), oss1);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(UnsignedLong index, const PersistentObject & obj)
      {
	OSS oss1;
	oss1 << index;
	OSS oss2;
	oss2 << obj.getClassName();
	OSS oss3;
	oss3 << obj.getId();

	state_.top().sub_ = XML::NewNode(getEntityString(ObjectEntity));
	XML::SetAttribute(state_.top().sub_, getAttributeString(IndexAttribute), oss1);
	XML::SetAttribute(state_.top().sub_, getAttributeString(ClassAttribute), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IdAttribute), oss3);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }


      void XMLStorageManager::writeValue(UnsignedLong index, const String & st)
      {
	OSS oss;
	oss << index;

	state_.top().sub_ = XML::NewNode(getEntityString(StringEntity), st);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IndexAttribute), oss);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(UnsignedLong index, UnsignedLong value)
      {
	OSS oss;
	oss << index;
	OSS oss2;
	oss2 << value;

	state_.top().sub_ = XML::NewNode(getEntityString(UnsignedLongEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IndexAttribute), oss);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(UnsignedLong index, Bool value)
      {
	OSS oss;
	oss << index;
	OSS oss2;
	oss2 << value;

	state_.top().sub_ = XML::NewNode(getEntityString(BoolEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IndexAttribute), oss);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(const String & name, NumericalScalar value)
      {
	OSS oss2;
	oss2.setPrecision(20) << value;

	state_.top().sub_ = XML::NewNode(getEntityString(NumericalScalarEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(NameAttribute), name);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(const String & name, NumericalComplex value)
      {
	OSS oss2;
	oss2.setPrecision(20) << value;

	state_.top().sub_ = XML::NewNode(getEntityString(NumericalComplexEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(NameAttribute), name);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(const String & name, const PersistentObject & obj)
      {
	OSS oss2;
	oss2 << obj.getClassName();
	OSS oss3;
	oss3 << obj.getId();

	state_.top().sub_ = XML::NewNode(getEntityString(ObjectEntity));
	XML::SetAttribute(state_.top().sub_, getAttributeString(NameAttribute), name);
	XML::SetAttribute(state_.top().sub_, getAttributeString(ClassAttribute), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IdAttribute), oss3);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }


      void XMLStorageManager::writeValue(const String & name, const String & st)
      {
	state_.top().sub_ = XML::NewNode(getEntityString(StringEntity), st);
	XML::SetAttribute(state_.top().sub_, getAttributeString(NameAttribute), name);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(const String & name, UnsignedLong value)
      {
	OSS oss2;
	oss2 << value;

	state_.top().sub_ = XML::NewNode(getEntityString(UnsignedLongEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(NameAttribute), name);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(const String & name, Bool value)
      {
	OSS oss2;
	oss2 << value;

	state_.top().sub_ = XML::NewNode(getEntityString(BoolEntity), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(NameAttribute), name);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }

      void XMLStorageManager::writeValue(const PersistentObject & obj)
      {
	OSS oss2;
	oss2 << obj.getClassName();
	OSS oss3;
	oss3 << obj.getId();

	state_.top().sub_ = XML::NewNode(getEntityString(ObjectEntity));
	XML::SetAttribute(state_.top().sub_, getAttributeString(ClassAttribute), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IdAttribute), oss3);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }


      void XMLStorageManager::writeValue(const PersistentObject & obj, AttributeName attribute, const String & st)
      {
	OSS oss2;
	oss2 << obj.getClassName();
	OSS oss3;
	oss3 << obj.getId();

	state_.top().sub_ = XML::NewNode(getEntityString(ObjectEntity));
	XML::SetAttribute(state_.top().sub_, getAttributeString(ClassAttribute), oss2);
	XML::SetAttribute(state_.top().sub_, getAttributeString(IdAttribute), oss3);
	XML::SetAttribute(state_.top().sub_, getAttributeString(attribute), st);
	XML::AddChild( state_.top().current_, state_.top().sub_ );
      }






      /* Methods to read class members */
      void XMLStorageManager::readAttribute(AttributeName attribute, String & st)
      {
	st = XML::GetAttributeByName( state_.top().current_, getAttributeString(attribute) );
      }

      void XMLStorageManager::readAttribute(AttributeName attribute, UnsignedLong & ul)
      {
	String stul = XML::GetAttributeByName( state_.top().current_, getAttributeString(attribute) );
	std::istringstream iss (stul);
	iss >> ul;
      }

      void XMLStorageManager::readValue(UnsignedLong & index, NumericalScalar & value)
      {
	StorageManager::List objectList = getList( NumericalScalarEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, value );
      }

      void XMLStorageManager::readValue(UnsignedLong & index, NumericalComplex & value)
      {
	StorageManager::List objectList = getList( NumericalComplexEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, value );
      }

      void XMLStorageManager::readValue(UnsignedLong & index, InterfaceObject & obj)
      {
	StorageManager::List objectList = getList( ObjectEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, obj );
      }


      void XMLStorageManager::readValue(UnsignedLong & index, PersistentObject & obj)
      {
	StorageManager::List objectList = getList( ObjectEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, obj );
      }

      void XMLStorageManager::readValue(UnsignedLong & index, String & st)
      {
	StorageManager::List objectList = getList( StringEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, st );
      }

      void XMLStorageManager::readValue(UnsignedLong & index, UnsignedLong & value)
      {
	StorageManager::List objectList = getList( UnsignedLongEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, value );
      }

      void XMLStorageManager::readValue(UnsignedLong & index, Bool & value)
      {
	StorageManager::List objectList = getList( BoolEntity );
	objectList.firstValueToRead();
	objectList.readValue( index, value );
      }

      void XMLStorageManager::readValue(String & name, NumericalScalar & value)
      {
	StorageManager::List objectList = getList( NumericalScalarEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, value );
      }

      void XMLStorageManager::readValue(String & name, NumericalComplex & value)
      {
	StorageManager::List objectList = getList( NumericalComplexEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, value );
      }

      void XMLStorageManager::readValue(String & name, InterfaceObject & obj)
      {
	StorageManager::List objectList = getList( ObjectEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, obj );
      }

      void XMLStorageManager::readValue(String & name, PersistentObject & obj)
      {
	StorageManager::List objectList = getList( ObjectEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, obj );
      }

      void XMLStorageManager::readValue(String & name, String & st)
      {
	StorageManager::List objectList = getList( StringEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, st );
      }

      void XMLStorageManager::readValue(String & name, UnsignedLong & value)
      {
	StorageManager::List objectList = getList( UnsignedLongEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, value );
      }

      void XMLStorageManager::readValue(String & name, Bool & value)
      {
	StorageManager::List objectList = getList( BoolEntity );
	objectList.firstValueToRead();
	objectList.readValue( name, value );
      }
      
      void XMLStorageManager::readValue(InterfaceObject & obj, AttributeName attribute, const String & st)
      {
	StorageManager::List objectList = getList( ObjectEntity );
	objectList.readValue(obj, attribute, st);
      }

      void XMLStorageManager::readValue(PersistentObject & obj, AttributeName attribute, const String & st)
      {
	StorageManager::List objectList = getList( ObjectEntity );
	objectList.readValue(obj, attribute, st);
      }





    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
