//                                               -*- C++ -*-
/**
 * @file  ComputedNumericalMathEvaluationImplementation.hxx
 * @brief Abstract top-level class for all ComputedNumerical math functions
 *
 * (C) Copyright 2005-2006 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2008-10-17 14:26:02 +0200 (ven 17 oct 2008) $
 */

#ifndef OPENTURNS_COMPUTEDNUMERICALMATHFUNCTIONIMPLEMENTATION_HXX
#define OPENTURNS_COMPUTEDNUMERICALMATHFUNCTIONIMPLEMENTATION_HXX

#include "NumericalMathEvaluationImplementation.hxx"
#include "Pointer.hxx"
#include "NumericalPoint.hxx"
#include "WrapperFile.hxx"
#include "WrapperObject.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class ComputedNumericalMathEvaluationImplementation
       *
       * This class acts like an real numerical mathematical function
       * into the platform. It realizes the computation of the image vector
       * of a vector (aka NumericalPoint) through the function. The
       * function may be loaded in an external way through a wrapper.
       */
      class ComputedNumericalMathEvaluationImplementation 
	: public NumericalMathEvaluationImplementation
      {
	CLASSNAME;
	friend class NumericalMathFunctionImplementation;

      public:

	/* Some typedefs to ease reading */
	typedef Common::WrapperInternalException                                WrapperInternalException;
	typedef NumericalMathEvaluationImplementation::InvalidArgumentException InvalidArgumentException;
	typedef NumericalMathEvaluationImplementation::InternalException        InternalException;
	typedef Common::Pointer<WrapperObject>                                  InternalFunction;
	typedef NumericalMathEvaluationImplementation::NumericalPoint           NumericalPoint;
	typedef NumericalMathEvaluationImplementation::NumericalSample          NumericalSample;
	typedef NumericalMathEvaluationImplementation::Description              Description;

	/** Default constructor */
	explicit ComputedNumericalMathEvaluationImplementation(const String & name, const WrapperFile & file)
	  throw(WrapperInternalException);

	/** Copy constructor */
	ComputedNumericalMathEvaluationImplementation(const ComputedNumericalMathEvaluationImplementation & other)
	  throw(WrapperInternalException);

	/** Virtual constructor */
	virtual ComputedNumericalMathEvaluationImplementation * clone() const;

	/** Destructor */
	virtual ~ComputedNumericalMathEvaluationImplementation();

	/** Comparison operator */
	Bool operator ==(const ComputedNumericalMathEvaluationImplementation & other) const;

	/** String converter */
	String str() const;



	/* Here is the interface that all derived class must implement */
	
	/** Operator () */
	virtual NumericalPoint operator() (const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException);
	virtual NumericalSample operator() (const NumericalSample & in) const
	  throw(InvalidArgumentException,InternalException);

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for description */
	virtual Description getDescription() const
	  throw(InternalException);

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);


      protected:

	/** The function object that is linked to the external code */
	InternalFunction p_function_;

	/** State management */
    	void * getState() const;

      private:

	ComputedNumericalMathEvaluationImplementation() {};
	friend class Common::Factory<ComputedNumericalMathEvaluationImplementation>;

	/** A pointer to an internal structure that saves the state of the function into the wrapper */
	void * p_state_;

      }; /* class ComputedNumericalMathEvaluationImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_COMPUTEDNUMERICALMATHFUNCTIONIMPLEMENTATION_HXX */
