//                                               -*- C++ -*-
/**
 *  @file  ComputedNumericalMathHessianImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: ComputedNumericalMathHessianImplementation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "ComputedNumericalMathHessianImplementation.hxx"
#include "ComputedNumericalMathHessianImplementationFactory.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ComputedNumericalMathHessianImplementation);

      static Common::Factory<ComputedNumericalMathHessianImplementation> RegisteredFactory("ComputedNumericalMathHessianImplementation");

      /* Default constructor */
      ComputedNumericalMathHessianImplementation::ComputedNumericalMathHessianImplementation(const String & name,
											     const WrapperFile & file,
											     void * p_state)
	throw(WrapperInternalException)
	: NumericalMathHessianImplementation(),
	  p_hessian_(0),
	  p_state_(0),
	  commonState_(file.getWrapperData().getParameters().state_ == WRAPPER_SHAREDSTATE)
      {
	setName(name);
	if (file.getWrapperData().getHessianDescription().provided_) {
	  p_hessian_.reset(new WrapperObject( file.getWrapperData().getLibraryPath(),
					      file.getWrapperData().getHessianDescription().name_,
					      file.getWrapperData(),
					      WrapperObject::HESSIAN ));
	}

	if (p_hessian_.isNull()) throw WrapperInternalException(HERE) << "Unable to allocate wrapper for hessian (Note: this message is normal if hessian is NOT provided. Check wrapper description file '" << file.getDescriptionFilePath() << "'.)";

	// Initialize the state into the wrapper
	p_state_ = commonState_ ? p_state : p_hessian_->createNewState();
      }


      /* Copy constructor */
      ComputedNumericalMathHessianImplementation::ComputedNumericalMathHessianImplementation(const ComputedNumericalMathHessianImplementation & other)
	throw(WrapperInternalException)
	: NumericalMathHessianImplementation(other),
	  p_hessian_(other.p_hessian_),
	  p_state_(0),
	  commonState_(other.commonState_)
      {
	if (p_hessian_.isNull()) throw WrapperInternalException(HERE) << "Unable to allocate wrapper";

	// Initialize the state into the wrapper
	p_state_ = commonState_ ? other.p_state_ : p_hessian_->createNewState();
      }


      /* Virtual constructor */
      ComputedNumericalMathHessianImplementation * ComputedNumericalMathHessianImplementation::clone() const
      {
	return new ComputedNumericalMathHessianImplementation(*this);
      }

      /* Destructor */
      ComputedNumericalMathHessianImplementation::~ComputedNumericalMathHessianImplementation()
      {
	// Delete the state into the wrapper
	if (!commonState_) p_hessian_->deleteState( p_state_ );
      }


      /* Comparison operator */
      Bool ComputedNumericalMathHessianImplementation::operator ==(const ComputedNumericalMathHessianImplementation & other) const
      {
	return true;
      }

  
      /* String converter */
      String ComputedNumericalMathHessianImplementation::str() const {
	OSS oss;
	oss << "class=" << ComputedNumericalMathHessianImplementation::GetClassName()
	    << " name=" << getName();
	return oss;
      }

  
      /* State accessor */
      void * ComputedNumericalMathHessianImplementation::getState() const
      {
	return p_state_;
      }







      /* Here is the interface that all derived class must implement */
	
      /* Hessian method */
      ComputedNumericalMathHessianImplementation::SymmetricTensor
      ComputedNumericalMathHessianImplementation::hessian(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	try {
	  // Execute the wrapper
	  callsNumber_++;
	  return p_hessian_->hessian( getState(), in );
	}
	catch (InvalidArgumentException & ex) {
	  throw;
	}
 	catch (WrapperInternalException & ex) {
	  throw InternalException(HERE) << ex.str();
	}
      }



      /* Accessor for input point dimension */
      UnsignedLong ComputedNumericalMathHessianImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	UnsignedLong inputDimension = 0;

	try {
	  inputDimension = p_hessian_->getInNumericalPointDimension(getState());
	}
 	catch (WrapperInternalException & ex) {
	  throw InternalException(HERE) << ex.str();
	}

	return inputDimension;
      }



      /* Accessor for output point dimension */
      UnsignedLong ComputedNumericalMathHessianImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	UnsignedLong outputDimension = 0;

	try {
	  outputDimension = p_hessian_->getOutNumericalPointDimension(getState());
	}
 	catch (WrapperInternalException & ex) {
	  throw InternalException(HERE) << ex.str();
	}

	return outputDimension;
      }

      /* Method save() stores the object through the StorageManager */
      void ComputedNumericalMathHessianImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathHessianImplementation::save(adv);
      }

      /* Method load() reloads the object from the StorageManager */
      void ComputedNumericalMathHessianImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathHessianImplementation::load(adv);
	*this = *static_cast<ComputedNumericalMathHessianImplementation*>(ComputedNumericalMathHessianImplementationFactory::getInstance()->buildObject(getName()));
      }

      
    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
