//                                               -*- C++ -*-
/**
 *  @file  LibraryLoader.cxx
 *  @brief The class LibraryLoader loads ans manages all dynamic libraries
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-22 11:34:11 +0200 (lun 22 sep 2008) $
 *  Id:      $Id: LibraryLoader.cxx 941 2008-09-22 09:34:11Z dutka $
 */
#include <vector>                // for std::vector
#include <dlfcn.h>               // for dlopen, dlsym, dlclose, dlerror
#include "OSS.hxx"
#include "LibraryLoader.hxx"
#include "Path.hxx"
#include "Exception.hxx"
#include "Log.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      CLASSNAMEINIT(LibraryLoader);

      typedef OT::Base::Common::Log Log;

      /* The unique instance of the Singleton model */
      LibraryLoader * LibraryLoader::p_instance_ = 0;

      /* Default constructor */
      LibraryLoader::LibraryLoader()
	: libraryCatalog_()
      {
        // Nothing to do
      }

      /* Copy constructor */
      LibraryLoader::LibraryLoader(const LibraryLoader & other ///< The copied LibraryLoader
				   )
	: libraryCatalog_()
      {
        // Nothing to do
      }

      /* Destructor */
      LibraryLoader::~LibraryLoader()
      {
	Lock lock(this);

        // We close all libraries one after another
	CatalogType::const_iterator iterator;
	for(iterator =  libraryCatalog_.begin();
	    iterator != libraryCatalog_.end();
	    iterator++) {
	  if (dlclose((*iterator).second))
	    throw DynamicLibraryException(HERE) << "Cannot dlclose library located at '"
						<< (*iterator).first << "'. Reason : "
						<< dlerror();
	} /* end for */

      }


      /* Assignment */
      LibraryLoader & LibraryLoader::operator =(const LibraryLoader & other)
      {
	return *this;
      }


      /* String converter */
      String LibraryLoader::str() const
      {
	OSS oss;
	oss << "class=LibraryLoader"
	    << " catalog=[";

	{ /* Lock block */
	  Lock lock(this);

	  const char * separator = "";
	  CatalogType::const_iterator iterator;
	  for(iterator =  libraryCatalog_.begin();
	      iterator != libraryCatalog_.end();
	      iterator++, separator=", ") {
	    oss << separator
		<< "('" << (*iterator).first << "'," << (*iterator).second << ")";
	  } /* end for */

	} /* end Lock block */

	oss << "]";
	return oss;
      }


      /* The accessor to the Singleton element */
      LibraryLoader & LibraryLoader::GetInstance()
      {
	if (!p_instance_) {
	  // this is the first call, we have to create the instance
	  p_instance_ = new LibraryLoader;
	} /* end if */
	return *p_instance_;
      }


      /* 
       * Loads the library located at path
       */
      Library LibraryLoader::load(const FileName & path)
	throw(FileNotFoundException,DynamicLibraryException)
      {
	Lock lock(this);

	CatalogType::const_iterator foundLibraryIterator = libraryCatalog_.find(path);

	// Load the library if not already done
	if (foundLibraryIterator == libraryCatalog_.end()) {

	  // Get the directory list...
	  std::vector<FileName> directoryList = Common::Path::GetWrapperDirectoryList();

	  // ... and search it for the file
	  FileName libraryFullPath;
	  libraryFullPath = Common::Path::FindFileByNameInDirectoryList(path, directoryList);
	  

	  void * theLibraryHandle = dlopen(libraryFullPath.c_str(), RTLD_NOW);
	  if (theLibraryHandle == NULL)
	    throw DynamicLibraryException(HERE) << "Cannot dlopen library located at '"
						<< libraryFullPath << "'. Reason : "
						<< dlerror();

	  Log::Info(OSS() << "Successful load of dynamic library located at " << libraryFullPath );

	  // Add the library to the catalog
	  libraryCatalog_[path] = theLibraryHandle;

	  return Library(theLibraryHandle, path);


	} else { // The library is already loaded
	  return Library((*foundLibraryIterator).second, (*foundLibraryIterator).first);
	} /* end if */
      }



      /*
       * @fn std::ostream & operator <<(std::ostream & os, const LibraryLoader & obj)
       * @brief Output stream converter
       * @param os A STL output stream object
       * @param obj The object read by \em os
       * @return A reference to \em os
       */
      std::ostream & operator <<(std::ostream & os, const LibraryLoader & obj)
      {
        return os << obj.str();
      }



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
