//                                               -*- C++ -*-
/**
 *  @file  BoundConstrainedAlgorithm.hxx
 *  @brief BoundConstrainedAlgorithm implements an algorithm for finding the 
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: BoundConstrainedAlgorithm.hxx 929 2008-09-13 20:37:56Z dutka $
 */
#ifndef OPENTURNS_BOUNDCONSTRAINEDALGORITHM_HXX
#define OPENTURNS_BOUNDCONSTRAINEDALGORITHM_HXX

#include "TypedInterfaceObject.hxx"
#include "Threadable.hxx"
#include "BoundConstrainedAlgorithmImplementation.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optimisation
    {

      /**
       * @class BoundConstrainedAlgorithm
       * BoundConstrainedAlgorithm implements an interface to an algorithm for finding the 
       * point of an interval that minimize the given objective function
       */

      class BoundConstrainedAlgorithm
	: public Common::TypedInterfaceObject<BoundConstrainedAlgorithmImplementation>,
	  public Common::Threadable
      {

	CLASSNAME;
      public:

	typedef Common::Pointer<BoundConstrainedAlgorithmImplementation>          Implementation;
	typedef BoundConstrainedAlgorithmImplementation::NumericalMathFunction    NumericalMathFunction;
	typedef BoundConstrainedAlgorithmImplementation::NumericalPoint           NumericalPoint;
	typedef BoundConstrainedAlgorithmImplementation::Interval                 Interval;
	typedef BoundConstrainedAlgorithmImplementation::InternalException        InternalException;
	typedef BoundConstrainedAlgorithmImplementation::InvalidArgumentException InvalidArgumentException;
	typedef BoundConstrainedAlgorithmImplementation::OptimizationProblem      OptimizationProblem;
#ifndef SWIG
	typedef BoundConstrainedAlgorithmImplementation::Result                   Result;
#else
	typedef BoundConstrainedAlgorithmImplementationResult                     Result;
#endif

	/** Default constructor */
	explicit BoundConstrainedAlgorithm();

	/** Constructor from an implementation */
	BoundConstrainedAlgorithm(const BoundConstrainedAlgorithmImplementation & implementationconst);

	/** Constructor from a Pointer to an implementation */
	BoundConstrainedAlgorithm(const Implementation & p_implementation);

	/** Constructor with parameters: no constraint, starting from the origin */
	BoundConstrainedAlgorithm(const NumericalMathFunction & objectiveFunction,
				  const Bool verbose = false);

	/** Constructor with parameters: bound constraints, starting from the given point */
	BoundConstrainedAlgorithm(const NumericalMathFunction & objectiveFunction,
				  const Interval & boundConstraints,
				  const NumericalPoint & startingPoint,
				  const OptimizationProblem optimization  = BoundConstrainedAlgorithmImplementation::MINIMIZATION,
				  const Bool verbose = false) throw(InvalidArgumentException);

	/** Performs the actual computation. Must be overloaded by the actual optimisation algorithm */
	virtual void run()
          throw(InternalException);

	/** Starting point accessor */
	NumericalPoint getStartingPoint() const;
	void setStartingPoint(const NumericalPoint & startingPoint);

	/** Objective function accessor */
	NumericalMathFunction getObjectiveFunction() const;
	void setObjectiveFunction(const NumericalMathFunction & objectiveFunction);

	/** Bound constraints accessor */
	Interval getBoundConstraints() const;
	void setBoundConstraints(const Interval & boundConstraints);

	/** Optimization problem accessor */
	OptimizationProblem getOptimizationProblem() const;
	void setOptimizationProblem(const OptimizationProblem optimization);

	/** Result accessor */
	Result getResult() const;
	void setResult(const Result & result);

	/** Maximum iterations number accessor */
	UnsignedLong getMaximumEvaluationsNumber() const;
	void setMaximumEvaluationsNumber(const UnsignedLong maximumEvaluationsNumber);

	/** Maximum absolute error accessor */
	NumericalScalar getMaximumAbsoluteError() const;
	void setMaximumAbsoluteError(const NumericalScalar maximumAbsoluteError);

	/** Maximum relative error accessor */
	NumericalScalar getMaximumRelativeError() const;
	void setMaximumRelativeError(const NumericalScalar maximumRelativeError);

	/** Maximum objective error accessor */
	NumericalScalar getMaximumObjectiveError() const;
	void setMaximumObjectiveError(const NumericalScalar maximumObjectiveError);

	/** Maximum constraint error accessor */
	NumericalScalar getMaximumConstraintError() const;
	void setMaximumConstraintError(const NumericalScalar maximumConstraintError);

	/** String converter */
	virtual String str() const;

	/** Verbose accessor */
	Bool getVerbose() const;
	void setVerbose(const Bool verbose);

      } ; /* class BoundConstrainedAlgorithm */

    } /* namespace Optimisation */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_BOUNDCONSTRAINEDALGORITHM_HXX */
