//                                               -*- C++ -*-
/**
 *  @file  LinearModel.cxx
 *  @brief LinearModel implements the linear model
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: LinearModel.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <fstream>
#include "LinearModel.hxx"
#include "Path.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"
#include "Matrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {
      using Stat::ConfidenceInterval;

      TEMPLATE_CLASSNAMEINIT(PersistentCollection<ConfidenceInterval>);
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<NumericalScalar>);

      static Common::Factory<PersistentCollection<ConfidenceInterval> > RegisteredFactory1("PersistentCollection<ConfidenceInterval>");
      static Common::Factory<PersistentCollection<NumericalScalar> > RegisteredFactory2("PersistentCollection<NumericalScalar>");
    } /* namespace Type */



    namespace Stat
    {
      CLASSNAMEINIT(LinearModel);

      static Common::Factory<LinearModel> RegisteredFactory("LinearModel");

      typedef Common::Path      Path;
      typedef Common::Exception InternalException;
      typedef Type::Matrix      Matrix;

      /* Default constructor */
      LinearModel::LinearModel()
      {
	// Nothing to do
      }

      /* Standard constructor */
      LinearModel::LinearModel(const NumericalPoint & vectorR,
                               const ConfidenceIntervalPersistentCollection & intervalsColl,
                               const NumericalScalarPersistentCollection & pValuesOfR)
	throw(InvalidArgumentException)
	: PersistentObject(),
	  regression_(vectorR),
	  confidenceIntervals_(intervalsColl),
	  pValues_(pValuesOfR)
      {
	// Nothing to do
      }

      /* Constructor from NumericalPoint */
      LinearModel::LinearModel(const NumericalPoint & vectorR)
	: PersistentObject(),
	  regression_(vectorR)
      {
	UnsignedLong dimension(vectorR.getDimension());
	ConfidenceIntervalPersistentCollection intervalsColl(dimension, ConfidenceInterval(0., 0.));
	NumericalScalarPersistentCollection pValuesOfR(dimension, 0);
	  
	for (UnsignedLong i = 0; i < dimension; i++) intervalsColl[i] = ConfidenceInterval(vectorR[i], vectorR[i]);
	  
	confidenceIntervals_ = intervalsColl;
	pValues_ = pValuesOfR;
      }

      /* Virtual constructor */
      LinearModel * LinearModel::clone() const
      {
	return new LinearModel(*this); 
      }

      /* String converter */
      String LinearModel::str() const
      {
	return OSS() << "class=" << LinearModel::GetClassName()
		     << " name=" << getName()
		     << " regression =" << getRegression()
		     << " confidence intervals =" << getConfidenceIntervals()
		     << " p-Values =" << getPValues();
      }
	
      /* get vector, get intervals, get p-Values*/
      LinearModel::NumericalPoint LinearModel::getRegression() const
      {
	return regression_;
      }
	
      LinearModel::ConfidenceIntervalPersistentCollection LinearModel::getConfidenceIntervals() const
      {
	return confidenceIntervals_;
      }
	
      LinearModel::NumericalScalarPersistentCollection LinearModel::getPValues() const
      {
	return pValues_;
      }
	
      /* getPredicted : build an sample of predicted values */
      NumericalSample LinearModel::getPredicted(const NumericalSample & predictor) const throw(InvalidArgumentException)
      {
       	if (predictor.getDimension() + 1 != regression_.getDimension()) throw InvalidArgumentException(HERE) << "Error: predictors must have a dimension compatible with the linear model dimension";
	UnsignedLong size(predictor.getSize());
	NumericalSample predicted(size, 1);
	UnsignedLong dimension(predictor.getDimension());
	NumericalPoint linear(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    linear[i] = regression_[i + 1];
	  }
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    predicted[i][0] = NumericalPoint::dot(linear, predictor[i]) + regression_[0];
	  }
	return predicted;
      }
	
      /* getResidual */
      NumericalSample LinearModel::getResidual(const NumericalSample & predictor,
                                               const NumericalSample & measured) const throw(InvalidArgumentException)
      {
       	if (predictor.getDimension() + 1 != regression_.getDimension()) throw InvalidArgumentException(HERE) << "Error: predictors must have a dimension compatible with the linear model dimension";
	UnsignedLong size(predictor.getSize());
	if (measured.getSize() != size) throw InvalidArgumentException(HERE) << "Error: measured must have the same size as predictor";
	NumericalSample residual(size, 1);
	UnsignedLong dimension(predictor.getDimension());
	NumericalPoint linear(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    linear[i] = regression_[i + 1];
	  }
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    residual[i][0] = measured[i][0] - NumericalPoint::dot(linear, predictor[i]) - regression_[0];
	  }
	return residual;
      }

      /* Method save() stores the object through the StorageManager */
      void LinearModel::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeValue(regression_, StorageManager::MemberNameAttribute, "regression_");
 	adv.writeValue(confidenceIntervals_, StorageManager::MemberNameAttribute, "confidenceIntervals_");
 	adv.writeValue(pValues_, StorageManager::MemberNameAttribute, "pValues_");
      }

      /* Method load() reloads the object from the StorageManager */
      void LinearModel::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	adv.readValue(regression_, StorageManager::MemberNameAttribute, "regression_");
 	adv.readValue(confidenceIntervals_, StorageManager::MemberNameAttribute, "confidenceIntervals_");
 	adv.readValue(pValues_, StorageManager::MemberNameAttribute, "pValues_");
      }

      
      /* Comparison operator */
      Bool operator ==(const LinearModel & lhs, const LinearModel & rhs)
      {
	return ( lhs.getRegression()==rhs.getRegression() && lhs.getConfidenceIntervals()==rhs.getConfidenceIntervals() && lhs.getPValues()==rhs.getPValues() );
      }  
	
    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
