//                                               -*- C++ -*-
/**
 *  @file  NumericalSample.cxx
 *  @brief The class NumericalSample implements blank free samples
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalSample.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <iomanip>
#include <fstream>
#include <stdexcept>
#include "NumericalSample.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      CLASSNAMEINIT(NumericalSample);


      /* Factory of NumericalSample from CSV file */
      NumericalSample NumericalSample::ImportFromCSVFile(const FileName & fileName)
      {
	NumericalSample sample(NumericalSampleImplementation::GetFromFile(fileName, NumericalSampleImplementation::CSV));
	sample.setName(fileName);
	return sample;
      }

      /* Save to CSV file */
      void NumericalSample::exportToCSVFile(const FileName & filename) const
      {
	getImplementation()->exportToCSVFile(filename);
      }


      /* Store a sample in a temporary text file, one realization by line. Returns the file name. */
      String NumericalSample::storeToTemporaryFile() const
      {
	return getImplementation()->storeToTemporaryFile();
      }

      /* Export a sample as a matrix, one row by realization, in a format suitable to exchange with R. */
      String NumericalSample::streamToRFormat() const
      {
	return getImplementation()->streamToRFormat();
      }

      /* Default constructor */
      NumericalSample::NumericalSample()
	: Common::TypedCollectionInterfaceObject<NumericalSampleImplementation>(new NumericalSampleImplementation(0, 1))
      {
        // Nothing to do
      }

      /* Constructor with size and dimension */
      NumericalSample::NumericalSample(UnsignedLong size,
				       UnsignedLong dim)
	: Common::TypedCollectionInterfaceObject<NumericalSampleImplementation>(new NumericalSampleImplementation(size, dim))
      {
        // Nothing to do
      }

      /* Constructor from implementation */
      NumericalSample::NumericalSample(const NumericalSampleImplementation & implementation)
	: Common::TypedCollectionInterfaceObject<NumericalSampleImplementation>(implementation.clone())
      {
	// Nothing to do
      }

      /* Constructor from implementation */
      NumericalSample::NumericalSample(const Implementation & implementation)
	: Common::TypedCollectionInterfaceObject<NumericalSampleImplementation>(implementation)
      {
	// Nothing to do
      }

      /* Constructor from a NumericalPoint (all elements are equal to the NumericalPoint) */
      NumericalSample::NumericalSample(const UnsignedLong size,
				       const NumericalPoint & point)
	: Common::TypedCollectionInterfaceObject<NumericalSampleImplementation>(new NumericalSampleImplementation(size, point))
      {
	// Nothing to do
      }

      /* Comparison operator */
      Bool NumericalSample::operator ==(const NumericalSample & other) const
      {
	return true;
      }

      /* String converter */
      String NumericalSample::str() const
      {
	return OSS() << "class=" << getClassName()
		     << " name=" << getName()
		     << " description=" << getDescription()
		     << " implementation=" << getImplementation()->str();
      }


      /* Description accessor */
      void NumericalSample::setDescription(const NumericalSample::Description & description)
      {
	copyOnWrite();
	getImplementation()->setDescription(description);
      }



      /* Description accessor */
      NumericalSample::Description NumericalSample::getDescription() const
      {
	return getImplementation()->getDescription();
      }


      /* Element accessor */
      NumericalSample::NumericalPoint & NumericalSample::operator[](const UnsignedLong index)
	throw (OutOfBoundException, InvalidArgumentException)
      {
	copyOnWrite();

	try {
	  return getImplementation()->at(index);

	} catch (std::out_of_range & ex) {
	  throw OutOfBoundException(HERE) << "sample size=" << getSize()
					  << " - erroneous index=" << index
					  << " - " << ex.what();
	}
      }


      /* Element accessor */
      const NumericalSample::NumericalPoint & NumericalSample::operator[](const UnsignedLong index) const
	throw (OutOfBoundException,InvalidArgumentException)
      {
	try {
	  return getImplementation()->at(index);

	} catch (std::out_of_range & ex) {
	  throw OutOfBoundException(HERE) << "sample size=" << getSize()
					  << " - erroneous index=" << index
					  << " - " << ex.what();
	}
      }


      /* Method __getitem__() is for Python */
      const NumericalSample::NumericalPoint & NumericalSample::__getitem__(const UnsignedLong i) const
      {
	return (*this)[i];
      }
      
      /* Method __setitem__() is for Python */
      void NumericalSample::__setitem__(UnsignedLong i, const NumericalPoint & val)
      {
	(*this)[i] = val;
      }


      /* Dimension accessor */
      UnsignedLong NumericalSample::getDimension() const
      {
	return getImplementation()->getDimension();
      }

      
      /* Size accessor */
      UnsignedLong NumericalSample::getSize() const
      {
	return getImplementation()->getSize();
      }

      /* Maximum accessor */
      NumericalSample::NumericalPoint NumericalSample::getMax() const
      {
	return getImplementation()->getMax();
      }

      /* Minimum accessor */
      NumericalSample::NumericalPoint NumericalSample::getMin() const
      {
	return getImplementation()->getMin();
      }


      /* Method add() appends an element to the collection */
      void NumericalSample::add(const NumericalPoint & point)
	throw(InvalidArgumentException)
      {
	if ( (getSize() > 0) && (getDimension() != point.getDimension()) )
	  throw InvalidArgumentException(HERE)
	    << "Point has invalid dimension (dim=" << point.getDimension()
	    << ") for sample (dim=" << getDimension() << ")";
	
	copyOnWrite();
	getImplementation()->add(point);
      }



      /*
       * Method split() trunk the sample before the index passed as argument
       * and returns the remainder as new sample. This method tries its best not for doubling
       * memory usage.
       */
      NumericalSample NumericalSample::split(const UnsignedLong index)
	throw (OutOfBoundException)
      {
	// We first save the size of the original sample
	UnsignedLong theSize = getSize();

	// We first check that the index is in the sample's range
	if (index >= theSize) throw OutOfBoundException(HERE) << "Index over size. Index=" << index << " size=" << theSize;

	if (index >= theSize/2) { // Strategy 1.
	  // We take two iterators :
	  //  * one on the first element of the returned sample (at the i-th element)
	  //  * one at the end of the returned sample
	  NumericalSampleImplementation::iterator first = begin() + index;
	  NumericalSampleImplementation::iterator last  = end();
	  
	  // We create a new sample by copy-constructing it from the two iterators
	  NumericalSample newSample(new NumericalSampleImplementation(*getImplementation(), first, last));

	  // We clear the copied elements of the original sample (*this)
	  erase(first,last);

	  // We return the new sample
	  return newSample;

	} else { // Strategy 2.
	  // We take two iterators :
	  //  * one at the beginning of the original sample
	  //  * one before the beginning of the returned sample
	  NumericalSampleImplementation::iterator first = getImplementation()->begin();
	  NumericalSampleImplementation::iterator last  = getImplementation()->begin() + index;
	  
	  // We create a new sample by copy-constructing it from the two iterators
	  NumericalSample newSample(new NumericalSampleImplementation(*getImplementation(), first, last));
	  
	  // We clear the copied elements of the original sample (*this)
	  erase(first,last);

	  // We swap the two samples
	  this->swap(newSample);

	  // We return the new sample
	  return newSample;
	}
      }

      /* 
       * Method computeMean() gives the mean of the sample, based on the formula
       * mean = sum of the elements in the sample / size of the sample
       */
      NumericalSample::NumericalPoint NumericalSample::computeMean() const
      {
	return getImplementation()->computeMean();
      }

      /* 
       * Method computeCovariance() gives the covariance of the sample
       */
      NumericalSample::CovarianceMatrix NumericalSample::computeCovariance() const
      {
	return getImplementation()->computeCovariance();
      }

      /*
       * Method computeStandardDeviation() gives the standard deviation of the sample
       */
      NumericalSample::SquareMatrix NumericalSample::computeStandardDeviation() const
      {
	return getImplementation()->computeStandardDeviation();
      }

      /*
       * Method computeStandardDeviationPerComponent() gives the standard deviation of each component of the sample
       */
      NumericalSample::NumericalPoint NumericalSample::computeStandardDeviationPerComponent() const
      {
	return getImplementation()->computeStandardDeviationPerComponent();
      }

      /*
       * Method computePearsonCorrelation() gives the Pearson correlation matrix of the sample
       */
      NumericalSample::CorrelationMatrix NumericalSample::computePearsonCorrelation() const
      {
	return getImplementation()->computePearsonCorrelation();
      }

      /*
       * Method computeSpearmanCorrelation() gives the Spearman correlation matrix of the sample
       */
      NumericalSample::CorrelationMatrix NumericalSample::computeSpearmanCorrelation() const
      {
	return getImplementation()->computeSpearmanCorrelation();
      }

      /*
       * Method computeKendallTau() gives the Kendall tau matrix of the sample
       */
      NumericalSample::CorrelationMatrix NumericalSample::computeKendallTau() const
      {
	return getImplementation()->computeKendallTau();
      }

      /*
       * Method computeRangePerComponent gives the range of the sample (by component)
       */
      NumericalSample::NumericalPoint NumericalSample::computeRangePerComponent() const
      {
	return getImplementation()->computeRangePerComponent();
      }       
   
      /*
       * Method computeMedian() gives the median of the sample (by component)
       */
      NumericalSample::NumericalPoint NumericalSample::computeMedianPerComponent() const
      {
	return getImplementation()->computeMedianPerComponent();
      }

      
      /*
       * Method computeVariance() gives the variance of the sample (by component)
       */
      NumericalSample::NumericalPoint NumericalSample::computeVariancePerComponent() const
      {
	return getImplementation()->computeVariancePerComponent();
      }            

      /*
       * Method computeSkewness() gives the skewness of the sample (by component)
       */
      NumericalSample::NumericalPoint NumericalSample::computeSkewnessPerComponent() const
      {
	return getImplementation()->computeSkewnessPerComponent();
      }        

      /*
       * Method computeKurtosis() gives the kurtosis of the sample (by component)
       */
      NumericalSample::NumericalPoint NumericalSample::computeKurtosisPerComponent() const
      {
	return getImplementation()->computeKurtosisPerComponent();
      }       
      
      /*
       * Method computeQuantilePerComponent() gives the quantile per component of the sample
       */
      NumericalSample::NumericalPoint NumericalSample::computeQuantilePerComponent(const NumericalScalar prob) const
      {
	return getImplementation()->computeQuantilePerComponent(prob);
      }

      /*
       * Method computeQuantile() gives the N-dimension quantile of the sample
       */
      NumericalSample::NumericalPoint NumericalSample::computeQuantile(const NumericalScalar prob) const
      {
	return getImplementation()->computeQuantile(prob);
      }

      /*
       * Get the empirical CDF of the sample
       */
      NumericalScalar NumericalSample::computeEmpiricalCDF(const NumericalPoint & point) const
      {
	return getImplementation()->computeEmpiricalCDF(point);
      }

      /*
       * Get the position of a point in the sample.
       * Returns size+1 if the point does not belong to the sample.
       */
      UnsignedLong NumericalSample::find(const NumericalPoint & point) const
      {
	return std::find(getImplementation()->begin(),  getImplementation()->end(), point) - getImplementation()->begin();
      }

      /*
       * Translate realizations in-place
       */
      void NumericalSample::translate(const NumericalPoint & translation)
      {
	copyOnWrite();
	getImplementation()->translate(translation);
      }

      /*
       * Scale realizations componentwise in-place
       */
      void NumericalSample::scale(const NumericalPoint & scaling)
      {
	copyOnWrite();
	getImplementation()->scale(scaling);
      }

      /* Ranked sample */
      NumericalSample NumericalSample::rank() const
      {
	return getImplementation()->rank();
      }

      /* Ranked component */
      NumericalSample NumericalSample::rank(const UnsignedLong index) const
      {
	return getImplementation()->rank(index);
      }

      /* Sorted sample */
      NumericalSample NumericalSample::sort() const
      {
	return getImplementation()->sort();
      }

      /* Sorted component */
      NumericalSample NumericalSample::sort(const UnsignedLong index) const
      {
	return getImplementation()->sort(index);
      }

      /* Sorted component */
      NumericalSample NumericalSample::sortAccordingAComponent(const UnsignedLong index) const
      {
	return getImplementation()->sortAccordingAComponent(index);
      }

      /* Get the i-th marginal sample */
      NumericalSample NumericalSample::getMarginal(const UnsignedLong index) const throw(InvalidArgumentException)
      {
	return getImplementation()->getMarginal(index);
      }

      /* Get the marginal sample corresponding to indices dimensions */
      NumericalSample NumericalSample::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	return getImplementation()->getMarginal(indices);
      }

      /* Returns a pointer to the underlying implementation object */
      NumericalSample::ImplementationAsPersistentObject NumericalSample::getImplementationAsPersistentObject() const
      {
        return getImplementation();
      }

      /* Sets the pointer to the underlying implementation object */
      void NumericalSample::setImplementationAsPersistentObject(const ImplementationAsPersistentObject & obj)
      {
        getImplementation().assign(obj);
      }


    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
